/*
 * @author		Alfonso Muñoz-Pomer Fuentes, 
 * 				<a href="mailto:alfonso.munozpomer@biotechvana.com">
 * 				alfonso.munozpomer@biotechvana.com</a>,  
 * 				<a href="http://www.biotechvana.com">Biotechvana</a>
 *
 * @date		2010-11-01
 * 
 * @copyright	Copyright Biotech Vana, S.L. 2006-2011
 */

package com.biotechvana.javabiotoolkit;

import java.io.Serializable;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.ListIterator;
import java.util.RandomAccess;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;


/**
 * Superclass of all single-stranded biological sequences. 
 *   
 * @version	1.4, 2011-07-19
 * 
 * @author	<a href="mailto:alfonso.munozpomer@biotechvana.com">Alfonso Muñoz-Pomer Fuentes</a>,
 * 			<a href="http://www.biotechvana.com">Biotechvana</a>.
 * 
 * @see	DNASequence
 * @see	RNASequence
 * @see	ProteinSequence
 * 
 * <style type="text/css">
 * 		table.t0 {
 * 			border:0px solid black;
 * 			border-collapse: collapse;
 * 		}
 * 		table.t0 td {
 * 			text-align: center;
 * 			padding: 4px;
 * 		}
 * 		tr.d0 td {
 * 			background-color: #FFFFFF; color: black;
 * 		}
 * 		tr.d1 td {
 * 			background-color: #DDDDDD; color: black;
 * 		}
 * </style>
 */
public abstract class BioSequence implements Cloneable, Serializable
{
	private static final long serialVersionUID = 6705995642354072553L; 	// Autogenerated by Eclipse for Serializable
	
	protected static final NumberFormat numberFormat = NumberFormat.getInstance();
	
	protected StringBuilder descriptionSB;
	// ArrayList of DnaBase/RnaBase/AminoAcid symbols and directionality
	protected List<BioResidue> sequence;
	protected BioSequenceDirectionality direction;
	
	/**
	 * Default constructor for single stranded biological sequences.
	 * 
	 * @param	description		description text (e.g. FASTA header).
	 * @param	direction		directionality of the biological sequence.
	 * @param	initialLength	initial length of the sequence.
	 *
	 * @since	0.1, 2010-11-01
	 */
	//public
	BioSequence(String description, BioSequenceDirectionality direction, int initialLength)
	{
		this.descriptionSB = new StringBuilder(description);
		this.direction = direction;
		this.sequence = new ArrayList<BioResidue>(initialLength);
	}
	
	/**
	 * Add a residue to the end of the sequence.
	 * 
	 * @param	residue	new residue to append to the sequence.
	 * @return	this sequence.
	 *
	 * @since	0.2
	 */
	// Because sequence is of type List<BioResidue>, in the case of an empty sequence there is no way to know which 
	// subclass of BioResidue is allowed to be inserted. Each subclass of BioSequence must implement this method 
	// separately.
	public abstract BioSequence add(BioResidue residue)
	throws IllegalArgumentException;

	/**
	 * 
	 * @param index
	 * @param residue
	 * @return
	 *
	 * @since	x.y.z
	 */
	public abstract BioSequence add(int index, BioResidue residue)
	throws IllegalArgumentException, IndexOutOfBoundsException;
	
	/**
	 * 
	 * @param insertSequence
	 * @return
	 *
	 * @since	x.y.z
	 */
	public BioSequence add(BioSequence insertSequence)
	throws IllegalArgumentException, NullPointerException
	{
		if (insertSequence.getClass() != this.getClass())
		{
			throw new IllegalArgumentException(
				insertSequence.getClass().getName() + " illegal BioSequence subclass when adding to a sequence of " +
				"class " + getClass().getName());
		}
		
		sequence.addAll(insertSequence.sequence);
		return this;
	}
	
	/**
	 * 
	 * @param insertSequence
	 * @return
	 *
	 * @since	x.y.z
	 */
	public BioSequence add(int index, BioSequence insertSequence)
	throws IllegalArgumentException, IndexOutOfBoundsException, NullPointerException
	{
		if (insertSequence.getClass() != this.getClass())
		{
			throw new IllegalArgumentException(
				insertSequence.getClass().getName() + " illegal BioSequence subclass when adding to a sequence of " +
				"class " + getClass().getName());
		}
		
		sequence.addAll(index, insertSequence.sequence);
		return this;
	}
	
	/**
	 * 
	 * @param index
	 * @return
	 *
	 * @since	x.y.z
	 */
	public BioResidue get(int index)
	throws IndexOutOfBoundsException
	{
		return sequence.get(index);
	}

	// TODO Check proper BioResidue subtype (may need to abstract this method and implement in subclasses)
	public BioResidue set(int index, BioResidue b)
	{
		return sequence.set(index, b);
	}
	
	/**
	 * 
	 * @param index
	 * @return
	 *
	 * @since	x.y.z
	 */
	public BioResidue remove(int index)
	throws IndexOutOfBoundsException
	{
		return sequence.remove(index);
	}
	
	/**
	 * 
	 * @param fromIndex
	 * @param toIndex
	 * @return
	 *
	 * @since	x.y.z
	 */
	public BioSequence removeRange(int fromIndex, int toIndex)
	throws IndexOutOfBoundsException
	{
		sequence.subList(fromIndex, toIndex).clear();
		return this;
	}
	
	public BioSequence reverse()
	{
		return this.reverse(new NullProgressMonitor());
	}
	
	
	/**
	 * 
	 * @return
	 *
	 * @since	x.y.z
	 */
	public BioSequence reverse(IProgressMonitor monitor)
	{
		// From: public static void reverse(List<?> list)
		int REVERSE_THRESHOLD = 18;
		int size = sequence.size();
        if (size < REVERSE_THRESHOLD || sequence instanceof RandomAccess)
        {
        	for (int i = 0, mid = size >> 1, j = size - 1 ; i < mid ; i++, j--, monitor.worked(2))
        	{
        		sequence.set(i, sequence.set(j, sequence.get(i)));
        		
        		if (monitor.isCanceled())
    			{
    				throw new OperationCanceledException();
    			}
        	}
        }
        else
        {
        	ListIterator<BioResidue> fwd = sequence.listIterator();
        	ListIterator<BioResidue> rev = sequence.listIterator(size);
        	for (int i = 0, mid = sequence.size() >> 1 ; i < mid ; i++, monitor.worked(1))
        	{
        		BioResidue tmp = fwd.next();
        		fwd.set(rev.previous());
        		rev.set(tmp);
        		
        		if (monitor.isCanceled())
    			{
    				throw new OperationCanceledException();
    			}
        	}
        }
				
		//Collections.reverse(sequence);
		return this;
	}
	
	/**
	 * 
	 * @return
	 *
	 * @since	x.y.z
	 */
	public BioSequence switchDirectionality()
	{
		direction = direction.reverse();
		return this;
	}
	
	/**
	 * 
	 * @param includeAmbiguousResidues
	 * @return
	 *
	 * @since	x.y.z
	 */
	@Deprecated
	protected abstract List<BioSequence> expand(boolean includeAmbiguousResidues);
	
	/**
	 * 
	 * @return
	 *
	 * @since	x.y.z
	 */
	@Deprecated
	public List<BioSequence> expandToUnambiguous()
	{
		return expand(false);
	}
	
	/**
	 * 
	 * @return
	 *
	 * @since	x.y.z
	 */
	@Deprecated
	public List<BioSequence> expandToAll()
	{
		return expand(true);
	}

	/**
	 * 
	 * @param motif
	 * @param matchAmbiguous
	 * @return
	 * @throws IllegalArgumentException		the motif is of a different class than the receiver
	 * @throws IndexOutOfBoundsException	index is out of bounds.
	 * @throws InterruptedException 
	 *
	 * @since	x.y.z
	 */
	public List<Annotation> findNextMotif
	(int index, BioSequence motif, boolean matchAmbiguous, IProgressMonitor progressMonitor)
	throws IllegalArgumentException, OperationCanceledException
	{
		if (motif.getClass() != this.getClass())
		{
			throw new IllegalArgumentException(
				motif.getClass().getName() + " illegal BioSequence subclass when finding motifs in a sequence of " +
				"class " + getClass().getName());
		}
		
		// Truncate the sequence length if necessary in order to display it on the monitor
		StringBuilder searchSB = new StringBuilder();
		int searchStringMaxLength = 15;
		if (motif.sequence.size() <= searchStringMaxLength)
		{
			searchSB.append(motif.toString() + "+");
		}
		else // if (motif.sequence.size() > searchStringMaxLength)
		{
			for (int i = 0 ; i < searchStringMaxLength - 5 ; i++)
			{
				searchSB.append(motif.get(i).getUpperCaseChar());
			}
			searchSB.append("(...)+");
		}
		
		// Start by reporting the current task to the monitor
		progressMonitor.subTask("Searching for motif " + searchSB.toString());
				
		// Initialise and prepare all the motif variations
		List<List<BioResidue>> explodedMotif = new ArrayList<List<BioResidue>>(motif.getLength());
		for (BioResidue b : motif.sequence)
		{
			List<BioResidue> explodedResidue = new ArrayList<BioResidue>();
			if (matchAmbiguous)
			{
				explodedResidue.addAll(Arrays.asList(b.expandToAll()));
			}
			else
			{
				explodedResidue.addAll(Arrays.asList(b.expandToUnambiguous()));
			}
			explodedMotif.add(explodedResidue);
		}

		// Find the next occurrence of the given motif
		for (int i = index ; i + explodedMotif.size() <= sequence.size() ; i++, progressMonitor.worked(1))
		{
			boolean match = true;
			for (int j = 0 ; j < explodedMotif.size() ; j++)
			{
				if (explodedMotif.get(j).contains(sequence.get(i + j)))
				{
					continue;
				}
				match = false;
				break;
			}
			if (match)
			{
				List<Annotation> annotationInList = new ArrayList<Annotation>();
				annotationInList.add(
					new Annotation(
						motif.descriptionSB.toString(), i, i + motif.getLength(),
						false, this instanceof ProteinSequence));
				return annotationInList;
			}
			
			if (progressMonitor.isCanceled())
			{
				throw new OperationCanceledException();
			}
		}
		return null;
	}

	/**
	 * 
	 * @param motif
	 * @param matchAmbiguous
	 * @return
	 * @throws IllegalArgumentException		the motif is of a different class than the receiver
	 * @throws InterruptedException 
	 * @throws IndexOutOfBoundsException	index is out of bounds.
	 *
	 * @since	x.y.z
	 */
	public List<Annotation> findPreviousMotif
	(int index, BioSequence motif, boolean matchAmbiguous, IProgressMonitor progressMonitor)
	throws IllegalArgumentException, OperationCanceledException
	{
		if (motif.getClass() != this.getClass())
		{
			throw new IllegalArgumentException(
				motif.getClass().getName() + " motif queried for a sequence instance of " + getClass().getName());
		}
		
		// Truncate the sequence length if necessary in order to display it on the monitor
		StringBuilder searchSB = new StringBuilder();
		int searchStringMaxLength = 15;
		if (motif.sequence.size() <= searchStringMaxLength)
		{
			searchSB.append(motif.toString() + "+");
		}
		else // if (motif.sequence.size() > searchStringMaxLength)
		{
			for (int i = 0 ; i < searchStringMaxLength - 5 ; i++)
			{
				searchSB.append(motif.get(i).getUpperCaseChar());
			}
			searchSB.append("(...)+");
		}
		
		// Start by reporting the current task to the monitor
		progressMonitor.subTask("Searching for motif " + searchSB.toString());
				
		// Initialise and prepare all the motif variations
		List<List<BioResidue>> explodedMotif = new ArrayList<List<BioResidue>>(motif.getLength());
		for (BioResidue b : motif.sequence)
		{
			List<BioResidue> explodedResidue = new ArrayList<BioResidue>();
			if (matchAmbiguous)
			{
				explodedResidue.addAll(Arrays.asList(b.expandToAll()));
			}
			else
			{
				explodedResidue.addAll(Arrays.asList(b.expandToUnambiguous()));
			}
			explodedMotif.add(explodedResidue);
		}

		// Find the previous occurrence of the given motif
		for (int i = index ; i - explodedMotif.size() + 1 >= 0 ; i--, progressMonitor.worked(1))
		{
			boolean match = true;
			for (int j = 0 ; j < explodedMotif.size() ; j++)
			{
				if (explodedMotif.get(explodedMotif.size() - 1 - j).contains(sequence.get(i - j)))
				{
					continue;
				}
				match = false;
				break;
			}
			if (match)
			{
				List<Annotation> annotationInList = new ArrayList<Annotation>();
				annotationInList.add(
						new Annotation(
							motif.getDescription(), i - motif.getLength() + 1, i + 1,
							false, this instanceof ProteinSequence));
				return annotationInList;
			}
			
			if (progressMonitor.isCanceled())
			{
				throw new OperationCanceledException();
			}
		}
		return null;
	}
	
	/**
	 * 
	 * @return
	 *
	 * @since	x.y.z
	 */
	public int getLength()
	{
		return sequence.size();
	}
	
	/**
	 * 
	 * @return
	 *
	 * @since	x.y.z
	 */
	public String getDescription()
	{
		return descriptionSB.toString();
	}
	
	/**
	 * 
	 * @return
	 *
	 * @since	x.y.z
	 */
	public BioSequenceDirectionality getDirectionality()
	{
		return direction;
	}
	
	/**
	 * 
	 * @return
	 *
	 * @since	x.y.z
	 */
	public StringBuilder getSequenceStringBuilder()
	{
		StringBuilder sequenceSB = new StringBuilder(sequence.size());
		for (BioResidue r : sequence)
		{
			sequenceSB.append(r.getUpperCaseChar());
		}
		return sequenceSB;
	}
	
	/**
	 * 
	 * @param fromIndex
	 * @return
	 * @throws ArrayIndexOutOfBoundsException
	 *
	 * @since	x.y.z
	 */
	public abstract BioSequence getSubsequence(int fromIndex)
	throws ArrayIndexOutOfBoundsException;

	/**
	 * 
	 * @param fromIndex
	 * @param toIndex
	 * @return
	 * @throws ArrayIndexOutOfBoundsException
	 *
	 * @since	x.y.z
	 */
	public abstract BioSequence getSubsequence(int fromIndex, int toIndex)
	throws ArrayIndexOutOfBoundsException;

	/**
	 * 
	 * @param b
	 * @return
	 *
	 * @since	x.y.z
	 */
	public int getIndexOf(BioResidue r)
	{
		return sequence.indexOf(r);
	}
	
	/**
	 * 
	 * @param r
	 * @param fromIndex
	 * @return
	 *
	 * @since	x.y.z
	 */
	public int getIndexOf(BioResidue r, int fromIndex)
	{
		return sequence.subList(fromIndex, sequence.size()).indexOf(r);
	}
	
	/**
	 * TODO :: refactor this to accept just one annotation
	 * @param annotations
	 * @return
	 *
	 * @since	x.y.z
	 */
	public String getAnnotationSubsequence(List<Annotation> annotations)
	throws IndexOutOfBoundsException
	{
		StringBuilder annotationRegionSB = new StringBuilder("");
		
		// Add the whole region in lower case
		for (int i = annotations.get(0).normalizedStart() ;
				 i < annotations.get(annotations.size() - 1).normalizedEnd() ; i++)
		{
			annotationRegionSB.append(sequence.get(i).getLowerCaseChar());
		}
		//TODO :: chech this
		// Change to upper case the annotated parts, no problem if we uppercase twice in overlapping annotations
		for (Annotation a : annotations)
		{
			for (int i = a.normalizedStart() ; i < a.normalizedEnd() ; i++)
			{
				int position = i - annotations.get(0).normalizedStart();
				annotationRegionSB.setCharAt(position, sequence.get(i).getUpperCaseChar());
			}
		}
		
		return annotationRegionSB.toString();
	}
	
	
	/**
	 * 
	 * @param annotation
	 * @return
	 *
	 * @since	x.y.z
	 */
	public String getAnnotationSubsequence(Annotation annotation)
	throws IndexOutOfBoundsException
	{
		StringBuilder annotationRegionSB = new StringBuilder("");
		
		// Add the whole region in lower case
		for (int i = annotation.normalizedStart() ;
				 i < annotation.normalizedEnd() ; i++)
		{
			annotationRegionSB.append(sequence.get(i));
		}
		//TODO :: chech this
		// Change to upper case the annotated parts, no problem if we uppercase twice in overlapping annotations
//		for (Annotation a : annotations)
//		{
//			for (int i = annotation.normalizedStart() ; i < annotation.normalizedEnd() ; i++)
//			{
//				int position = i - annotation.normalizedStart();
//				annotationRegionSB.setCharAt(position, sequence.get(i).getUpperCaseChar());
//			}
//		}
		
		return annotationRegionSB.toString();
	}
	
	
	/**
	 * 
	 * @param description
	 * @return
	 *
	 * @since	x.y.z
	 */
	public BioSequence setDescription(String description)
	{
		return setDescription(new StringBuilder(description));
	}
	
	/**
	 * 
	 * @param description
	 * @return
	 *
	 * @since	x.y.z
	 */
	public BioSequence setDescription(StringBuilder descriptionSB)
	{
		this.descriptionSB = descriptionSB;
		return this;
	}
	
	/* 
	 * (non-Javadoc) @see java.lang.Object#clone()
	 */
	@Override
	public Object clone() throws CloneNotSupportedException
	{
		BioSequence bs = (BioSequence)super.clone();
		
		// Maybe some of these are already covered when calling super.clone() above
		bs.descriptionSB = new StringBuilder(descriptionSB.toString());
		bs.direction = direction;
		bs.sequence = new ArrayList<BioResidue>(sequence);
		
		return bs;
	}
	
	/* 
	 * (non-Javadoc) @see java.lang.Object#equals(java.lang.Object)
	 * @since	0.1
	 */
	@Override
	public boolean equals(Object o)
	{
		if (o instanceof BioSequence)
		{
			boolean sameContents =  Arrays.deepEquals(sequence.toArray(), ((BioSequence)o).sequence.toArray());
			boolean sameDirection = (((BioSequence)o).direction == direction); 
			return sameContents && sameDirection;
		}
		else
		{
			return false;
		}
	}
	
	/*
	 * (non-Javadoc) @see java.lang.Object#hashCode()
	 * @since	0.1
	 */
	public int hashCode()
	{
		return Arrays.deepHashCode(sequence.toArray());
	}
	
	/*
	 * (non-Javadoc) @see java.lang.Object#toString()
	 */
	public String toString()
	{
		StringBuilder sequenceSB = new StringBuilder(sequence.size());
		for (BioResidue r : sequence)
		{
			sequenceSB.append(r.getUpperCaseChar());
		}
		return sequenceSB.toString();
	}

	
	String seqName = null;
	/**
	 * WE should use this in a genral way. Only return the first part of the seq name from the fasta files.
	 * @return
	 */
	public String getSeqName() {
		if(seqName != null)
			return seqName;
		return getDescription().split(" ")[0];
	}

	boolean renameEvent = false;
	StringBuilder oldDescriptionSB = null;
	public String getOrginalDescription() {
		
		if(oldDescriptionSB == null ) {
			return getDescription();
		}
		return oldDescriptionSB.toString();
	}
	
	public void addRenameEvent(String newDescLine) {
		if(renameEvent == false ) {
			renameEvent = true;
			oldDescriptionSB = descriptionSB;
		}
		setDescription(newDescLine);
	}
	public void removeRenameEvent() {
		renameEvent = false;
		oldDescriptionSB = null;
	}
}
