/*
 * @author		Alfonso Muñoz-Pomer Fuentes, 
 * 				<a href="mailto:alfonso.munozpomer@biotechvana.com">
 * 				alfonso.munozpomer@biotechvana.com</a>,  
 * 				<a href="http://www.biotechvana.com">Biotechvana</a>
 *
 * @date		26-08-2011
 * 
 * @license		<a href="http://creativecommons.org/licenses/by-nc-sa/3.0/">
 * 				Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License</a>
 *
 * @copyright	Copyright Biotech Vana, S.L. 2006-2011
 */

package com.biotechvana.javabiotoolkit.utils;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.nio.channels.FileLock;
import java.util.HashMap;
import java.util.Map;
import java.util.Stack;

/**
 * @author	<a href="mailto:alfonso.munozpomer@biotechvana.com">Alfonso Muñoz-Pomer Fuentes</a>,
 * 			<a href="http://www.biotechvana.com">Biotechvana</a>.
 *
 * @version	0.1
 *
 */
public class UndoManager
{
	/**
	 * 
	 * @author	<a href="mailto:alfonso.munozpomer@biotechvana.com">Alfonso Muñoz-Pomer Fuentes</a>,
	 * 			<a href="http://www.biotechvana.com">Biotechvana</a>.
	 *
	 * @version	0.1
	 *
	 */
	public class UndoableAction implements Runnable
	{
		private String actionId;
		private Object[] parameters;
		private String dataFilePath;
		private Serializable data;
		
		/**
		 * 
		 * @param actionId
		 * @param parameterList
		 *
		 * @since	x.y.z
		 */
		private UndoableAction(String actionId, String dataFilePath, Serializable data, Object ... parameters)
		{
			this.actionId = actionId;
			this.parameters = parameters;
			this.dataFilePath = dataFilePath;
			this.data = data;
		}
		
		public String getActionId()
		{
			return actionId;
		}
		
		public Object[] getParameters()
		{
			return parameters;
		}
		
		public Serializable getData()
		{
			return data;
		}

		/* (non-Javadoc)
		 * @see java.lang.Runnable#run()
		 */
		@Override
		public void run()
		{
			FileOutputStream outFile = null;
			try
			{
				// Open and lock file
				outFile = new FileOutputStream(dataFilePath);
				FileLock outLock = outFile.getChannel().tryLock();
				try
				{
					// Serialize data to file
					ObjectOutputStream out = new ObjectOutputStream(outFile);
					out.writeObject(data);
				}
				finally
				{
					outLock.release();
				}
			}
			catch (Exception exception)
			{
				throw new RuntimeException(exception);
			}
			finally
			{
				if (outFile != null)
				{
					// Close file
					try
					{
						outFile.close();
						data = null;
					}
					catch (IOException exception)
					{
						throw new RuntimeException(exception);
					}
				}
			}
			// After a new undoable action has been cached we don't need the current redoable actions
			while (!redoStack.empty())
			{
				UndoableAction ua = redoStack.pop();
				File f = new File(ua.dataFilePath);
				if (f.exists())
				{
					f.delete();
				}
			}
		}
	}
	
	private String applicationId;
	private Map<String, Class<?>[]> undoableActionMap;
	private Stack<UndoableAction> undoStack;
	private Stack<UndoableAction> redoStack;
	
	/**
	 * 
	 * 
	 *
	 * @since	x.y.z
	 */
	public UndoManager(String applicationId)
	{
		this.applicationId = applicationId;
		undoableActionMap = new HashMap<String, Class<?>[]>();
		undoStack = new Stack<UndoableAction>();
		redoStack = new Stack<UndoableAction>();
	}
	
	/**
	 * 
	 * @param id
	 * @param parameterList
	 * @return 
	 *
	 * @since	x.y.z
	 */
	public Class<?>[] registerUndoableAction(String actionId, Class<?> ... parameterList)
	{
		return undoableActionMap.put(actionId, parameterList);
	}
	
	/**
	 * 
	 * @param actionId
	 * @return
	 *
	 * @since	x.y.z
	 */
	public Class<?>[] deregisterUndoableAction(String actionId)
	{
		return undoableActionMap.remove(actionId);
	}
	
	/**
	 * 
	 * @param actionId
	 * @return
	 *
	 * @since	x.y.z
	 */
	public Class<?>[] getParameterTypes(String actionId)
	{
		return undoableActionMap.get(actionId);
	}
	
	/**
	 * 
	 * @param actionId
	 * @param parameters
	 * @param data
	 * @throws IOException 
	 *
	 * @since	x.y.z
	 */
	public void pushUndoAction(String actionId, final Serializable data, Object ... parameters) throws IOException
	{
		Class<?>[] parameterClasses = undoableActionMap.get(actionId);
		
		// Check that number of parameters is right
		if (parameters.length != parameterClasses.length)
		{
			throw new IllegalArgumentException("Action parameters do not match " + actionId + " pattern");
		}
		
		// Check that the type of the parameters is right
		for (int i = 0 ; i < parameters.length ; i++)
		{
			if (!parameters[i].getClass().equals(parameterClasses[i]))
			{
				throw new IllegalArgumentException(
						"Parameter in position " + i + " does not match its template " + parameterClasses[i]);
			}
		}
		
		File tempDataFile = File.createTempFile(applicationId, Long.toString(System.currentTimeMillis()));
		
		UndoableAction ua = new UndoableAction(actionId, tempDataFile.getCanonicalPath(), data, parameters);
		undoStack.push(ua);
		
		Thread thread = new Thread(ua);
		thread.start();
	}
	
	/**
	 * 
	 * @return
	 * @throws IOException 
	 *
	 * @since	x.y.z
	 */
	public UndoableAction popUndoAction() throws IOException
	{
		UndoableAction ua = undoStack.pop();
		redoStack.push(ua);
		
		FileInputStream inFile = null;
	
		try
		{
			inFile = new FileInputStream(ua.dataFilePath);
			FileLock inLock = inFile.getChannel().tryLock(0L, Long.MAX_VALUE, true);
			try
			{
				ObjectInputStream in = new ObjectInputStream(inFile);
				ua.data = (Serializable) in.readObject();
			}
			catch (ClassNotFoundException exception)
			{
				// TODO Auto-generated catch block
				exception.printStackTrace();
			}
			finally
			{
				inLock.release();
			}
		}
		catch (FileNotFoundException exception)
		{
			// TODO Auto-generated catch block
			exception.printStackTrace();
		}
		catch (IOException exception)
		{
			// TODO Auto-generated catch block
			exception.printStackTrace();
		}
		finally
		{
			if (inFile != null)
			{
				inFile.close();
			}
		}
		
		return ua;
	}
	
	/**
	 * 
	 * @return
	 * @throws IOException 
	 *
	 * @since	x.y.z
	 */
	public UndoableAction popRedoAction() throws IOException
	{
		UndoableAction ua = redoStack.pop();
		undoStack.push(ua);
		
		FileInputStream inFile = null;
	
		try
		{
			inFile = new FileInputStream(ua.dataFilePath);
			FileLock inLock = inFile.getChannel().tryLock(0L, Long.MAX_VALUE, true);
			try
			{
				ObjectInputStream in = new ObjectInputStream(inFile);
				ua.data = (Serializable) in.readObject();
			}
			catch (ClassNotFoundException exception)
			{
				// TODO Auto-generated catch block
				exception.printStackTrace();
			}
			finally
			{
				inLock.release();
			}
		}
		catch (FileNotFoundException exception)
		{
			// TODO Auto-generated catch block
			exception.printStackTrace();
		}
		catch (IOException exception)
		{
			// TODO Auto-generated catch block
			exception.printStackTrace();
		}
		finally
		{
			if (inFile != null)
			{
				inFile.close();
			}
		}
		
		return ua;
	}
	
	public boolean emptyRedoStack()
	{
		return redoStack.empty();
	}
	
	public boolean emptyUndoStack()
	{
		return undoStack.empty();
	}
	
	public void dispose()
	{
		while (!undoStack.empty())
		{
			UndoableAction ua = undoStack.pop();
			File f = new File(ua.dataFilePath);
			if (f.exists())
			{
				f.delete();
			}
		}
		while (!redoStack.empty())
		{
			UndoableAction ua = redoStack.pop();
			File f = new File(ua.dataFilePath);
			if (f.exists())
			{
				f.delete();
			}
		}
		undoableActionMap.clear();
	}
}
