/*
 * @author		Alfonso Muñoz-Pomer Fuentes, 
 * 				<a href="mailto:alfonso.munozpomer@biotechvana.com">
 * 				alfonso.munozpomer@biotechvana.com</a>,  
 * 				<a href="http://www.biotechvana.com">Biotechvana</a>
 *
 * @date		2010-09-01
 * 
 * @license		See <a href="http://www.biotechvana.com></a>
 *
 * @copyright	Copyright Biotech Vana, S.L. 2006-2010
 */

package com.biotechvana.javabiotoolkit.io;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.charset.Charset;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;

import com.biotechvana.javabiotoolkit.BioSequence;
import com.biotechvana.javabiotoolkit.exceptions.FastaReaderNotParsedException;
import com.biotechvana.javabiotoolkit.exceptions.FastaWriterIllegalWrapException;
import com.biotechvana.javabiotoolkit.exceptions.FastaWriterMultipleSequenceException;
import com.biotechvana.javabiotoolkit.exceptions.InvalidSequenceCharacterException;
import com.biotechvana.javabiotoolkit.exceptions.SequenceTooLongException;
import com.biotechvana.javabiotoolkit.text.LineSeparatorFormat;
import com.biotechvana.javabiotoolkit.text.NewLineFileEnder;

/**
 * Instances of this class are objects associated to plain text files which can write biological sequences in FASTA 
 * format. Typically, a <code>FastaWriter</code> is used in two steps:
 * <ol>
 * <li>Creation of the object, which looks for the specified file.</li>
 * <li>Append or replace a sequence in the file.</li>
 * </ol>
 * 
 * @version	1.1
 * 
 * @author	<a href="mailto:alfonso.munozpomer@biotechvana.com">Alfonso Muñoz-Pomer Fuentes</a>,
 * 			<a href="http://www.biotechvana.com">Biotechvana</a>.
 * 
 * @see FASTAReader
 * @see	DNASequence
 * @see	RNASequence
 * @see	ProteinSequence
 *
 */
public class FASTAWriter
{
	static final int I_charBufferSize = FASTAReader.I_byteBufferSize;	// 128 KB (usually)
	@SuppressWarnings("unused")
	private static final int I_byteBufferSize = 512*1024;	// 512 KB
	static final int I_maxWrap = 160;

	// File associated to this object and some format information
	private File filePath;
	private Charset fileCharset;
	private LineSeparatorFormat fileLineSeparatorFormat;
	private FASTAReader far;
	
	/**
	 * 
	 * @param filePath
	 * @param bufferedSequences
	 * @param charsetName
	 * @param bufferSize
	 * 
	 * @throws	IOException
	 */
	public FASTAWriter
	(File filePath, Charset fileCharset, LineSeparatorFormat fileLineSeparatorFormat, boolean overwrite)
	throws IOException
	{
		this.filePath = filePath;
		this.fileCharset = fileCharset;
		this.fileLineSeparatorFormat = fileLineSeparatorFormat;
		
		if (overwrite || !filePath.exists())
		{
			filePath.createNewFile();
		}
				
		// Now that the file exists for sure, we can create a new FastaReader
		far = new FASTAReader(this.filePath, this.fileCharset, this. fileLineSeparatorFormat, true);
	}
	
	/**
	 * For existing files
	 * 
	 * @param data.filePath
	 * @param bufferedSequences
	 * @param charsetName
	 * @param bufferSize
	 */
	public FASTAWriter(FASTAReader far)
	{
		this.filePath = far.filePath();
		this.fileCharset = far.charset();
		this.fileLineSeparatorFormat = far.fileLineSeparatorFormat();
		this.far = far;
	}
	
	/**
	 * Append with progressmonitor
	 * 
	 * @param bs
	 * @param lineWrap
	 * @param progressMonitor
	 * @throws FastaWriterIllegalWrapException
	 * @throws FileNotFoundException
	 * @throws IOException
	 * @throws InvalidSequenceCharacterException 
	 * @throws SequenceTooLongException 
	 *
	 * @since	1.0
	 */
	public void append(BioSequence bs, int lineWrap, boolean updateReader, IProgressMonitor progressMonitor)
	throws IllegalArgumentException, FileNotFoundException, IOException, SequenceTooLongException, InvalidSequenceCharacterException
	{
		if (lineWrap < 0 || lineWrap > I_maxWrap)
		{
			throw new IllegalArgumentException(lineWrap + " illegal wrap value. Must be in the range 0.." + I_maxWrap);
		}
		NewLineFileEnder.ensureLastLineNewLine(filePath, fileCharset, fileLineSeparatorFormat);
		
		// Open channel and go to end
		FileChannel outFC =	new RandomAccessFile(filePath, "rw").getChannel();	// FileNotFoundException
		// Lock channel for writing
		FileLock outLock = null;
		try
		{
			progressMonitor.subTask("Appending sequence to file...");
			outFC.position(outFC.size());	// IOException

			outLock = outFC.tryLock();
			if (outLock != null)
			{	
				// Allocate CharBuffer
				CharBuffer cBuffer = CharBuffer.allocate(I_charBufferSize * 2);	// The extra space avoids an overflow
				// Write FASTA record mark and description in portions of I_charBufferSize
				outFC.write(ByteBuffer.wrap(">".getBytes(fileCharset)));
				for (int i = 0 ; i < bs.getDescription().length() ; i += I_charBufferSize)
				{
					for (int j = 0 ; (j < I_charBufferSize) && ((i + j) < bs.getDescription().length()) ; j++)
					{
						cBuffer.put(bs.getDescription().charAt(i + j));
					}
					cBuffer.flip();
					outFC.write(fileCharset.encode(cBuffer));
					progressMonitor.worked(cBuffer.capacity());
					cBuffer.clear();
				}
				
				if (progressMonitor.isCanceled())
				{
					throw new OperationCanceledException();
				}
				
				// Write sequence in portions of charsPerIteration with line wrap
				for (int i = 0 ; i < bs.getLength() ; i += I_charBufferSize)
				{
					for (int j = 0 ; j < I_charBufferSize && i + j < bs.getLength() ; j++)
					{
						if (((i + j) % lineWrap) == 0)
						{
							cBuffer.put(fileLineSeparatorFormat.lineSeparator());
						}
						cBuffer.put(bs.get(i + j).getUpperCaseChar());
					}
					cBuffer.flip();
					outFC.write(fileCharset.encode(cBuffer));
					progressMonitor.worked(cBuffer.capacity());
					cBuffer.clear();
				}
				
				if (progressMonitor.isCanceled())
				{
					throw new OperationCanceledException();
				}
			}
		}
		finally
		{
			outLock.release();
			outFC.close();
			// TODO :: this is very slow when wrting the whole file
			// when appedning we need to parse only from the last point 
			// no need to parse the whole file from the begening
			if(updateReader)
				far.parse(true, progressMonitor);
		}
	}
	
	/**
	 * Append with progress monitor and default width 72
	 * 
	 * @param	bs
	 * @param	progressMonitor
	 * @throws	FastaWriterIllegalWrapException
	 * @throws	FileNotFoundException
	 * @throws	IOException
	 * @throws InvalidSequenceCharacterException 
	 * @throws SequenceTooLongException 
	 *
	 * @since	1.1
	 */
	public void append(BioSequence bs, boolean updateReader, IProgressMonitor progressMonitor)
	throws FastaWriterIllegalWrapException, FileNotFoundException, IOException, SequenceTooLongException, InvalidSequenceCharacterException
	{
		append(bs, 72,updateReader, progressMonitor);
	}
	
	/**
	 * 
	 * @param bs
	 * @param lineWrap
	 * @param charBufferSize
	 * @throws IOException
	 * @throws FastaWriterIllegalWrapException 
	 * @throws FastaReaderNotParsedException 
	 * @throws FastaWriterMultipleSequenceException 
	 * @throws InvalidSequenceCharacterException 
	 * @throws SequenceTooLongException 
	 */
	public void replace(BioSequence bs, int lineWrap, IProgressMonitor progressMonitor)
	throws IllegalArgumentException, FastaReaderNotParsedException, FileNotFoundException, IOException, 
		   FastaWriterMultipleSequenceException, SequenceTooLongException, InvalidSequenceCharacterException
	{
		if (lineWrap < 0 || lineWrap > I_maxWrap)
		{
			throw new IllegalArgumentException(lineWrap + " illegal wrap value. Must be in the range 0.." + I_maxWrap);
		}
		
		// Extra check: If the record does not exist append it to the end of the file
		progressMonitor.subTask("Locating sequence...");
		if (far.findRecords(bs.getOrginalDescription()).size() == 0)
		{	
			append(bs, lineWrap,true, progressMonitor);
		}
		// Copy to temp file, append sequence, skip the old record, copy the rest of the original file, delete it and 
		// copy the temp file back to the original location
		else if (far.findRecords(bs.getOrginalDescription()).size() == 1)
		{	
			FileChannel originalFC = null;
			FileLock originalLock = null;
			FileChannel tempFC = null;
			FileLock tempLock = null;
			try
			{
				FASTAFileRecord oldRecord = far.findRecords(bs.getOrginalDescription()).get(0);
				int oldRecordIndex = far.getFastaRecords().indexOf(oldRecord);
				long oldRecordStartByte = oldRecord.offset();
				// End byte is the end of the file or the start of the next sequence
				long oldRecordEndByte = filePath.length();
				if (oldRecordIndex < far.getFastaRecords().size() - 1)
				{
					oldRecordEndByte = far.getFastaRecords().get(oldRecordIndex + 1).offset();
				}
				
				// Create a new temp file
				File tempFilePath = File.createTempFile("time.", null);

				originalFC = new RandomAccessFile(filePath, "rw").getChannel();
				tempFC = new RandomAccessFile(tempFilePath, "rw").getChannel();
				
				progressMonitor.subTask("Writing temporary file...");
				tempLock = tempFC.tryLock();
				originalLock = originalFC.tryLock();
				if (tempLock != null && originalLock != null)
				{
					// Copy up to old sequence
					originalFC.transferTo(0, oldRecordStartByte, tempFC);
					progressMonitor.worked((int)oldRecordStartByte);
					// Write new sequence
					progressMonitor.subTask("Adding sequence...");
					tempFC.position(tempFC.size());	// IOException
					// Allocate CharBuffer
					CharBuffer cBuffer = CharBuffer.allocate(I_charBufferSize * 2);	// The extra space avoids an overflow
					// Write FASTA record mark and description in portions of I_charBufferSize
					tempFC.write(ByteBuffer.wrap(">".getBytes(fileCharset)));
					progressMonitor.worked(1);
					for (int i = 0 ; i < bs.getDescription().length() ; i += I_charBufferSize)
					{
						for (int j = 0 ; (j < I_charBufferSize) && ((i + j) < bs.getDescription().length()) ; j++)
						{
							cBuffer.put(bs.getDescription().charAt(i + j));
						}
						cBuffer.flip();
						tempFC.write(fileCharset.encode(cBuffer));
						progressMonitor.worked(cBuffer.limit());
						cBuffer.clear();
					}
					// Write sequence in portions of charsPerIteration with line wrap
					for (int i = 0 ; i < bs.getLength() ; i += I_charBufferSize)
					{
						for (int j = 0 ; j < I_charBufferSize && i + j < bs.getLength() ; j++)
						{
							if (((i + j) % lineWrap) == 0)
							{
								//TODO Use fileLineSeparartor (more than one char can truncate the buffer)
								cBuffer.put('\n');
							}
							cBuffer.put(bs.get(i + j).getUpperCaseChar());
						}
						cBuffer.flip();
						tempFC.write(fileCharset.encode(cBuffer));
						progressMonitor.worked(cBuffer.limit());
						cBuffer.clear();
					}
					tempFC.write(fileCharset.encode(fileLineSeparatorFormat.lineSeparator()));
					progressMonitor.worked(fileLineSeparatorFormat.lineSeparator().length());
					tempLock.release();
					// Copy rest of the file
					progressMonitor.subTask("Adding remaining sequences...");
					originalFC.transferTo(oldRecordEndByte, originalFC.size(), tempFC);
					progressMonitor.worked((int)(originalFC.size() - oldRecordEndByte));

					// "Delete" original file
					originalFC.truncate(0);
					
					// Copy temp file to original location
					progressMonitor.subTask("Rebuilding file...");
					tempFC.transferTo(0, tempFC.size(), originalFC);
					progressMonitor.worked((int)tempFC.size());
					
					originalLock.release();
					tempLock.release();
					
					// Delete temp file and update FastaReader
					tempFilePath.delete();
					far.parse(true, progressMonitor);
				}
			}
			finally
			{
				originalFC.close();
				tempFC.close();
			}
		}
		else // if (far.findRecords(bs.getDescription()).size() > 1)
		{
			throw new FastaWriterMultipleSequenceException("\"" + 
					bs.getDescription() + "\" has " + far.findRecords(bs.getDescription()).size() + 
					" copies in the destination file");
		}
	}
	
	/**
	 * 
	 * @param bs
	 * @param lineWrap
	 * @param charBufferSize
	 * @throws IOException
	 * @throws FastaWriterIllegalWrapException 
	 * @throws FastaReaderNotParsedException 
	 * @throws FastaWriterMultipleSequenceException 
	 * @throws InvalidSequenceCharacterException 
	 * @throws SequenceTooLongException 
	 */
	public void replace(BioSequence bs, IProgressMonitor progressMonitor)
	throws FastaWriterIllegalWrapException, FastaReaderNotParsedException, FileNotFoundException, IOException, FastaWriterMultipleSequenceException, SequenceTooLongException, InvalidSequenceCharacterException
	{
		replace(bs, 72, progressMonitor);
	}
	
	/**
	 * 
	 * @return
	 */
	public FASTAReader getFastaReader()
	{
		return far;
	}
}
