/*
 * @author		Alfonso Muñoz-Pomer Fuentes, 
 * 				<a href="mailto:alfonso.munozpomer@biotechvana.com">
 * 				alfonso.munozpomer@biotechvana.com</a>,  
 * 				<a href="http://www.biotechvana.com">Biotechvana</a>
 *
 * @date		2010-11-01
 * 
 * @copyright	Copyright Biotech Vana, S.L. 2006-2010
 */

package com.biotechvana.javabiotoolkit;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.ListIterator;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;

import com.biotechvana.javabiotoolkit.exceptions.InvalidSequenceClassException;
import com.biotechvana.javabiotoolkit.utils.Permutator;

/**
 * 
 * @author	<a href="mailto:alfonso.munozpomer@biotechvana.com">Alfonso Muñoz-Pomer Fuentes</a>,
 * 			<a href="http://www.biotechvana.com">Biotechvana</a>.
 *
 * @version	1.5, 2011-10-19
 * 
 * @see		RNABase
 * 
 * <style type="text/css">
 * 		table.t0 {
 * 			border:0px solid black;
 * 			border-collapse: collapse;
 * 		}
 * 		table.t0 td {
 * 			text-align: center;
 * 			padding: 4px;
 * 		}
 * 		tr.d0 td {
 * 			background-color: #FFFFFF; color: black;
 * 		}
 * 		tr.d1 td {
 * 			background-color: #DDDDDD; color: black;
 * 		}
 * </style>
 */
public class RNASequence extends BioSequence
{
	private static final long serialVersionUID = 3925767380972452848L;	// Autogenerated by Eclipse

	/**
	 * Constructs a DNA sequence from the specified description, nucleotide base sequence and directionality. Gaps may 
	 * be optionally removed.
	 * 
	 * @param	description	description text (e.g. FASTA header).
	 * @param	sequence	succession of letters, each of which represents a nucleotide base.
	 * @param	dir			5'-3' or 3'-5' directionality.
	 * @throws	InvalidSequenceCharacterException	If the argument <code>sequenceSB</code> contains invalid characters 
	 * 												(i.e. those that do not match {@link DNABase} regexes)
	 * 
	 * @since 1.1
	 */
	public RNASequence
	(String description, NucleotideSequenceDirectionality direction, String sequenceString)
	throws IllegalArgumentException
	{
		super(description, direction, sequenceString.length());

		// Fill sequence and check text integrity on-the-fly
		for (int i = 0; i < sequenceString.length(); i++)
		{
			if (RNABase.valueOf(sequenceString.charAt(i)) != null)
			{
				sequence.add(RNABase.valueOf(sequenceString.charAt(i)));
			}
			else
			{
				throw new IllegalArgumentException(
					"'" + sequenceString.charAt(i) + "' invalid character when creating a sequence of class " + 
					this.getClass().getCanonicalName());
			}
		}
	}
	
	/**
	 * 
	 * @param sequenceString
	 * @throws InvalidSequenceCharacterException
	 *
	 * @since	x.y.z
	 */
	public RNASequence(String sequenceString)
	throws IllegalArgumentException
	{
		this(sequenceString, NucleotideSequenceDirectionality.C5_C3, sequenceString);
	}
	
	/**
	 * Constructs a DNA sequence from the specified description, nucleotide base sequence and directionality.
	 * <p>
	 * NOTE: This constructor is made public as an auxiliary testing method. Client programmers should use and rely on 
	 * the <code>String</code>-based and <code>StringBuilder<code>-based constructors.
	 * 
	 * @param	description	description text (e.g. FASTA header).
	 * @param	sequence	sequence of DNA bases.
	 * @param	dir			5'-3' or 3'-5' directionality.
	 * @throws InvalidSequenceClassException 
	 * 
	 * @since 0.9
	 */
	public RNASequence(String description, NucleotideSequenceDirectionality direction, List<BioResidue> sequence)
	throws IllegalArgumentException
	{
		super(description, direction, 0);

		// Check that the sequence matches RnaBase
		for (BioResidue r : sequence)
		{
			if (!(r instanceof RNABase))
			{
				throw new IllegalArgumentException(
					r.getClass().getName() + " illegal BioResidue subclass when creating a sequence of class " + 
					getClass().getName());
			}
		}
		
		// Fill nucleotides
		this.sequence = sequence;
	}
	
	/**
	 * 
	 * @param includeAmbiguousResidues
	 * @return
	 *
	 * @since	x.y.z
	 */
	@Override
	protected List<BioSequence> expand(boolean includeAmbiguousResidues)
	{
		List<List<BioResidue>> residuesList = new ArrayList<List<BioResidue>>(sequence.size());
		List<Integer> indices = new ArrayList<Integer>(sequence.size());
		int sequences = 1;
		if (includeAmbiguousResidues)
		{
			for (BioResidue r : sequence)
			{
				// Expand ambiguous bases to all possible ambiguous and unambiguous replacements
				residuesList.add(Arrays.asList(r.expandToAll()));
				// Store how many bases there are in each position
				indices.add(r.expandToAll().length);
				// Find out the number of total sequences that must be generated
				sequences = sequences * r.expandToAll().length;
			}
		}
		else
		{
			for (BioResidue r : sequence)
			{
				// Expand ambiguous bases to all possible unambiguous replacements
				residuesList.add(Arrays.asList(r.expandToUnambiguous()));
				// Store how many bases there are in each position
				indices.add(r.expandToUnambiguous().length);
				// Find out the number of total sequences that must be generated
				sequences = sequences * r.expandToUnambiguous().length;
			}
		}
	
		// Create all possible permutations (indices power set)
		Permutator petator = new Permutator(indices);
		List<List<Integer>> residuesPermutations = petator.generatePermutations();
	
		// For each permutation create a base sequence
		List<BioSequence> expandedSequences = new ArrayList<BioSequence>(sequences);
		
		for (List<Integer> l : residuesPermutations)
		{
			int basePosition = 0;
			ArrayList<BioResidue> residueList = new ArrayList<BioResidue>(sequence.size());
			for (Integer i : l)
			{
				residueList.add((residuesList.get(basePosition)).get(i));
				basePosition++;
			}
			
			expandedSequences.add(
				new RNASequence(descriptionSB.toString(), (NucleotideSequenceDirectionality) direction, residueList));
		}
		return expandedSequences;
	}

	/* 
	 * (non-Javadoc) @see com.biotechvana.javabiotoolkit.BioSequence#add(com.biotechvana.javabiotoolkit.BioResidue)
	 */
	@Override
	public BioSequence add(BioResidue residue)
	throws IllegalArgumentException
	{
		// With instanceof false is returned in the case of residue == null. Safer in this case.
		// if (residue.getClass() == RnaBase.class)
		if (residue instanceof RNABase)
		{
			sequence.add(residue);
		}
		else
		{
			throw new IllegalArgumentException(
				residue.getClass().getName() + " illegal BioResidue subclass when adding to a sequence of class " + 
				getClass().getName());
		}
		return this;
	}

	/* 
	 * (non-Javadoc) @see com.biotechvana.javabiotoolkit.BioSequence#add(int, com.biotechvana.javabiotoolkit.BioResidue)
	 */
	@Override
	public BioSequence add(int index, BioResidue residue)
	throws IllegalArgumentException, IndexOutOfBoundsException
	{
		// With instanceof false is returned in the case of residue == null. Safer in this case.
		// if (residue.getClass() == RnaBase.class)
		if (residue instanceof RNABase)
		{
			sequence.add(index, residue);
		}
		else
		{
			throw new IllegalArgumentException(
				residue.getClass().getName() + " illegal BioResidue subclass when adding to a sequence of class " + 
				getClass().getName());
		}
		return this;
	}
	
	/**
	 * 
	 * @return
	 *
	 * @since	x.y.z
	 */
	public RNASequence complement(IProgressMonitor monitor)
	{
		for (int i = 0 ; i < sequence.size() ; i++, monitor.worked(1))
		{
			sequence.set(i, ((RNABase)sequence.get(i)).getComplementary());
			
			if (monitor.isCanceled())
			{
				throw new OperationCanceledException();
			}
		}
		return this;
	}
	
	/**
	 * 
	 * @return
	 *
	 * @since	x.y.z
	 */
	public DNASequence reverseTranscribe(IProgressMonitor monitor)
	{
		DNASequence ds = 
			new DNASequence(
					descriptionSB.toString(), (NucleotideSequenceDirectionality)direction,
					new ArrayList<BioResidue>(sequence.size()));
			
		for (BioResidue b : sequence)
		{
			ds.sequence.add(((RNABase)b).getReverseTranscript());
			
			monitor.worked(1);
			if (monitor.isCanceled())
			{
				throw new OperationCanceledException();
			}
		}
		return ds;
	}

	/* 
	 * (non-Javadoc) @see com.biotechvana.javabiotoolkit.BioSequence#subsequence(int)
	 */
	@Override
	public BioSequence getSubsequence(int fromIndex)
	throws IndexOutOfBoundsException, IllegalArgumentException
	{
		List<BioResidue> newSubsequence = new ArrayList<BioResidue>(sequence.size() - fromIndex);
		newSubsequence.addAll(sequence.subList(fromIndex, sequence.size()));

		return new RNASequence(descriptionSB.toString(), (NucleotideSequenceDirectionality)direction, newSubsequence);
	}

	/* 
	 * (non-Javadoc) @see com.biotechvana.javabiotoolkit.BioSequence#subsequence(int, int)
	 */
	@Override
	public BioSequence getSubsequence(int fromIndex, int toIndex)
	throws IndexOutOfBoundsException
	{
		List<BioResidue> newSubsequence = new ArrayList<BioResidue>(toIndex - fromIndex);
		newSubsequence.addAll(sequence.subList(fromIndex, toIndex));
		
		return new RNASequence(descriptionSB.toString(), (NucleotideSequenceDirectionality)direction, newSubsequence);
	}

	/**
	 * 
	 * @param fromIndex
	 * @param motif
	 * @param matchAmbiguous
	 * @return
	 * @throws IllegalArgumentException		the motif is of a different class than the receiver
	 * @throws InterruptedException 
	 *
	 * @since	1.5
	 */
	private List<Annotation> findMotifs
	(int fromIndex, BioSequence motif, boolean matchAmbiguous, 
	 boolean forward, boolean reverse, boolean complement, boolean reverseComplement, 
	 IProgressMonitor progressMonitor)
	throws IllegalArgumentException, IndexOutOfBoundsException, OperationCanceledException
	{
		if (motif.getClass() != this.getClass())
		{
			throw new IllegalArgumentException(
				motif.getClass().getName() + " illegal BioSequence subclass when finding motifs in a " +
				"sequence of class " + getClass().getName());
		}
	
		// Truncate the sequence length if necessary in order to display it on the monitor
		StringBuilder searchSB = new StringBuilder();
		String trailingString = "(...)";
		int searchStringMaxLength = 15;
		if (motif.sequence.size() <= searchStringMaxLength)
		{
			searchSB.append(motif.toString());
		}
		else // if (motif.sequence.size() > searchStringMaxLength)
		{
			for (int i = 0 ; i < searchStringMaxLength - trailingString.length() ; i++)
			{
				searchSB.append(motif.get(i).getUpperCaseChar());
			}
			searchSB.append(trailingString);
		}
		
		// Start by reporting the current task to the monitor
		progressMonitor.subTask("Preparing motif variations.");
		
		// Initialise and prepare all the motif variations per residue
		List<List<BioResidue>> explodedMotif = new ArrayList<List<BioResidue>>(motif.getLength());
		List<List<BioResidue>> explodedMotifComplement = new ArrayList<List<BioResidue>>(motif.getLength());
		for (BioResidue b : motif.sequence)
		{
			List<BioResidue> explodedResidue = new ArrayList<BioResidue>();
			List<BioResidue> explodedResidueComplement = new ArrayList<BioResidue>();
			if (matchAmbiguous)
			{
				explodedResidue.addAll(Arrays.asList(b.expandToAll()));
				explodedResidueComplement.addAll(
						Arrays.asList(((RNABase)b).getComplementary().expandToAll()));
			}
			else
			{
				explodedResidue.addAll(Arrays.asList(b.expandToUnambiguous()));
				explodedResidueComplement.addAll(
						Arrays.asList(((RNABase)b).getComplementary().expandToUnambiguous()));
			}
			explodedMotif.add(explodedResidue);
			explodedMotifComplement.add(explodedResidueComplement);
		}
		
		// Initialise results list
		List<Annotation> motifAnnotations = new ArrayList<Annotation>();
		
		// Forward search?
		if (forward)
		{
			// Start by reporting the current task to the monitor
			String monitorMessage = "Searching for motif " + searchSB.toString() + " in forward strand.";
			progressMonitor.subTask(String.format(monitorMessage));
			
			for (int i = fromIndex ; i <= sequence.size() - motif.sequence.size() ;
				 i++, progressMonitor.worked(1))
			{
				boolean match = true;
				for (int j = 0 ; j < explodedMotif.size() ; j++)
				{
					if (explodedMotif.get(j).contains(sequence.get(i + j)))
					{
						continue;
					}
					match = false;
					break;
				}
				if (match)
				{
					motifAnnotations.add(
						new Annotation(motif.getDescription(), i, i + motif.getLength(), false, false));
					progressMonitor.subTask(
						String.format(
							monitorMessage + "%n" + " Hits found: " +
							numberFormat.format(motifAnnotations.size())));
				}
				
				if (progressMonitor.isCanceled())
				{
					throw new OperationCanceledException();
				}
			}
		}
	
		// Reverse search?
		if (reverse)
		{
			// Notify the monitor, turn around and go... backwards!
			String monitorMessage = "Searching for motif " + searchSB.toString() + " in reverse strand.";
			progressMonitor.subTask(String.format(monitorMessage));
			
			int forwardMotifCount = motifAnnotations.size();
			for (int i = sequence.size() ; i >= fromIndex + motif.sequence.size() ;
				 i--, progressMonitor.worked(1))
			{
				boolean match = true;
				for (int j = 0 ; j < explodedMotif.size() ; j++)
				{
					if (explodedMotif.get(j).contains(sequence.get(i - j - 1)))
					{
						continue;
					}
					match = false;
					break;
				}
				if (match)
				{
					// Because we go backwards, the new (lower) elements are inserted in the middle
					motifAnnotations.add(forwardMotifCount, 
						new Annotation(
							motif.descriptionSB.toString() + " [r]", i - 1, i - motif.getLength() - 1,
							false, false));
					progressMonitor.subTask(
							String.format(
								monitorMessage + "%n" + " Hits found: " +
								numberFormat.format(motifAnnotations.size())));
				}
				
				if (progressMonitor.isCanceled())
				{
					throw new OperationCanceledException();
				}
			}
		}
		
		// Complement search?
		if (complement)
		{
			// Notify the monitor
			String monitorMessage = "Searching for motif " + searchSB.toString() + " in complement strand.";
			progressMonitor.subTask(String.format(monitorMessage));
			
			for (int i = fromIndex ; i <= sequence.size() - motif.sequence.size() ;
					i++, progressMonitor.worked(1))
			{
				boolean match = true;
				for (int j = 0 ; j < explodedMotif.size() ; j++)
				{
					if (explodedMotifComplement.get(j).contains(sequence.get(i + j)))
					{
						continue;
					}
					match = false;
					break;
				}
				if (match)
				{
					motifAnnotations.add(
							new Annotation(
								motif.descriptionSB.toString() + " [c]", i, i + motif.getLength(),
								true, false));
					progressMonitor.subTask(
							String.format(
								monitorMessage + "%n" + " Hits found: " +
								numberFormat.format(motifAnnotations.size())));
				}
	
				if (progressMonitor.isCanceled())
				{
					throw new OperationCanceledException();
				}
			}
		}
		
		// Reverse search?
		if (reverseComplement)
		{
			// Notify the monitor
			String monitorMessage =
					"Searching for motif " + searchSB.toString() + " in reverse complement strand.";
			progressMonitor.subTask(String.format(monitorMessage));
	
			int forwardMotifCount = motifAnnotations.size();
			for (int i = sequence.size() ; i >= fromIndex + motif.sequence.size() ;
				 i--, progressMonitor.worked(1))
			{
				boolean match = true;
				for (int j = 0 ; j < explodedMotif.size() ; j++)
				{
					if (explodedMotifComplement.get(j).contains(sequence.get(i - j - 1)))
					{
						continue;
					}
					match = false;
					break;
				}
				if (match)
				{
					// Because we go backwards, the new (lower) elements are inserted in the middle
					motifAnnotations.add(forwardMotifCount, 
							new Annotation(
									motif.descriptionSB.toString() + " [cr]",
									i - 1, i - motif.getLength() - 1, true, false));
					progressMonitor.subTask(
						String.format(
							monitorMessage + "%n" + " Hits found: " + 
							numberFormat.format(motifAnnotations.size())));
				}
	
				if (progressMonitor.isCanceled())
				{
					throw new OperationCanceledException();
				}
			}
		}
		
		return motifAnnotations;
	}

	/**
	 * 
	 * @param index
	 * @param motif
	 * @param matchAmbiguous
	 * @return
	 * @throws InterruptedException 
	 * @throws InvalidSequenceClassException
	 *
	 * @since	1.5
	 */
	public List<List<Annotation>> findMotifsList
	(int fromIndex, BioSequence motif, boolean matchAmbiguous, 
	 boolean forward, boolean reverse, boolean complement, boolean reverseComplement, 
	 IProgressMonitor progressMonitor)
	throws IllegalArgumentException, IndexOutOfBoundsException, OperationCanceledException
	{
		List<List<Annotation>> motifAnnotationsList = new ArrayList<List<Annotation>>();
		
		// Repack the annotations to have uniform return values like the clusters method
		for (Annotation a :
			 findMotifs(
				fromIndex, motif, matchAmbiguous, forward, reverse, complement, reverseComplement,
				progressMonitor))
		{
			List<Annotation> la = new ArrayList<Annotation>(1);
			la.add(a);
			motifAnnotationsList.add(la);
		}
		
		return motifAnnotationsList;
	}

	/**
	 * Returns all the annotations where any combination of motifs are found in this sequence, within a distance in 
	 * bases. The minimum number of motifs required in a cluster may also be specified.
	 * <p>
	 * The clusters reported by this method are disjoint. This means that if several motifs occur in overlapping 
	 * positions, the ones located nearer
	 * the initial positions of <code>motifs</code> will be the ones found. In
	 * order to find all possible combinations see
	 * {@link findOverlappingClusters}.
	 * <p>
	 * Note: If the <code>motifs</code> argument has only one sequence and
	 * <code>maxDistance</code> is 0 the return value is the same as
	 * {@link findMotifs}.
	 * @param motifs
	 *            <code>List</code> of sequences to look for.
	 * @param maxDistance
	 *            maximum distance where two or more motifs can occur and be
	 *            considered to belong in the same cluster.
	 * @param minMotifs
	 *            minimum number of motifs required in clusters.
	 * 
	 * @return <code>List</code> of annotations of the clusters found. The
	 *         descriptions have the clusters sequentially numbered, plus the
	 *         motifs found in it and in which positions they are. If no motifs
	 *         are found within <code>maxDistance</code> bases, an empty list is
	 *         returned.
	 * @throws InterruptedException 
	 * 
	 * @throws InvalidSequenceClassException
	 * @throws AnnotationInvalidRangeException
	 * 
	 * @see SequenceAnnotation
	 * 
	 * @since 1.5
	 */
	public List<List<Annotation>> findDisjointClusters
	(int fromIndex, List<BioSequence> motifs, List<Boolean> matchAmbiguous,
	 List<Boolean> forward, List<Boolean> reverse, List<Boolean> complement, List<Boolean> reverseComplement,
	 int maxDistance, int minMotifs, IProgressMonitor progressMonitor)
	throws IllegalArgumentException, IndexOutOfBoundsException, OperationCanceledException
	{
		// Check for errors
		if (motifs.size() != matchAmbiguous.size() || motifs.size() != forward.size() || 
			motifs.size() != reverse.size() || motifs.size() != complement.size() || 
			motifs.size() != complement.size() || motifs.size() != reverseComplement.size())
		{
			throw new IllegalArgumentException(
				"The size of the motifs list does not match the options list (ambiguous symbols and reverse " +
				"search lists)");
		}
		for (BioSequence motif : motifs)
		{
			if (motif.getClass() != this.getClass())
			{
				throw new IllegalArgumentException(
					motif.getClass().getName() + " motif queried for a sequence instance of " + 
					getClass().getName());
			}
		}
		
		// Collect and sort all motif positions
		List<Annotation> motifAnnotations = new ArrayList<Annotation>();
		for (int i = 0 ; i < motifs.size() ; i++)
		{
			motifAnnotations.addAll(
				findMotifs(
					fromIndex, motifs.get(i), matchAmbiguous.get(i), 
					forward.get(i), reverse.get(i), complement.get(i), reverseComplement.get(i),
					progressMonitor));
		}
		Collections.sort(motifAnnotations);
	
		// Find clusters
		int previousWindowStart = fromIndex;
		List<List<Annotation>> clusterList = new ArrayList<List<Annotation>>();
		while (!motifAnnotations.isEmpty())
		{
			progressMonitor.subTask("Searching for disjoint clusters.");
			
			// Use first motif in the sequence to anchor the cluster "window"
			List<Annotation> cluster = new ArrayList<Annotation>();
			cluster.add(motifAnnotations.remove(0));
	
			int windowStart = cluster.get(0).normalizedStart();
			int windowEnd = windowStart + maxDistance;
	
			// Keep adding motifs until we go over the window end
			Annotation nextMotif;
			ListIterator<Annotation> it = motifAnnotations.listIterator();
			while (it.hasNext())
			{
				nextMotif = it.next();
				// If the next motif is not beyond the window end...
				if (nextMotif.normalizedEnd() <= windowEnd)
				{
					cluster.add(nextMotif);
				}
				
				// If we've reached window end or last motif and we have enough motifs...
				if ((nextMotif.normalizedEnd() >= windowEnd || !it.hasNext()) && cluster.size() >= minMotifs)
				{
					clusterList.add(cluster);
					progressMonitor.subTask(
						String.format(
							"Searching for disjoint clusters.%n" + "Hits : " + 
							numberFormat.format(clusterList.size())));
					// Go back an extra position not to remove nextMotif
					for (int i = it.previousIndex() - 1; i >= 0; i--)
					{
						motifAnnotations.remove(i);
					}
					break;
				}
			}
			progressMonitor.worked(windowStart - previousWindowStart);
			previousWindowStart = windowStart;
			
			if (progressMonitor.isCanceled())
			{
				throw new OperationCanceledException();
			}
		}
		return clusterList;
	}

	/**
	 * Returns all the annotations where any combination of motifs are found in
	 * this sequence, within a distance in bases. The minimum number of motifs
	 * required in a cluster may also be specified.
	 * <p>
	 * The clusters reported by this method are overlapping. This means that if
	 * several motifs occur in the same positions, all possible combinations are
	 * annotated. However, small clusters occurring in bigger clusters are
	 * ignored.
	 * <p>
	 * @param motifs
	 *            <code>List</code> of sequences to look for.
	 * @param maxDistance
	 *            maximum distance where two or more motifs can occur and be
	 *            considered to belong in the same cluster.
	 * @param minMotifs
	 *            minimum number of motifs required in clusters.
	 * 
	 * @return <code>List</code> of annotations of the clusters found. The
	 *         descriptions have the clusters sequentially numbered, plus the
	 *         motifs found in it and in which positions they are. If no motifs
	 *         are found within <code>maxDistance</code> bases, an empty list is
	 *         returned.
	 * @throws InterruptedException 
	 * 
	 * @throws InvalidSequenceClassException
	 * @throws AnnotationInvalidRangeException
	 * 
	 * @see Annotation
	 * 
	 * @since 1.5
	 */
	public List<List<Annotation>> findOverlappingClusters
	(int index, List<BioSequence> motifs, List<Boolean> matchAmbiguous,
	 List<Boolean> forward, List<Boolean> reverse, List<Boolean> complement, List<Boolean> reverseComplement, 
	 int maxDistance, int minMotifs, IProgressMonitor progressMonitor)
	throws IllegalArgumentException, IndexOutOfBoundsException, OperationCanceledException
	{
		// Check for errors
		if (motifs.size() != matchAmbiguous.size() || motifs.size() != forward.size() || 
			motifs.size() != reverse.size() || motifs.size() != complement.size() || 
			motifs.size() != complement.size() || motifs.size() != reverseComplement.size())
		{
			throw new IllegalArgumentException(
				"The size of the motifs list does not match the options list (ambiguous symbols and reverse " +
				"search lists)");
		}
		for (BioSequence motif : motifs)
		{
			if (motif.getClass() != this.getClass())
			{
				throw new IllegalArgumentException(
					motif.getClass().getName() + " motif queried for a sequence instance of " + 
					getClass().getName());
			}
		}
		
		// Collect and sort all motif positions
		List<Annotation> motifAnnotations = new ArrayList<Annotation>();
		for (int i = 0 ; i < motifs.size() ; i++)
		{
			motifAnnotations.addAll(
				findMotifs(
					index, motifs.get(i), matchAmbiguous.get(i),
					forward.get(i), reverse.get(i), complement.get(i), reverseComplement.get(i), 
					progressMonitor));
		}
		Collections.sort(motifAnnotations);
	
		// Find clusters
		int previousWindowStart = index;
		List<List<Annotation>> clusterList = new ArrayList<List<Annotation>>();
		List<Annotation> previousCluster = new ArrayList<Annotation>(0);
		while (!motifAnnotations.isEmpty())
		{
			progressMonitor.subTask("Searching for overlapping clusters.");
			
			// Use first motif in the sequence to anchor the cluster "window"
			List<Annotation> cluster = new ArrayList<Annotation>();
			cluster.add(motifAnnotations.remove(0));
	
			int windowStart = cluster.get(0).normalizedStart();
			int windowEnd = windowStart + maxDistance;	// Exclusive
	
			// Add motifs until one goes over windowEnd
			Annotation nextMotif;
			ListIterator<Annotation> it = motifAnnotations.listIterator();
			while (it.hasNext())
			{
				nextMotif = it.next();
				// If the next motif is not beyond the window end...
				if (nextMotif.normalizedEnd() <= windowEnd)
				{
					cluster.add(nextMotif);
				}
	
				// If we've reached window end or last motif and we have enough motifs...
				if ((nextMotif.normalizedEnd() > windowEnd || !it.hasNext()) && cluster.size() >= minMotifs)
				{
					// ... and this cluster is not a subcluster of the last one...
					if (previousCluster.containsAll(cluster))
					{
						break;
					}
					// ... add it to the search results
					else
					{
						clusterList.add(cluster);
						progressMonitor.subTask(
							String.format(
								"Searching for overlapping clusters.%n" + "Hits : " + 
								numberFormat.format(cluster.size())));
						previousCluster = cluster;
						break;
					}
				}
			}
			progressMonitor.worked(windowStart - previousWindowStart);
			previousWindowStart = windowStart;
			
			if (progressMonitor.isCanceled())
			{
				throw new OperationCanceledException();
			}
		}
		return clusterList;
	}
	
	
}