/*
 * @author		Alfonso Muñoz-Pomer Fuentes, 
 * 				<a href="mailto:alfonso.munozpomer@biotechvana.com">
 * 				alfonso.munozpomer@biotechvana.com</a>,  
 * 				<a href="http://www.biotechvana.com">Biotechvana</a>
 *
 * @date		2010-11-01
 * 
 * @license		See <a href="http://www.biotechvana.com></a>
 *
 * @copyright	Copyright Biotech Vana, S.L. 2006-2010
 */

package com.biotechvana.javabiotoolkit.utils;

import java.util.ArrayList;
import java.util.List;

/**
 * <code>Permutator</code>s create lists of all possible positive integer-valued vectors over a multidimensional space, 
 * up to a specified value for each of the finite dimensions.
 *
 * @version	1.1, 2010-11-01
 * 
 * @author	<a href="mailto:alfonso.munozpomer@biotechvana.com">Alfonso Muñoz-Pomer Fuentes</a>,
 * 			<a href="http://www.biotechvana.com">Biotechvana</a>.
 */
public class Permutator
{
	// The indices up to which the permutations are generated
	private List<Integer> maxIndices;
	
	/**
	 * Constructs a permutator for a n-dimensional space. Where n = <code>maxIndices.size()</code> and each position in 
	 * <code>maxIndices</code> holds the number of possible values for each of the dimensions.
	 * <p>
	 * For an example, see {@link #generatePermutations()}.
	 * 
	 * @param	maxIndices	defines the number of dimensions and the &ldquo;size&rdquo; of each of them.
	 * 
	 * @since	0.1
	 */
	public Permutator(List<Integer> maxIndices)
	{
		this.maxIndices = maxIndices;
	}
	
	/**
	 * Example:<br /> 
	 * If an instance is created with the list (4, 9, 1, 6, 3), the number of elements represents the number of 
	 * dimensions, (5 in this example) and we obtain a list with the following vectors (each one of them represented as 
	 * a <code>List&lt;Integer%gt;</code>): 
	 * <br />
	 * (0, 0, 0, 0, 0) <br />
	 * (0, 0, 0, 0, 1) <br />
	 * (0, 0, 0, 0, 2) <br />
	 * (0, 0, 0, 1, 0) <br />
	 * ... <br />
	 * (0, 0, 0, 5, 2) <br />
	 * (0, 0, 1, 0, 0) <br />
	 * ... <br />
	 * (0, 0, 0, 5, 2) <br />
	 * (0, 1, 0, 0, 0) <br />
	 * ... <br />
	 * (0, 8, 0, 5, 2) <br /> 
	 * (1, 0, 0, 0, 0) <br />
	 * ... <br />
	 * (3, 8, 0, 5, 2) <br />
	 * 
	 * @return	a list holding all positive integer-valued lists as explained above.
	 * 
	 * @since	0.1
	 */
	public List<List<Integer>> generatePermutations()
	{
		return generatePermutations(new ArrayList<Integer>(0));
	}
	
	/*
	 * Recursive method that starts with a zero-sized list and expands it to maxIndices size and starts filling them 
	 * up using the values in maxIndices.
	 */
	private List<List<Integer>> generatePermutations(List<Integer> partialIndices)
	{
		/// Before going any further, beware of EXPSPACE! It's a nine-headed combinatorial monster lurking behind 
		/// all power sets. You have been warned!
		
		List<List<Integer>> permutations = new ArrayList<List<Integer>>();
		
		int recursionLevel = partialIndices.size();
		
		// If we are in the last index, add these elements and "bottom out"
		if (recursionLevel == maxIndices.size() - 1)
		{
			for (int i = 0 ; i < maxIndices.get(recursionLevel) ; i++)
			{
				List<Integer> p = new ArrayList<Integer>(partialIndices);
				p.add(i);
				permutations.add(p);
			}
			return permutations;
		}
		// If not in the last recursive call create recursively (0, 0, 0, ... ,0), fill in the last values with the 
		// bottom loop above, increase the penultimate value one, fill again recursively, etc.
		else
		{
			for (int i = 0 ; i < maxIndices.get(recursionLevel) ; i++)
			{
				List<Integer> p = new ArrayList<Integer>(partialIndices);
				p.add(i);
				// ... generate this step indices, fill each value to the end recursively...
				for (List<Integer> l : generatePermutations(p))
				{
					// ... and "extract-add" them to the permutations
					permutations.add(l);
				}
			}
			return permutations;
		}
	}

	/*
	 * Tester inner class (remove DNASequence$Tester.class in final build)
	 */
	/*
	@SuppressWarnings("unused")
	private static class Tester
	{
		public static void main (String[] args)
		{
			List<Integer> indices = new ArrayList<Integer>(4);
			indices.add(3);
			indices.add(4);
			indices.add(5);
			indices.add(2);
			System.out.println(indices);

			Permutator ptator = new Permutator(indices);
			System.out.println(ptator.generatePermutations());
		}
	}
	*/
}
