package com.biotechvana.javabiotoolkit.io;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.charset.Charset;
import java.util.List;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.Map.Entry;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;

import com.biotechvana.javabiotoolkit.BioSequence;
import com.biotechvana.javabiotoolkit.exceptions.FastaReaderNotParsedException;
import com.biotechvana.javabiotoolkit.exceptions.FastaWriterMultipleSequenceException;
import com.biotechvana.javabiotoolkit.exceptions.InvalidSequenceCharacterException;
import com.biotechvana.javabiotoolkit.exceptions.SequenceTooLongException;
import com.biotechvana.javabiotoolkit.text.LineSeparatorFormat;
import com.biotechvana.javabiotoolkit.text.NewLineFileEnder;

public class FASTAWriter2 {
	int lineWrap = 72;

	public void setLineWrap(int lineWrap) {
		if (lineWrap < 0 || lineWrap > FASTAWriter.I_maxWrap)
		{
			throw new IllegalArgumentException(lineWrap + " illegal wrap value. Must be in the range 0.." + FASTAWriter.I_maxWrap);
		}
		this.lineWrap = lineWrap;
	}

	FASTAFileHandler fastaFileHandler;
	File filePath;
	public FASTAWriter2(File filePath, FASTAFileHandler fastaFileHandler) {
		this.fastaFileHandler = fastaFileHandler;
		this.filePath  = filePath;
	}
	public FASTAWriter2(FASTAFileHandler fastaFileHandler) {
		this.fastaFileHandler = fastaFileHandler;
		this.filePath  = fastaFileHandler.getFilePath();
	}





	/**
	 * Call to append new sequence to the file , no record has beed created before
	 * @param bs , sequence to add with all info desc and seq
	 * @param progressMonitor
	 * @return
	 * @throws IllegalArgumentException
	 * @throws FileNotFoundException
	 * @throws IOException
	 * @throws SequenceTooLongException
	 * @throws InvalidSequenceCharacterException
	 */
	public FASTAFileRecord append(FASTAFileRecord oldRecord,  BioSequence bs, IProgressMonitor progressMonitor)
			throws FileNotFoundException, IOException, SequenceTooLongException, InvalidSequenceCharacterException
	{

		// ############################ Record
		// To keep track of the offset within the file
		long fileByteOffset = 0;
		long cBytes = 0;
		// To keep track of the description block 
		long descriptionByteOffset = 0;
		long descriptionBytes = 0;
		// To keep track of the sequence block
		long sequenceByteOffset = 0;
		long sequenceBytes = 0;
		long sequenceLength = 0;
		SortedMap<Long, Long> componentsByteOffsets =  new TreeMap<Long, Long>(); 
		// ###############################################

		//File filePath = fastaFileHandler.getFilePath();

		Charset fileCharset = fastaFileHandler.getFileCharset();

		LineSeparatorFormat fileLineSeparatorFormat = fastaFileHandler.getFileLineSeparatorFormat();


		NewLineFileEnder.ensureLastLineNewLine(filePath, fileCharset, fileLineSeparatorFormat);

		// TODO :: get the same record and modify it
		//		FASTAFileRecord lastRecord = fastaFileHandler.getFastaRecords().lastRecord();

		// Open channel and go to end
		FileChannel outFC =	new RandomAccessFile(filePath, "rw").getChannel();	// FileNotFoundException
		// Lock channel for writing
		FileLock outLock = null;
		try
		{
			progressMonitor.subTask("Appending sequence to file : " + bs.getDescription());
			outFC.position(outFC.size());	// IOException

			fileByteOffset = outFC.position();
			descriptionByteOffset = fileByteOffset;

			outLock = outFC.tryLock();
			if (outLock != null)
			{	
				// Allocate CharBuffer
				CharBuffer cBuffer = CharBuffer.allocate(FASTAWriter.I_charBufferSize * 2);	// The extra space avoids an overflow
				// Write FASTA record mark and description in portions of I_charBufferSize
				cBytes  = outFC.write(ByteBuffer.wrap(">".getBytes(fileCharset)));

				descriptionBytes+= cBytes;
				fileByteOffset+= cBytes;

				for (int i = 0 ; i < bs.getDescription().length() ; i += FASTAWriter.I_charBufferSize)
				{
					for (int j = 0 ; (j < FASTAWriter.I_charBufferSize) && ((i + j) < bs.getDescription().length()) ; j++)
					{
						cBuffer.put(bs.getDescription().charAt(i + j));
					}
					cBuffer.flip();
					cBytes = outFC.write(fileCharset.encode(cBuffer));
					descriptionBytes+= cBytes;
					fileByteOffset+= cBytes;
					// progressMonitor.worked(cBuffer.capacity());
					cBuffer.clear();
				}

				if (progressMonitor.isCanceled())
				{
					throw new OperationCanceledException();
				}
				sequenceByteOffset = outFC.position();
				// sequenceLength = bs.getLength();
				// Write sequence in portions of charsPerIteration with line wrap
				for (int i = 0 ; i < bs.getLength() ; i += FASTAWriter.I_charBufferSize)
				{

					for (int j = 0 ; j < FASTAWriter.I_charBufferSize && i + j < bs.getLength() ; j++)
					{
						cBytes = 0;
						if (((i + j) % lineWrap) == 0)
						{
							cBuffer.put(fileLineSeparatorFormat.lineSeparator());
							//fileByteOffset+= fileLineSeparatorFormat.lineSeparator().length();
							cBytes += fileLineSeparatorFormat.lineSeparator().length();

							if((i+j) == 0  ) {
								// this is part of the description line
								descriptionBytes+=cBytes;
								sequenceByteOffset+=cBytes;
								fileByteOffset += cBytes;
								cBytes -= fileLineSeparatorFormat.lineSeparator().length();

							}
						}
						cBuffer.put(bs.get(i + j).getUpperCaseChar()).limit();
						cBytes+= 1;
						if (sequenceLength % IContigProvider.I_componentLength == 0)
						{
							componentsByteOffsets.put(sequenceLength, fileByteOffset);
						}
						sequenceLength++;
						fileByteOffset += cBytes;
					}
					cBuffer.flip();
					cBytes = outFC.write(fileCharset.encode(cBuffer));
					sequenceBytes+= cBytes;

					// progressMonitor.worked(cBuffer.capacity());
					cBuffer.clear();
				}

				// put new line at the end
				{
					cBuffer.put(fileLineSeparatorFormat.lineSeparator());
					cBytes = outFC.write(fileCharset.encode(fileLineSeparatorFormat.lineSeparator()));

					//					cBytes = fileLineSeparatorFormat.lineSeparator().length();
					//sequenceBytes+= cBytes;
					fileByteOffset += cBytes;
				}
				if (progressMonitor.isCanceled())
				{
					throw new OperationCanceledException();
				}
			}
			progressMonitor.worked(1);
		}
		finally
		{
			outLock.release();
			outFC.close();

		}
		componentsByteOffsets.put(sequenceLength, sequenceByteOffset + sequenceBytes);




		FASTAFileRecord newRecord =  new FASTAFileRecord(fastaFileHandler.getFilePath(), fastaFileHandler.getFileCharset(), fastaFileHandler.getFileLineSeparatorFormat() ,
				0, 0,
				0, 0, new TreeMap<Long, Long>() ,
				0);

		newRecord.descriptionByteOffset = descriptionByteOffset;
		newRecord.descriptionBytes = descriptionBytes;
		newRecord.sequenceByteOffset = sequenceByteOffset;
		newRecord.sequenceBytes = sequenceBytes;
		newRecord.componentsByteOffsets.clear();
		newRecord.componentsByteOffsets.putAll(componentsByteOffsets);
		newRecord.sequenceLength = sequenceLength;
		newRecord.headerLineSB = new StringBuilder();
		newRecord.headerLineSB.append(bs.getDescription());
		return newRecord;
	}


	/**
	 * replace sequence in a file
	 * @param oldRecord
	 * @param bs
	 * @param progressMonitor
	 * @return
	 * @throws FileNotFoundException
	 * @throws IOException
	 * @throws FastaWriterMultipleSequenceException
	 * @throws SequenceTooLongException
	 * @throws InvalidSequenceCharacterException
	 */
	public FASTAFileRecord replace(FASTAFileRecord oldRecord,BioSequence bs, IProgressMonitor progressMonitor)
			throws  FileNotFoundException, IOException, 
			FastaWriterMultipleSequenceException, SequenceTooLongException, InvalidSequenceCharacterException
	{

		File filePath = fastaFileHandler.getFilePath();

		Charset fileCharset = fastaFileHandler.getFileCharset();

		LineSeparatorFormat fileLineSeparatorFormat = fastaFileHandler.getFileLineSeparatorFormat();


		// Extra check: If the record does not exist append it to the end of the file
		// first find record
		progressMonitor.subTask("Locating sequence...");
		List<FASTAFileRecord> results = fastaFileHandler.getFastaRecords().findRecords(bs.getOrginalDescription());
		//				FASTAFileRecord seqRecord = results.get(0);

		if (results.size() == 0)
		{	
			return append(oldRecord,bs, progressMonitor);
		}
		// Copy to temp file, append sequence, skip the old record, copy the rest of the original file, delete it and 
		// copy the temp file back to the original location
		//else if (results.size() == 1)
		{	
			FileChannel originalFC = null;
			FileLock originalLock = null;
			FileChannel tempFC = null;
			FileLock tempLock = null;

			// ############################ Record
			// To keep track of the offset within the file
			long fileByteOffset = 0;
			long cBytes = 0;
			// To keep track of the description block 
			long descriptionByteOffset = 0;
			long descriptionBytes = 0;
			// To keep track of the sequence block
			long sequenceByteOffset = 0;
			long sequenceBytes = 0;
			long sequenceLength = 0;
			SortedMap<Long, Long> componentsByteOffsets =  new TreeMap<Long, Long>(); ;

			try
			{
				// FASTAFileRecord oldRecord =  results.get(0);//  far.findRecords(bs.getOrginalDescription()).get(0);
				int oldRecordIndex = fastaFileHandler.getFastaRecords().indexOf(oldRecord);
				long oldRecordStartByte = oldRecord.offset();
				// End byte is the end of the file or the start of the next sequence
				long oldRecordEndByte = filePath.length();
				if (oldRecordIndex < fastaFileHandler.getFastaRecords().size() - 1)
				{
					oldRecordEndByte = fastaFileHandler.getFastaRecords().get(oldRecordIndex + 1).offset();
				}

				// Create a new temp file
				File tempFilePath = File.createTempFile("time.", null);

				originalFC = new RandomAccessFile(filePath, "rw").getChannel();
				tempFC = new RandomAccessFile(tempFilePath, "rw").getChannel();

				progressMonitor.subTask("Writing temporary file...");
				tempLock = tempFC.tryLock();
				originalLock = originalFC.tryLock();
				if (tempLock != null && originalLock != null)
				{
					// Copy up to old sequence
					originalFC.transferTo(0, oldRecordStartByte, tempFC);
					progressMonitor.worked((int)oldRecordStartByte);
					// Write new sequence
					progressMonitor.subTask("Adding sequence...");
					tempFC.position(tempFC.size());	// IOException

					fileByteOffset = tempFC.position();
					descriptionByteOffset = fileByteOffset;

					// Allocate CharBuffer
					CharBuffer cBuffer = CharBuffer.allocate(FASTAWriter.I_charBufferSize * 2);	// The extra space avoids an overflow
					// Write FASTA record mark and description in portions of I_charBufferSize
					cBytes = tempFC.write(ByteBuffer.wrap(">".getBytes(fileCharset)));

					descriptionBytes+= cBytes;
					fileByteOffset+= cBytes;

					progressMonitor.worked(1);
					for (int i = 0 ; i < bs.getDescription().length() ; i += FASTAWriter.I_charBufferSize)
					{
						for (int j = 0 ; (j < FASTAWriter.I_charBufferSize) && ((i + j) < bs.getDescription().length()) ; j++)
						{
							cBuffer.put(bs.getDescription().charAt(i + j));
						}
						cBuffer.flip();
						cBytes =  tempFC.write(fileCharset.encode(cBuffer));
						descriptionBytes+= cBytes;
						fileByteOffset+= cBytes;
						progressMonitor.worked(cBuffer.limit());
						cBuffer.clear();
					}

					sequenceByteOffset = tempFC.position();


					// Write sequence in portions of charsPerIteration with line wrap
					for (int i = 0 ; i < bs.getLength() ; i += FASTAWriter.I_charBufferSize)
					{
						for (int j = 0 ; j < FASTAWriter.I_charBufferSize && i + j < bs.getLength() ; j++)
						{
							cBytes = 0;
							if (((i + j) % lineWrap) == 0)
							{
								//TODO Use fileLineSeparartor (more than one char can truncate the buffer)
								cBuffer.put(fileLineSeparatorFormat.lineSeparator());
								cBytes += fileLineSeparatorFormat.lineSeparator().length();
								if((i+j) == 0  ) {
									// this is part of the description line
									descriptionBytes+=cBytes;
									sequenceByteOffset+=cBytes;
									fileByteOffset += cBytes;
									cBytes -= fileLineSeparatorFormat.lineSeparator().length();

								}
							}
							cBuffer.put(bs.get(i + j).getUpperCaseChar());
							cBytes+= 1;
							if (sequenceLength % IContigProvider.I_componentLength == 0)
							{
								componentsByteOffsets.put(sequenceLength, fileByteOffset);
							}
							sequenceLength++;
							fileByteOffset += cBytes;
						}
						cBuffer.flip();
						cBytes = tempFC.write(fileCharset.encode(cBuffer));
						sequenceBytes+= cBytes;
						progressMonitor.worked(cBuffer.limit());
						cBuffer.clear();
					}
					cBytes = tempFC.write(fileCharset.encode(fileLineSeparatorFormat.lineSeparator()));
					fileByteOffset += cBytes;
					long restOffSet = tempFC.position();
					progressMonitor.worked(fileLineSeparatorFormat.lineSeparator().length());
					//					tempLock.release();
					// Copy rest of the file
					progressMonitor.subTask("Adding remaining sequences...");
					originalFC.transferTo(oldRecordEndByte, originalFC.size(), tempFC);
					progressMonitor.worked((int)(originalFC.size() - oldRecordEndByte));

					// "Delete" original file
					originalFC.truncate(0);

					// Copy temp file to original location
					progressMonitor.subTask("Rebuilding file...");
					tempFC.transferTo(0, tempFC.size(), originalFC);
					progressMonitor.worked((int)tempFC.size());
					originalLock.release();
					tempLock.release();

					// Delete temp file and update FastaReader
					tempFilePath.delete();
					//					far.parse(true, progressMonitor);



					// TODO shift all records
					long shiftOffset = 0;
					for(int restR = oldRecordIndex+1; restR < fastaFileHandler.getFastaRecords().size();restR++ ) {
						FASTAFileRecord sRecord = fastaFileHandler.getFastaRecords().get(restR);
						if(shiftOffset == 0) {
							shiftOffset = restOffSet - sRecord.offset();
							if(shiftOffset == 0)
								// if still 0 then nothing happened do nothing
								break;
						}
						sRecord.shiftOffset(shiftOffset);
					}
				}
			}
			finally
			{
				originalFC.close();
				tempFC.close();
			}
			componentsByteOffsets.put(sequenceLength, sequenceByteOffset + sequenceBytes);

			FASTAFileRecord newRecord =  new FASTAFileRecord(fastaFileHandler.getFilePath(), fastaFileHandler.getFileCharset(), fastaFileHandler.getFileLineSeparatorFormat() ,
					0, 0,
					0, 0, new TreeMap<Long, Long>() ,
					0);

			newRecord.descriptionByteOffset = descriptionByteOffset;
			newRecord.descriptionBytes = descriptionBytes;
			newRecord.sequenceByteOffset = sequenceByteOffset;
			newRecord.sequenceBytes = sequenceBytes;
			newRecord.componentsByteOffsets.clear();
			newRecord.componentsByteOffsets.putAll(componentsByteOffsets);
			newRecord.sequenceLength = sequenceLength;
			newRecord.headerLineSB = new StringBuilder();
			newRecord.headerLineSB.append(bs.getDescription());
			
//			oldRecord.descriptionByteOffset = descriptionByteOffset;
//			oldRecord.descriptionBytes = descriptionBytes;
//			oldRecord.sequenceByteOffset = sequenceByteOffset;
//			oldRecord.sequenceBytes = sequenceBytes;
//			oldRecord.componentsByteOffsets.clear();
//			oldRecord.componentsByteOffsets.putAll(componentsByteOffsets);
//			oldRecord.sequenceLength = sequenceLength;
			return newRecord;
		}
		//else // if (far.findRecords(bs.getDescription()).size() > 1)
		//{
		//	throw new FastaWriterMultipleSequenceException("\"" + 
		//			bs.getDescription() + "\" has " + results.size() + 
		//			" copies in the destination file");
		//}
	}



	/** copy one record back to the ref file
	 * @return 
	 * 
	 */
	public FASTAFileRecord copy(String descLine,FASTAFileRecord oldRec, IProgressMonitor progressMonitor) 
	throws FileNotFoundException, IOException, SequenceTooLongException, InvalidSequenceCharacterException {
		// ############################ Record
		// To keep track of the offset within the file
		long fileByteOffset = 0;
		long cBytes = 0;
		// To keep track of the description block 
		long descriptionByteOffset = 0;
		long descriptionBytes = 0;
		// To keep track of the sequence block
		long sequenceByteOffset = 0;
		long sequenceBytes = 0;
		long sequenceLength = 0;
		SortedMap<Long, Long> componentsByteOffsets =  new TreeMap<Long, Long>(); 
		// ###############################################

		//File filePath = fastaFileHandler.getFilePath();

		Charset fileCharset = fastaFileHandler.getFileCharset();

		LineSeparatorFormat fileLineSeparatorFormat = fastaFileHandler.getFileLineSeparatorFormat();


		NewLineFileEnder.ensureLastLineNewLine(filePath, fileCharset, fileLineSeparatorFormat);

		// TODO :: get the same record and modify it
		//				FASTAFileRecord lastRecord = fastaFileHandler.getFastaRecords().lastRecord();

		// Open channel and go to end
		FileChannel outFC =	new RandomAccessFile(filePath, "rw").getChannel();	// FileNotFoundException
		// Lock channel for writing
		FileLock outLock = null;
		try
		{
			progressMonitor.subTask("Appending sequence to file : " + descLine);
			outFC.position(outFC.size());	// IOException

			fileByteOffset = outFC.position();
			descriptionByteOffset = fileByteOffset;

			outLock = outFC.tryLock();
			if (outLock != null)
			{	
				// Allocate CharBuffer
				CharBuffer cBuffer = CharBuffer.allocate(FASTAWriter.I_charBufferSize * 2);	// The extra space avoids an overflow
				// Write FASTA record mark and description in portions of I_charBufferSize
				cBytes  = outFC.write(ByteBuffer.wrap(">".getBytes(fileCharset)));

				descriptionBytes+= cBytes;
				fileByteOffset+= cBytes;

				for (int i = 0 ; i <  descLine.length() ; i += FASTAWriter.I_charBufferSize)
				{
					for (int j = 0 ; (j < FASTAWriter.I_charBufferSize) && ((i + j) < descLine.length()) ; j++)
					{
						cBuffer.put(descLine.charAt(i + j));
					}
					cBuffer.flip();
					cBytes = outFC.write(fileCharset.encode(cBuffer));
					descriptionBytes+= cBytes;
					fileByteOffset+= cBytes;
					// progressMonitor.worked(cBuffer.capacity());
					cBuffer.clear();
				}

				if (progressMonitor.isCanceled())
				{
					throw new OperationCanceledException();
				}
				cBytes = outFC.write(fileCharset.encode(fileLineSeparatorFormat.lineSeparator()));
				descriptionBytes+=cBytes;
				sequenceByteOffset = outFC.position();
				sequenceBytes = oldRec.sequenceBytes;
				sequenceLength = oldRec.sequenceLength;
				FileChannel originalFC = new RandomAccessFile(oldRec.filePath(), "r").getChannel();
//				originalFC.position(oldRec.sequenceByteOffset);
				originalFC.transferTo(oldRec.sequenceByteOffset, oldRec.sequenceBytes, outFC);
				long shiftOffset = sequenceByteOffset - oldRec.sequenceByteOffset;
				for(Entry<Long, Long> ent : oldRec.componentsByteOffsets.entrySet()) {
					componentsByteOffsets.put(ent.getKey(), ent.getValue()+shiftOffset);
				}
				originalFC.close();
				// copy sequence from old file to new file
					
				if (progressMonitor.isCanceled())
				{
					throw new OperationCanceledException();
				}
			}
			progressMonitor.worked(1);
		}
		finally
		{
			outLock.release();
			outFC.close();

		}
		componentsByteOffsets.put(sequenceLength, sequenceByteOffset + sequenceBytes);




		FASTAFileRecord newRecord =  new FASTAFileRecord(fastaFileHandler.getFilePath(), fastaFileHandler.getFileCharset(), fastaFileHandler.getFileLineSeparatorFormat() ,
				0, 0,
				0, 0, new TreeMap<Long, Long>() ,
				0);

		newRecord.descriptionByteOffset = descriptionByteOffset;
		newRecord.descriptionBytes = descriptionBytes;
		newRecord.sequenceByteOffset = sequenceByteOffset;
		newRecord.sequenceBytes = sequenceBytes;
		newRecord.componentsByteOffsets.clear();
		newRecord.componentsByteOffsets.putAll(componentsByteOffsets);
		newRecord.sequenceLength = sequenceLength;
		newRecord.headerLineSB = new StringBuilder();
		newRecord.headerLineSB.append(descLine);
		return newRecord;		
	}


}
