/*
 * @author		Alfonso Muñoz-Pomer Fuentes,
 * 				<a href="mailto:alfonso.munozpomer@biotechvana.com">
 * 				alfonso.munozpomer@biotechvana.com</a>,
 *				<a href="http://www.biotechvana.com">Biotechvana</a>
 *
 * @date		2011-07-19
 * 
 * @copyright	Copyright Biotech Vana, S.L. 2006-2011
 */

package com.biotechvana.javabiotoolkit;

/**
 * Superclass of the three types of biochemical residues (amino acids, DNA nucleotides and RNA nucleotides) that can 
 * form part of either a polypeptide (i.e. a <code>ProteinSequence</code>) or a nucleic acid (i.e. a 
 * <code>DnaSequence</code> or a <code>RnaSequence</code>).
 * 
 * @version	1.6, 2012-04-24
 * 
 * @author	<a href="mailto:alfonso.munozpomer@biotechvana.com">Alfonso Muñoz-Pomer Fuentes</a>,
 * 			<a href="http://www.biotechvana.com">Biotechvana</a>.
 * 
 * @see		AminoAcid
 * @see		DNABase
 * @see		RNABase
 */
public interface BioResidue
{
	/**
	 * Returns a descriptive <code>String</code> associated to the receiver.
	 * 
	 * @return	this residue&rsquo;s full name.
	 * 
	 * @since	0.1
	 */
	public String getFullName();
	
	/**
	 * Returns a lower case <code>char</code> value representing this residue. In the case of gaps and stop codons 
	 * there is no upper/lower case variant.
	 * 
	 * @return	a lower case <code>char</code> representation of this residue.
	 * 
	 * @since	1.5
	 */
	public char getLowerCaseChar();
	
	/**
	 * Returns an upper case <code>char</code> value representing this resisdue. In the case of gaps and stop codons 
	 * there is no upper/lower case variant.
	 * 
	 * @return	an upper case <code>char</code> representation of this residue.
	 * 
	 * @since	1.5
	 */
	public char getUpperCaseChar();
	
	/**
	 * Returns an array with the receiver&rsquo;s equivalent unambiguous residues. If this is a non-ambiguous residue, 
	 * then a one-element list with this same <code>enum</code> constant is returned.
	 * 
	 * @return	Array of equivalent <code>BioResidue</code>s.
	 * 
	 * @since	1.5
	 */
	public BioResidue[] expandToUnambiguous();
	
	/**
	 * Returns an array containing all the receiver's equivalent ambiguous and unambiguous residues. If an 
	 * unambiguous residue calls this method, the result is the same as calling {@link #expandToUnambiguous()}.
	 * 
	 * @return	Array of equivalent <code>BioResidue</code>s, including the receiver.
	 * 
	 * @since	1.5
	 */
	public BioResidue[] expandToAll();
	
	/**
	 * Returns a regular expression which matches a one-letter abbreviation of this <code>BioResidue</code>. The regex 
	 * is case-insensitive.
	 * <p>
	 * Note: the returned <code>String</code> does not include enclosing square brackets, so it cannot be used directly 
	 * as a character class.
	 * 
	 * @return	one-symbol regular expression.
	 * 
	 * @since	1.5
	 */
	public String getRegex();
}
