package com.biotechvana.netools.ui.editors.internal;

import java.util.HashSet;
import java.util.Set;

import org.eclipse.core.databinding.DataBindingContext;
import org.eclipse.core.databinding.observable.value.WritableValue;
import org.eclipse.e4.core.di.annotations.Optional;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.ColorDialog;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Spinner;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.Section;

import com.biotechvana.netools.INetworkView;
import com.biotechvana.netools.ISelectionEvent;
import com.biotechvana.netools.ISelectionListener;
import com.biotechvana.netools.ISelectionManager;
import com.biotechvana.netools.models.Edge;
import com.biotechvana.netools.models.IEdge;
import com.biotechvana.netools.models.Node;
import com.biotechvana.netools.ui.editors.IEdgeEditor;

import jakarta.annotation.PostConstruct;
import jakarta.inject.Inject;

public class EdgeEditor implements IEdgeEditor {
	
	private DataBindingContext bindingContext;
	private WritableValue<Edge> selectedEdge;
	
	private Text colorText;
	private Text colorHighlightText;
	private Text colorHoverText;
	private Text colorOpacityText;
	private Spinner widthSpinner;
	private Text labelPropertiesText;
	private Text showDashesText;
	private Text hideEdgesText;
	private Text showLabelsText;
	
	Set<Edge> selectedEdges = new HashSet<>();
	
	
	ISelectionManager selectionManager;
    
    INetworkView networkViewer;
    
    ISelectionListener currentSelectionListener;
	

    @Inject  
	public void setSelectionManager(@Optional ISelectionManager selectionManager) {
    	
    	if (this.selectionManager != null && currentSelectionListener != null) {
    		this.selectionManager.removeSelectionListener(currentSelectionListener);
    	}
    	
    	if (selectionManager == null) {
    		this.selectionManager = null;
    		return;
    	}
    	
    	
		this.selectionManager = selectionManager;
		if (currentSelectionListener == null) {
			currentSelectionListener = new ISelectionListener() {



				@Override
				public void nodesSelectionChanged(ISelectionEvent event) {
					// TODO Auto-generated method stub
					
				}

				@Override
				public void edgesSelectionChanged(ISelectionEvent event) {
					// TODO Auto-generated method stub
					// handle selection change event here
					System.out.println("Edge Selection Changed");
					System.out.println(event);
					selectedEdges = event.getSelectedEdges();
					
					if (selectedEdges.size() == 1) {
						setEdge((IEdge) selectedEdges.toArray()[0]);
					} else {
						resetParameters();
					}
					
				}
			};
		}
		
		this.selectionManager.addSelectionListener(currentSelectionListener);
	}
    
    @Inject
	public void setNetworkViewer(@Optional INetworkView networkViewer) {
		this.networkViewer = networkViewer;
	}
    
    
    @PostConstruct
    public void initEditor(Composite parent) {
        FormToolkit toolkit = new FormToolkit(parent.getDisplay());
        this.bindingContext = new DataBindingContext();
        this.selectedEdge = new WritableValue<>(null, Edge.class);

        ScrolledComposite scrolledComposite = new ScrolledComposite(parent, SWT.H_SCROLL | SWT.V_SCROLL);
        scrolledComposite.setExpandHorizontal(true);
        scrolledComposite.setExpandVertical(true);

        Composite editorComposite = toolkit.createComposite(scrolledComposite);
        editorComposite.setLayout(new GridLayout(1, false));
        editorComposite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
        ((GridLayout) editorComposite.getLayout()).marginWidth = 10;
        ((GridLayout) editorComposite.getLayout()).marginHeight = 10;
        
        // APPEARANCE SECTION
        Section appearanceSection = toolkit.createSection(editorComposite, Section.TITLE_BAR | Section.TWISTIE | Section.EXPANDED);
        appearanceSection.setText("Customize Appearance");
        appearanceSection.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false));
        Composite appearanceComposite = toolkit.createComposite(appearanceSection);
        appearanceComposite.setLayout(new GridLayout(3, false)); // Adjusted to 3 columns for color buttons
        appearanceSection.setClient(appearanceComposite);
        
        // Color
        toolkit.createLabel(appearanceComposite, "Color:");
        colorText = toolkit.createText(appearanceComposite, "", SWT.BORDER);
        colorText.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
        Button colorButton = toolkit.createButton(appearanceComposite, "Choose...", SWT.PUSH);
        colorButton.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                ColorDialog colorDialog = new ColorDialog(parent.getShell());
                RGB rgb = colorDialog.open();
                if (rgb != null) {
                    String colorHex = String.format("#%02x%02x%02x", rgb.red, rgb.green, rgb.blue);
                    updateColorInput(colorText, rgb);
                    
                    // CHANGE ASSOCIATED COLOR ATTRIBUTES
                    RGB darker = deriveColor(rgb, 0.75);
                    RGB lighter = deriveColor(rgb, 1.25);
                    updateColorInput(colorHighlightText, lighter);
                    updateColorInput(colorHoverText, lighter);
                                        
                    networkViewer.changeEdgeColor(selectedEdges, colorHex);
                }
            }
        });
                
        // Color Highlight
        toolkit.createLabel(appearanceComposite, "Color Highlight:");
        colorHighlightText = toolkit.createText(appearanceComposite, "", SWT.BORDER);
        colorHighlightText.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
        Button colorHighlightButton = toolkit.createButton(appearanceComposite, "Choose...", SWT.PUSH);
        colorHighlightButton.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                ColorDialog colorDialog = new ColorDialog(parent.getShell());
                RGB rgb = colorDialog.open();
                if (rgb != null) {
                    String colorHex = String.format("#%02x%02x%02x", rgb.red, rgb.green, rgb.blue);
                    updateColorInput(colorHighlightText, rgb);
                    networkViewer.changeEdgeColorHighlight(selectedEdges, colorHex);
                }
            }
        });
        
        // Color Hover
        toolkit.createLabel(appearanceComposite, "Color Hover:");
        colorHoverText = toolkit.createText(appearanceComposite, "", SWT.BORDER);
        colorHoverText.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
        Button colorHoverButton = toolkit.createButton(appearanceComposite, "Choose...", SWT.PUSH);
        colorHoverButton.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                ColorDialog colorDialog = new ColorDialog(parent.getShell());
                RGB rgb = colorDialog.open();
                if (rgb != null) {
                    String colorHex = String.format("#%02x%02x%02x", rgb.red, rgb.green, rgb.blue);
                    updateColorInput(colorHoverText, rgb);
                    networkViewer.changeEdgeColorHover(selectedEdges, colorHex);
                }
            }
        });
        
        // Color Opacity
        toolkit.createLabel(appearanceComposite, "Opacity:");
        colorOpacityText = toolkit.createText(appearanceComposite, "", SWT.BORDER);
        colorOpacityText.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false, 2, 1));
        colorOpacityText.addKeyListener(new KeyAdapter() {
            @Override
            public void keyPressed(KeyEvent e) {
                if (e.keyCode == SWT.CR || e.keyCode == SWT.KEYPAD_CR) {
                    String textValue = colorOpacityText.getText(); 
                    try {
                        float newValue = Float.parseFloat(textValue);
                        networkViewer.changeEdgeColorOpacity(selectedEdges, newValue);
                    } catch (NumberFormatException ex) {
                        System.out.println("Invalid opacity value: " + textValue);
                    }
                }
            }
        });
        
        // Width
        toolkit.createLabel(appearanceComposite, "Width:");
        widthSpinner = new Spinner(appearanceComposite, SWT.BORDER);
        widthSpinner.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false, 2, 1));
        toolkit.adapt(widthSpinner);
        widthSpinner.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                double newValue = widthSpinner.getSelection();
                networkViewer.changeEdgeWidth(selectedEdges, newValue);
            }
        });
        
        // Show Dashes
        toolkit.createLabel(appearanceComposite, "Show Dashes:");
        Button showDashesButton = toolkit.createButton(appearanceComposite, null, SWT.CHECK);
        showDashesButton.addSelectionListener(new SelectionAdapter() {
        	@Override
        	public void widgetSelected(SelectionEvent e) {
        		boolean isSelected = showDashesButton.getSelection();
        		networkViewer.showDashes(isSelected);
        	}
        });
        
        // Show Labels
        toolkit.createLabel(appearanceComposite, " ");
        toolkit.createLabel(appearanceComposite, "Show Labels:");
        Button showLabelsButton = toolkit.createButton(appearanceComposite, null, SWT.CHECK);
        showLabelsButton.addSelectionListener(new SelectionAdapter() {
        	@Override
        	public void widgetSelected(SelectionEvent e) {
        		boolean isSelected = showLabelsButton.getSelection();
        		networkViewer.showEdgeLabels(isSelected);
        	}
        });
        
        // Hide Edges
        toolkit.createLabel(appearanceComposite, " ");
        toolkit.createLabel(appearanceComposite, "Hide Edges:");
        Button hideEdgesButton = toolkit.createButton(appearanceComposite, null, SWT.CHECK);
        hideEdgesButton.addSelectionListener(new SelectionAdapter() {
        	@Override
        	public void widgetSelected(SelectionEvent e) {
        		boolean isSelected = hideEdgesButton.getSelection();
        		networkViewer.hideEdges(isSelected);
        	}
        });
        
        scrolledComposite.setContent(editorComposite);
        scrolledComposite.setMinSize(editorComposite.computeSize(SWT.DEFAULT, SWT.DEFAULT));
    }
    
    
    private void updateColorInput( Text colorInput, RGB rgb) {
        if (rgb != null) {
        	String colorHex = String.format("#%02x%02x%02x", rgb.red, rgb.green, rgb.blue);
        	colorInput.setText(colorHex);
            Color backgroundColor = new Color(colorInput.getDisplay(), rgb);
            colorInput.setBackground(backgroundColor);
            return;
        }
    }
    
    private void updateColorInput( Text colorInput, String colorHex) {
        if (colorHex != "" && colorHex != null) {
        	colorInput.setText(colorHex);
        	if (colorHex.startsWith("#")) colorHex = colorHex.substring(1);
        	int r = Integer.valueOf(colorHex.substring(0, 2), 16);
        	int g = Integer.valueOf(colorHex.substring(2, 4), 16);
        	int b = Integer.valueOf(colorHex.substring(4, 6), 16);
            RGB rgb = new RGB(r, g, b); // convert colorHex to RGB
            Color backgroundColor = new Color(colorInput.getDisplay(),rgb);
            colorInput.setBackground(backgroundColor);
            return;
        } else {
        	// if null reset or clear
            colorInput.setBackground(null);
            colorInput.setText("");
        }
    }
    
    private RGB deriveColor(RGB color, double factor) {
    	int r = Math.max(0, Math.min(255, (int)Math.round(color.red   * factor)));
	    int g = Math.max(0, Math.min(255, (int)Math.round(color.green * factor)));
	    int b = Math.max(0, Math.min(255, (int)Math.round(color.blue  * factor)));
	    return new RGB(r, g, b);
    }
    
    public void setEdge(IEdge edge) {
    	//updateColorInput(colorText, edge.getColor());
    	updateColorInput(colorText, networkViewer.getVisEdgeColor(edge));
    	//updateColorInput(colorHighlightText, edge.getColorHighlight());
    	updateColorInput(colorHighlightText, networkViewer.getVisEdgeColorHighlight(edge));
    	//updateColorInput(colorHoverText, edge.getColorHover());
    	updateColorInput(colorHoverText, networkViewer.getVisEdgeColorHover(edge));
    	colorOpacityText.setText(String.valueOf(edge.getOpacity()));
    	widthSpinner.setSelection((int) edge.getWidth());
    }
    
    public void resetParameters() {
    	updateColorInput(colorText, "");
    	updateColorInput(colorHighlightText, "");
    	updateColorInput(colorHoverText, "");
    	colorOpacityText.setText("");
    	widthSpinner.setSelection(0);
    }
    
    
}
