package com.biotechvana.netools.ui.dialogs;

import org.eclipse.jface.dialogs.TitleAreaDialog;
import org.eclipse.nebula.widgets.nattable.NatTable;
import org.eclipse.nebula.widgets.nattable.data.IColumnAccessor;
import org.eclipse.nebula.widgets.nattable.data.IDataProvider;
import org.eclipse.nebula.widgets.nattable.data.ListDataProvider;
import org.eclipse.nebula.widgets.nattable.grid.data.DefaultColumnHeaderDataProvider;
import org.eclipse.nebula.widgets.nattable.grid.data.DefaultCornerDataProvider;
import org.eclipse.nebula.widgets.nattable.grid.data.DefaultRowHeaderDataProvider;
import org.eclipse.nebula.widgets.nattable.grid.layer.ColumnHeaderLayer;
import org.eclipse.nebula.widgets.nattable.grid.layer.CornerLayer;
import org.eclipse.nebula.widgets.nattable.grid.layer.DefaultGridLayer;
import org.eclipse.nebula.widgets.nattable.grid.layer.GridLayer;
import org.eclipse.nebula.widgets.nattable.grid.layer.RowHeaderLayer;
import org.eclipse.nebula.widgets.nattable.hideshow.ColumnHideShowLayer;
import org.eclipse.nebula.widgets.nattable.layer.AbstractLayerTransform;
import org.eclipse.nebula.widgets.nattable.layer.CompositeLayer;
import org.eclipse.nebula.widgets.nattable.layer.DataLayer;
import org.eclipse.nebula.widgets.nattable.layer.IUniqueIndexLayer;
import org.eclipse.nebula.widgets.nattable.layer.cell.ColumnLabelAccumulator;
import org.eclipse.nebula.widgets.nattable.layer.stack.DefaultBodyLayerStack;
import org.eclipse.nebula.widgets.nattable.reorder.ColumnReorderLayer;
import org.eclipse.nebula.widgets.nattable.selection.SelectionLayer;
import org.eclipse.nebula.widgets.nattable.viewport.ViewportLayer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.TabFolder;
import org.eclipse.swt.widgets.TabItem;
import org.eclipse.swt.widgets.Text;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.biotechvana.netools.projects.DatasetType;
import com.biotechvana.netools.projects.Project;

import org.apache.commons.csv.CSVFormat;
import org.apache.commons.csv.CSVParser;
import org.apache.commons.csv.CSVRecord;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.Reader;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

public class ImportListDialog extends TitleAreaDialog {
	private static final Logger logger = LoggerFactory.getLogger(ImportListDialog.class);
    private Text csvFilePathText;
    private Combo csvSeparatorCombo;
    private File csvFile;
    private char csvSeparator = ',';
    private Composite previewContainer;
    //private Combo rowNameCombo;
    //private String selectedSampleNameClm;

    Project activeProject;

    public ImportListDialog(Shell parentShell) {
        super(parentShell);
    }

    public ImportListDialog(Shell parentShell, File csvFile) {
        super(parentShell);
        this.csvFile = csvFile;
    }

    public ImportListDialog(Shell parentShell, Project activeProject) {
        super(parentShell);
        this.activeProject = activeProject;
    }

    @Override
    public void create() {
        super.create();
        setTitle("List configuration");
        setMessage("Enter the details for a correct data import");
    }

    @Override
    protected Control createDialogArea(Composite parent) {
        Composite area = (Composite) super.createDialogArea(parent);
        Composite container = new Composite(area, SWT.NONE);
        container.setLayoutData(new GridData(GridData.FILL_BOTH));
        GridLayout layout = new GridLayout(3, false);
        container.setLayout(layout);

        createCsvFileField(container);
        createCsvSeparatorField(container);
        //createRowNameField(container);
        createCsvPreview(container);
        if (csvFile != null) {
            csvFilePathText.setText(csvFile.getAbsolutePath());
            loadCsvPreview(csvFile);
        }

        return area;
    }

    TabFolder previewTabFolder;
    TabItem previewTabItem;

    private void createCsvPreview(Composite container) {
        // create a tab folder
        previewTabFolder = new TabFolder(container, SWT.NONE);
        previewTabFolder.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true, 3, 1));

        // create a tab item
        previewTabItem = new TabItem(previewTabFolder, SWT.NONE);
        previewTabItem.setText("Data Preview");

    }

    private void createCsvFileField(Composite container) {
        Label lblCsvFile = new Label(container, SWT.NONE);
        lblCsvFile.setText("CSV File:");

        GridData dataCsvFilePath = new GridData();
        dataCsvFilePath.grabExcessHorizontalSpace = true;
        dataCsvFilePath.horizontalAlignment = GridData.FILL;

        csvFilePathText = new Text(container, SWT.BORDER);
        csvFilePathText.setLayoutData(dataCsvFilePath);
        csvFilePathText.setEditable(false);

        Button browseButton = new Button(container, SWT.PUSH);
        browseButton.setText("Browse...");
        browseButton.addListener(SWT.Selection, event -> {
            FileDialog fileDialog = new FileDialog(getShell(), SWT.OPEN);
            fileDialog.setFilterExtensions(new String[] { "*.csv" });
            String selectedFile = fileDialog.open();
            if (selectedFile != null) {
                csvFilePathText.setText(selectedFile);
                csvFile = new File(selectedFile);
                loadCsvPreview(csvFile);
            }
        });
    }

    private void createCsvSeparatorField(Composite container) {
        Label lblCsvSeparator = new Label(container, SWT.NONE);
        lblCsvSeparator.setText("CSV Separator:");

        GridData dataCsvSeparator = new GridData();
        dataCsvSeparator.grabExcessHorizontalSpace = true;
        dataCsvSeparator.horizontalAlignment = GridData.FILL;
        dataCsvSeparator.horizontalSpan = 2;

        csvSeparatorCombo = new Combo(container, SWT.DROP_DOWN | SWT.READ_ONLY);
        csvSeparatorCombo.setLayoutData(dataCsvSeparator);

        // Populate the combo with common CSV separators
        csvSeparatorCombo.add("Comma (,)");
        csvSeparatorCombo.add("Semicolon (;)");
        csvSeparatorCombo.add("Tab (\\t)");

        csvSeparatorCombo.select(0); // Default to comma

        csvSeparatorCombo.addListener(SWT.Selection, event -> {
            int index = csvSeparatorCombo.getSelectionIndex();
            switch (index) {
                case 0:
                    csvSeparator = ',';
                    break;
                case 1:
                    csvSeparator = ';';
                    break;
                case 2:
                    csvSeparator = '\t';
                    break;
            }
            if (csvFile != null) {
                loadCsvPreview(csvFile);
            }
        });
    }

    /*private void createRowNameField(Composite container) {
        Label lblRowName = new Label(container, SWT.NONE);
        lblRowName.setText("Sample ID Column:");

        GridData dataRowName = new GridData();
        dataRowName.grabExcessHorizontalSpace = true;
        dataRowName.horizontalAlignment = GridData.FILL;
        dataRowName.horizontalSpan = 2;

        rowNameCombo = new Combo(container, SWT.DROP_DOWN | SWT.READ_ONLY);
        rowNameCombo.setLayoutData(dataRowName);
    }*/

    private BodyLayerStack bodyLayer;
    private IDataProvider bodyDataProvider;
    private String[] propertyNames;
    private Map<String, String> propertyToLabels;

    NatTable natTable;
    private void loadCsvPreview(File csvFile) {
    	// getShell().setCursor(getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
    	if (previewContainer != null) {
            previewContainer.dispose();
        }

        previewContainer = new Composite(previewTabFolder, SWT.NONE);
        previewContainer.setLayoutData(new GridData(GridData.FILL_BOTH));
        previewContainer.setLayout(new FillLayout());
        
        try {
            Reader reader = new FileReader(csvFile);
            CSVParser csvParser = new CSVParser(reader,
                    CSVFormat.DEFAULT.withDelimiter(csvSeparator).withFirstRecordAsHeader());
            List<String[]> data = new ArrayList<>();
            List<String> headers = csvParser.getHeaderNames();
            logger.info("Headers: " + headers);
            for (CSVRecord csvRecord : csvParser) {
                String[] row = new String[headers.size()];
                for (int i = 0; i < headers.size(); i++) {
                    row[i] = csvRecord.get(i);
                }
                data.add(row);
            }
            logger.debug("data loaded : " + data.size() + " rows");

            propertyNames = headers.toArray(new String[0]);
            propertyToLabels = headers.stream().collect(Collectors.toMap(h -> h, h -> h));

            bodyDataProvider = new ListDataProvider<String[]>(data, new IColumnAccessor<String[]>() {
                @Override
                public Object getDataValue(String[] rowObject, int columnIndex) {
                    return rowObject[columnIndex];
                }

                @Override
                public void setDataValue(String[] rowObject, int columnIndex, Object newValue) {
                    rowObject[columnIndex] = newValue.toString();
                }

                @Override
                public int getColumnCount() {
                    return headers.size();
                }
            });

            DefaultColumnHeaderDataProvider colHeaderDataProvider = new DefaultColumnHeaderDataProvider(
                    this.propertyNames, this.propertyToLabels);
            DefaultRowHeaderDataProvider rowHeaderDataProvider = new DefaultRowHeaderDataProvider(
                    this.bodyDataProvider);

            this.bodyLayer = new BodyLayerStack(this.bodyDataProvider);
            ColumnHeaderLayerStack columnHeaderLayer = new ColumnHeaderLayerStack(
                    colHeaderDataProvider);
            RowHeaderLayerStack rowHeaderLayer = new RowHeaderLayerStack(
                    rowHeaderDataProvider);
            DefaultCornerDataProvider cornerDataProvider = new DefaultCornerDataProvider(
                    colHeaderDataProvider, rowHeaderDataProvider);
            CornerLayer cornerLayer = new CornerLayer(new DataLayer(
                    cornerDataProvider), rowHeaderLayer, columnHeaderLayer);

            GridLayer gridLayer = new GridLayer(this.bodyLayer, columnHeaderLayer,
                    rowHeaderLayer, cornerLayer);
            natTable = new NatTable(previewContainer, gridLayer);
            natTable.setLayoutData(new GridData(GridData.FILL_BOTH));

            // Populate the rowNameCombo with headers
            //rowNameCombo.setItems(headers.toArray(new String[0]));
            //rowNameCombo.select(0); // Default to the first header

			/*rowNameCombo.addListener(SWT.Selection, event -> {
				selectedSampleNameClm = rowNameCombo.getText();
				validate();
			});*/
            
            previewTabItem.setControl(previewContainer);
            previewContainer.layout();
            previewContainer.getParent().layout();

            // if length of headers is 1 then probably file is parsed incorrectly
            if (headers.size() == 1) {
                setErrorMessage(
                        "Input file contains only one column, Please check the file or change the csv separator");
                setValid(false);
            } else
                this.setValid(true);
        } catch (IllegalArgumentException e) {
            e.printStackTrace();
            this.setValid(false);
            // check msg if it A header name is missing in
            if (e.getMessage().contains("header name is missing")) {
                setErrorMessage("A header name is missing in the CSV file");
            } else if (e.getMessage().contains("duplicate name")) {
                setErrorMessage("Duplicate header name in the CSV file");
            } else {
                setErrorMessage(e.getMessage());
            }

        } catch (IOException e) {
            e.printStackTrace();
            this.setValid(false);
            setErrorMessage(e.getMessage());
        } catch (Exception e) {
            e.printStackTrace();
            this.setValid(false);
            setErrorMessage("Error loading CSV file, Please check the file or change the csv separator. \n" + e.getMessage());
        }
    }

    private void setValid(boolean b) {
        boolean isValid = b && validate();
        if (b) {
            // enable the ok button
            this.getButton(OK).setEnabled(true);
        } else {
            // disable the ok button
            this.getButton(OK).setEnabled(false);
        }

    }

    /**
     * Validate the dataset details
     * Dataset name cannot be empty
     * Dataset name should be unique in the project
     * CSV file should be selected
     * Other validations can be added 
     * right now, csv parsing is being done to check if the file is valid in loadCsvPreview method
     * @return
     */
    private boolean validate() {
        // Change the cursor to busy
        // getShell().setCursor(getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
        // sleep for 15 seconds
        
        try {

            if (csvFile == null) {
                // add an error message
                // setErrorMessage("CSV file not selected");
                return false;
            }

            // check if the selected column name is empty
            /*if (rowNameCombo.getText().isEmpty()) {
                setErrorMessage("Row name column cannot be empty");
                return false;
            }*/
            // check if the selected column name contains valid samples names or id and unique
            /*if (!checkSamplesNames()) {
                return false;
            }*/

            setErrorMessage(null);
            return true;
        } finally {
            // Return the cursor back to normal
            getShell().setCursor(null);
        }
    }

    /*private boolean checkSamplesNames() {
		String selectedRowName = rowNameCombo.getText();
		// get the index of the selected column
		int index = -1;
		for (int i = 0; i < propertyNames.length; i++) {
			if (propertyNames[i].equals(selectedRowName)) {
				index = i;
				break ;
			}
		}
		// check if the index is valid
		if (index == -1) {
			setErrorMessage("Invalid column name selected");
			return false;
		}
		// get the values of the selected column
		List<String> samples = new ArrayList<>();
		for (String[] row : ((ListDataProvider<String[]>) bodyDataProvider).getList()) {
			samples.add(row[index]);
		}
		// check if the samples are unique
		if (samples.size() != samples.stream().distinct().count()) {
			setErrorMessage("Samples names are not unique");
			return false;
		}
		// check if the samples are valid
		for (String sample : samples) {
			// sample name can contain only letters, digits and underscore or dot
//			if (!sample.matches("[a-zA-Z0-9_.]+")) {
//				setErrorMessage("Invalid sample name: " + sample);
//				return false;
//			}
		}
		// 
		
		return true;
		
	}*/

	public class BodyLayerStack extends AbstractLayerTransform {

        private SelectionLayer selectionLayer;

        public BodyLayerStack(IDataProvider dataProvider) {
            DataLayer bodyDataLayer = new DataLayer(dataProvider);
            ColumnReorderLayer columnReorderLayer = new ColumnReorderLayer(
                    bodyDataLayer);
            ColumnHideShowLayer columnHideShowLayer = new ColumnHideShowLayer(
                    columnReorderLayer);
            this.selectionLayer = new SelectionLayer(columnHideShowLayer);
            ViewportLayer viewportLayer = new ViewportLayer(this.selectionLayer);
            setUnderlyingLayer(viewportLayer);
        }

        public SelectionLayer getSelectionLayer() {
            return this.selectionLayer;
        }
    }

    public class ColumnHeaderLayerStack extends AbstractLayerTransform {

        public ColumnHeaderLayerStack(IDataProvider dataProvider) {
            DataLayer dataLayer = new DataLayer(dataProvider);
            ColumnHeaderLayer colHeaderLayer = new ColumnHeaderLayer(dataLayer,
                    ImportListDialog.this.bodyLayer, ImportListDialog.this.bodyLayer.getSelectionLayer());
            setUnderlyingLayer(colHeaderLayer);
        }
    }

    public class RowHeaderLayerStack extends AbstractLayerTransform {

        public RowHeaderLayerStack(IDataProvider dataProvider) {
            DataLayer dataLayer = new DataLayer(dataProvider, 50, 20);
            RowHeaderLayer rowHeaderLayer = new RowHeaderLayer(dataLayer,
                    ImportListDialog.this.bodyLayer, ImportListDialog.this.bodyLayer.getSelectionLayer());
            setUnderlyingLayer(rowHeaderLayer);
        }
    }

    @Override
    protected boolean isResizable() {
        return true;
    }

    @Override
    protected Point getInitialSize() {
        // TODO Auto-generated method stub
        return new Point(800, 600);
    }

    /*@Override
    protected void okPressed() {
        if (validate()) {
            selectedSampleNameClm = rowNameCombo.getText();
            super.okPressed();
        }

    }*/

    public File getCsvFile() {
        return csvFile;
    }

    /*public String getSelectedSampleNameClm() {
        return selectedSampleNameClm;
    }*/

    public String getCsvSeparator() {
        // switch over csvSeparator and return one of the following comma , semicolon,
        // tab or space
        switch (csvSeparator) {
            case ',':
                //return "comma";
            	return ",";
            case ';':
                //return "semicolon";
            	return ";";
            case '\t':
                //return "tab";
            	return "\t";
            default:
                //return "space";
            	return " ";
        }
    }
}
