package com.biotechvana.netools.projects;

import java.io.Externalizable;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInput;
import java.io.ObjectInputStream;
import java.io.ObjectOutput;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.eclipse.core.runtime.IProgressMonitor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.biotechvana.netools.models.DirectedWeightedNetwork;
import com.biotechvana.netools.models.NetworkModel;
import com.biotechvana.workflow.tracking.ExecStatus;
import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIdentityInfo;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonManagedReference;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.ObjectIdGenerators;

@JsonIdentityInfo(generator = ObjectIdGenerators.PropertyGenerator.class, property = "projectID")
@JsonAutoDetect(fieldVisibility = JsonAutoDetect.Visibility.ANY, getterVisibility = JsonAutoDetect.Visibility.NONE)

/**
 * Represents a project with a unique ID, name, description, and associated datasets.
 */
public class Project implements Serializable {
	
	private static final Logger logger = LoggerFactory.getLogger(Project.class);
	
    private static final long serialVersionUID = 1L;
	private String projectName;
	private String projectDescription;
	
	/**
	 * Project ID is a unique identifier for the project, it is the project name sanitized
	 */
	private String projectID;
	

	/**
	 * The transient projects manager is used to manage the project
	 */
	transient private IProjectsManager projectsManager;
	
	/**
	 * The datasets associated with the project
	 */
	@JsonManagedReference // ("project-dataset")
	private List<Dataset> datasets = new ArrayList<>();

	/**
	 * The network designs associated with the project
	 */
	@JsonManagedReference//("project-design")
    List<NetworkDesign> networkDesigns = new ArrayList<>();
	
	/**
	 * The network builds associated with the project
	 */
	@JsonManagedReference// ("project-network")
	List<NetworkBuild> networkBuilds = new ArrayList<>();

	@JsonManagedReference
	private List<Network> networks = new ArrayList<>();
    
    // Pipeline to create a new project
	/**
     * Constructs a new Project with the specified name and description.
     * @param projectName the name of the project
     * @param projectDescription the description of the project
     */
	@JsonCreator
	public Project(@JsonProperty("projectName") String projectName,@JsonProperty("projectDescription") String projectDescription) {
		// Sanitize the project name to remove spaces and special characters to create a unique project ID
		// first replace white spaces with underscore
		this.projectID = projectName.replaceAll("\\s+", "_");
		this.projectID = this.projectID.replaceAll("[^a-zA-Z0-9_]", "");	
		this.projectID = projectName.replaceAll("[^a-zA-Z0-9]", "");
		this.projectName = projectName;
		this.projectDescription = projectDescription;
	}
	

	
	/**
     * Adds a dataset to the project.
     * @param dataset the dataset to add
     * @return true if the dataset was added, false if it already exists
     * @throws IllegalArgumentException if a dataset with the same ID already exists
     */
	public boolean addDataset(Dataset dataset) {
		// check if the dataset is already in the list
		if (datasets.contains(dataset)) {
			return false;
		}
		// check if a dataset with the same name already exists
		for (Dataset d : datasets) {
            if (d.getDatasetID().equals(dataset.getDatasetID())) {
                throw new IllegalArgumentException("Dataset with the same name already exists");
            }
        }
		datasets.add(dataset);
		dataset.setProject(this);
		projectsManager.datasetAddedToProject(dataset);
		return true;
	}
	// Getters and setters
    /**
     * Gets the project name.
     * @return the project name
     */
    public String getProjectName() {
        return projectName;
    }

    /**
     * Sets the project name.
     * @param projectName the new project name
     */
    public void setProjectName(String projectName) {
        this.projectName = projectName;
        if (projectsManager != null)
        	projectsManager.projectNameChanged(this);
    }

    /**
     * Gets the project description.
     * @return the project description
     */
    public String getProjectDescription() {
        return projectDescription;
    }

    /**
     * Sets the project description.
     * @param projectDescription the new project description
     */
    public void setProjectDescription(String projectDescription) {
        this.projectDescription = projectDescription;
        if (projectsManager != null)
        	projectsManager.projectDescriptionChanged(this);
    }

    /**
     * Gets the project ID.
     * @return the project ID
     */
    public String getProjectID() {
        return projectID;
    }

//    public void setProjectID(String projectID) {
//        this.projectID = projectID;
//        
//    }

    
 // Custom deserialization to initialize transient fields
    /**
     * Custom deserialization to initialize transient fields.
     * @param ois the ObjectInputStream to read from
     * @throws IOException if an I/O error occurs
     * @throws ClassNotFoundException if the class of a serialized object could not be found
     */
    private void readObject(ObjectInputStream ois) throws IOException, ClassNotFoundException {
    	ois.defaultReadObject();
        // projectsManager = ProjectsManagerImpl.getInstance
    	datasets = new ArrayList<>();
    }
    
    /**
     * Writes the project to an external output.
     * @param out the ObjectOutput to write to
     * @throws IOException if an I/O error occurs
     */
    public void writeExternal(ObjectOutput out) throws IOException {
        out.writeObject(projectName);
        out.writeObject(projectID);
        out.writeObject(projectDescription);
    }

    /**
     * Reads the project from an external input.
     * @param in the ObjectInput to read from
     * @throws IOException if an I/O error occurs
     * @throws ClassNotFoundException if the class of a serialized object could not be found
     */
    public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException {
        projectName = (String) in.readObject();
        projectID = (String) in.readObject();
        projectDescription = (String) in.readObject();
    }

    /**
     * Serializes the project to a file.
     * @param project the project to serialize
     * @param filePath the file path to save the serialized project
     */
     public static void serializeProject(Project project, String filePath) {
        try (FileOutputStream fileOut = new FileOutputStream(filePath);
             ObjectOutputStream out = new ObjectOutputStream(fileOut)) {
            out.writeObject(project);
            System.out.println("Serialized data is saved in " + filePath);
        } catch (IOException i) {
            i.printStackTrace();
        }
    }

    /**
     * Serializes the project to an output stream.
     * @param project the project to serialize
     * @param fileOut the output stream to write the serialized project
     */
    public static void serializeProject(Project project, OutputStream fileOut) {
        try (ObjectOutputStream out = new ObjectOutputStream(fileOut)) {
            out.writeObject(project);
        } catch (IOException i) {
            i.printStackTrace();
        }
    }
    
    /**
     * Deserializes a project from a file.
     * @param filePath the file path to read the serialized project
     * @return the deserialized project
     */
	public static Project deserializeProject(String filePath) {
		Project project = null;
		try (FileInputStream fileIn = new FileInputStream(filePath);
				ObjectInputStream in = new ObjectInputStream(fileIn)) {
			project = (Project) in.readObject();
		} catch (IOException | ClassNotFoundException i) {
			i.printStackTrace();
		}
		return project;
	}

    /**
     * Deserializes a project from an input stream.
     * @param fileIn the input stream to read the serialized project
     * @return the deserialized project
     * @throws RuntimeException if an error occurs during deserialization
     */
	public static Project deserializeProject(InputStream fileIn) {
		Project project = null;
		try (ObjectInputStream in = new ObjectInputStream(fileIn)) {
			project = (Project) in.readObject();
		} catch (IOException | ClassNotFoundException i) {
			i.printStackTrace();
			throw new RuntimeException("Error deserializing project");
		}
		return project;
	}

    /**
     * Gets the list of datasets associated with the project.
     * @return the list of datasets
     */
	public List<Dataset> getDatasets() {
		// TODO Auto-generated method stub
		return datasets;
	}

    /**
     * Sets the projects manager.
     * @param projectsManager the projects manager to set
     */
	public void setProjectsManager(IProjectsManager projectsManager) {
		
		this.projectsManager = projectsManager;
	}

    /**
     * Gets the projects manager.
     * @return the projects manager
     */
	public IProjectsManager getProjectManager() {
		return projectsManager;
	}


	/**
	 * Get the directory where the project is
	 * @return .NETOOLS/projects/<projectID>
	 */
    /**
     * Gets the base directory of the project.
     * @return the base directory of the project
     */
	@JsonIgnore
	public String getBaseDirectory() {
		// TODO Auto-generated method stub
		return projectsManager.getBaseFolder() + "/" + projectID;
	}
	
	/**
	 * Get the directory where the datasets are stored
	 * @return .NETOOLS/projects/<projectID>/datasets
	 */
    /**
     * Gets the directory where the datasets are stored.
     * @return the datasets directory
     */
	@JsonIgnore
	public String getDatasetDirectory() {
		// TODO Auto-generated method stub
		return projectsManager.getBaseFolder() + "/" + projectID + "/" + RemoteStorage.DATASETS_DIRECTORY;
	}

	@JsonIgnore
	public String getDesignDirectory() {
		// TODO Auto-generated method stub
		return projectsManager.getBaseFolder() + "/" + projectID + "/" + RemoteStorage.DESIGNS_DIRECTORY;
	}
	

	/**
	 * Get the directory where the tasks are stored
	 * @return .NETOOLS/projects/<projectID>/tasks
	 */
    /**
     * Gets the directory where the tasks are stored.
     * @return the tasks directory
     */
	@JsonIgnore
	public String getTasksDirectory() {
		return getBaseDirectory()  + "/" + RemoteStorage.TASKS_DIRECTORY ;
	}
	public String getNetworksDirectory() {
		
		return  getBaseDirectory() + "/" + RemoteStorage.NETWORKS_DIRECTORY;
	}
	
	public String getNetworkDirectory(String networkBuildID) {
		
		return  getBaseDirectory() + "/" + RemoteStorage.NETWORKS_DIRECTORY + "/" + networkBuildID;
	}

    /**
     * Gets the processing tasks associated with the project.
     * @return an array of processing tasks
     */
	public DTFile[] getProcessingTasks() {
		// collect all the processing tasks from all datasets
		List<DTFile> tasks = new ArrayList<>();
		for (Dataset dataset : datasets) {
			for (DTFile dtFile : dataset.getDTFiles()) {
				if (dtFile.getExecStatus() == ExecStatus.Unknown || dtFile.getExecStatus() == ExecStatus.Scheduled
						|| dtFile.getExecStatus() == ExecStatus.Running) {
					tasks.add(dtFile);
				}
			}
		}
		return tasks.toArray(new DTFile[tasks.size()]);
	}

    /**
     * Removes a dataset from the project.
     * @param selectedDataset the dataset to remove
     * @param monitor the progress monitor
     */
	public void removeDataset(Dataset selectedDataset, IProgressMonitor monitor) {
		// TODO Auto-generated method stub
		logger.debug("Removing dataset " + selectedDataset.getDatasetName());
        datasets.remove(selectedDataset);
        logger.debug("Dataset removed from project. Now deleting dataset from remote storage");
        projectsManager.deleteDataset(selectedDataset, monitor);
        projectsManager.datasetRemovedFromProject(selectedDataset);
    
		
	}



	public List<NetworkBuild> getNetworkBuilds() {
		// TODO Auto-generated method stub
		return networkBuilds;
	}



	public void addNetworkBuild(NetworkBuild networkBuild) {
		if(networkBuilds.contains(networkBuild)) {
			return;
		}
		networkBuilds.add(networkBuild);
		networkBuild.setProject(this);
		projectsManager.networkBuildAddedToProject(networkBuild);
		
	}



	public void removeNetworkBuild(NetworkBuild build) {
		// TODO Auto-generated method stub
		networkBuilds.remove(build);
		projectsManager.networkBuildDeleted(build);
		
	}



	public String getTasksDirectory(String taskID) {
		// TODO Auto-generated method stub
		return getTasksDirectory() + "/" + taskID;
	}



	public Variable getVariable(String id) {
		for (Dataset dataset : datasets) {
			Variable variable = dataset.getVariables().stream().filter(v -> v.getId().equals(id)).findFirst().orElse(null);
			if (variable != null) {
				return variable;
			}
		}
		return null;
	}
	
	public Variable getVariableByName(String id) {
		for (Dataset dataset : datasets) {
			Variable variable = dataset.getVariables().stream().filter(v -> v.getName().equals(id)).findFirst().orElse(null);
			if (variable != null) {
				return variable;
			}
		}
		return null;
	}
	
	public List<Variable> getVariables() {
		List<Variable> variables = new ArrayList<>();
		for (Dataset dataset : datasets) {
			variables.addAll(dataset.getVariables());
		}
		return variables;
	}



	public void addNetwork( Network network) {
		networks.add(network);
		network.setProject(this);	
		projectsManager.networkAddedToProject(network);
	}
	
	public List<Network> getNetworks() {
		return networks;
	}


	// check if a network with the same name already exists
	public boolean hasNetwork(String networkName) {
		// check networks in the project using stream 
		return networks.stream().anyMatch(n -> n.getNetworkName().equals(networkName));
		
	}



	public Network getNetwork(String networkName) {
		// get network by name
		return networks.stream().filter(n -> n.getNetworkName().equals(networkName)).findFirst().orElse(null);
	}

	
	/**
	 * should be called once before creat
	 */
	static public String getUniqueID(String baseName, Set<String> existingIDs) {
		String newID = baseName;
		int i = 1;
		while (existingIDs.contains(newID)) {
			newID = baseName + "_" + i;
			i++;
		}
		return newID;

	}
	public String getUniqueBuildID (String buildID) {
		
		return getUniqueID(buildID, networkBuilds.stream().map(NetworkBuild::getBuildID).collect(Collectors.toSet()));
    	
	}

	public String getUniqueNetworkID(String networkName) {
		String networkID = networkName.replaceAll("\\s+", "_");
		return getUniqueID(networkID, networks.stream().map(Network::getNetworkID).collect(Collectors.toSet()));
	}



	public void removeNetwork(Network selectedNetwork, IProgressMonitor monitor) {
		this.networks.remove(selectedNetwork);
		projectsManager.deleteNetwork(selectedNetwork,monitor);
		projectsManager.saveProject(this,false,monitor);
		projectsManager.networkDeleted(selectedNetwork);
		// 
		
	}


	// Network Design
	public void addNetworkDesign(NetworkDesign design) {
		if (networkDesigns.contains(design)) {
			return;
		}
		// not with the same name
		for (NetworkDesign d : networkDesigns) {
			if (d.getDesignName().equals(design.getDesignName())) {
				throw new IllegalArgumentException("Network Design with the same name already exists");
			}
		}
		networkDesigns.add(design);
		design.setProject(this);
		projectsManager.networkDesignAddedToProject(design);
	}
	
	public List<NetworkDesign> getNetworkDesigns() {
		return networkDesigns;
	}


	
}
