package com.biotechvana.netools.projects;

import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.JsonBackReference;
import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIdentityInfo;
import com.fasterxml.jackson.annotation.JsonManagedReference;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.ObjectIdGenerators;

@JsonIdentityInfo(generator = ObjectIdGenerators.PropertyGenerator.class, property = "designID" , scope = NetworkDesign.class)
@JsonAutoDetect(fieldVisibility = JsonAutoDetect.Visibility.ANY, getterVisibility = JsonAutoDetect.Visibility.NONE)
public class NetworkDesign {

	@JsonManagedReference
	private Set<Arc> whitelist = new HashSet<>();
	@JsonManagedReference
	private Set<Arc> blacklist = new HashSet<>();


	private HashMap<String,List<String>> dataTypesWhitelist = new HashMap<>();
	private HashMap<String,List<String>> dataTypesBlacklist = new HashMap<>();

	
	String designID;
	
	String name;
	String designDescription;
	
	@JsonBackReference  //("project-design")
	Project project;
	
	
	@JsonCreator
	public NetworkDesign(@JsonProperty("name") String designName,@JsonProperty("designDescription")  String designDescription) {
		this.name = designName;
		this.designID = designName;
		this.designDescription = designDescription;
	}
	/**
	 * add to whitelist of the project
	 */
	public void addWhitelist(Variable source, Variable target) {
		Arc arc = new Arc(source.getId(), target.getId());
		whitelist.add(arc);
		arc.setNetworkDesign(this);
	}
	public void addWhitelist(Variable source, Variable target, String reason) {
		Arc arc = new Arc(source.getId(), target.getId());
		arc.setReason(reason);
		whitelist.add(arc);
		arc.setNetworkDesign(this);
	}

	/**
	 * remove from whitelist of the project
	 */
	public void removeWhitelist(Variable source, Variable target) {
		Arc arc = new Arc(source.getId(), target.getId());
		whitelist.remove(arc);
		arc.setNetworkDesign(this);
	}

	public Set<Arc> getWhitelist() {
		return whitelist;
	}

	public Arc getWhitelistArc(Variable source, Variable target) {
		for (Arc arc : whitelist) {
			if (arc.getSource().equals(source) && arc.getTarget().equals(target)) {
				return arc;
			}
		}
		return null;
	}

	/**
	 * add to blacklist of the project
	 */
	public void addBlacklist(Variable source, Variable target) {
		Arc arc = new Arc(source.getId(), target.getId());
		blacklist.add(arc);
		arc.setNetworkDesign(this);
	}

	public void addBlacklist(Variable source, Variable target, String reason) {
		Arc arc = new Arc(source.getId(), target.getId());
		arc.setReason(reason);
		blacklist.add(arc);
		arc.setNetworkDesign(this);
	}

	/**
	 * remove from blacklist of the project
	 */
	public void removeBlacklist(Variable source, Variable target) {
		Arc arc = new Arc(source.getId(), target.getId());
		blacklist.remove(arc);
		arc.setNetworkDesign(this);
	}

	public Set<Arc> getBlacklist() {
		return blacklist;
	}

	public Arc getBlacklistArc(Variable source, Variable target) {
		for (Arc arc : blacklist) {
			if (arc.getSource().equals(source) && arc.getTarget().equals(target)) {
				return arc;
			}
		}
		return null;
	}


	public void addWhitelistDataType(String source, String target) {
		if (!dataTypesWhitelist.containsKey(source)) {
			dataTypesWhitelist.put(source, new ArrayList<>());
		}
		dataTypesWhitelist.get(source).add(target);
	}

	public void addBlacklistDataType(String source, String target) {
		if (!dataTypesBlacklist.containsKey(source)) {
			dataTypesBlacklist.put(source, new ArrayList<>());
		}
		dataTypesBlacklist.get(source).add(target);
	}

	public void removeWhitelistDataType(String source, String target) {
		if (dataTypesWhitelist.containsKey(source)) {
			dataTypesWhitelist.get(source).remove(target);
		}
	}

	public void removeBlacklistDataType(String source, String target) {
		if (dataTypesBlacklist.containsKey(source)) {
			dataTypesBlacklist.get(source).remove(target);
		}
	}

	public HashMap<String, List<String>> getDataTypesWhitelist() {
		return dataTypesWhitelist;
	}

	public HashMap<String, List<String>> getDataTypesBlacklist() {
		return dataTypesBlacklist;
	}
	public String getDesignID() {
		// TODO Auto-generated method stub
		return designID;
	}
	public void setProject(Project project) {
		this.project = project;
		
	}
	public String getDesignName() {
		return name;
	}
	public void setDesignName(String name) {
		this.name = name;
	}

	public String getDescription() {
		return designDescription;
	}

	public void setDescription(String description) {
		this.designDescription = description;
	}
	public Project getProject() {
		// TODO Auto-generated method stub
		return project;
	}
	public void writeWhitelist(OutputStream outputStream) {
		// write header
		String separator = ",";
		String header = "from" + separator + "to" + "\n";
		try {
			outputStream.write(header.getBytes());
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		for (Arc arc : whitelist) {
			String line = arc.getSource().getId() + separator  + arc.getTarget().getId() + "\n";
			try {
				outputStream.write(line.getBytes());
			} catch (Exception e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}
		
	}
	
	public void writeBlacklist(OutputStream outputStream) {
		// write header
		String separator = ",";
		String header = "from" + separator + "to" + "\n";
		try {
			outputStream.write(header.getBytes());
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}

		for (Arc arc : blacklist) {
			String line = arc.getSource().getId() + separator + arc.getTarget().getId() + "\n";
			try {
				outputStream.write(line.getBytes());
			} catch (Exception e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}

	}
	public String getWhitelistFileName() {
		return this.project.getDesignDirectory()+ "/" + this.designID + "/whitelist.csv";
	}

	public String getBlacklistFileName() {
		return this.project.getDesignDirectory() + "/" + this.designID + "/blacklist.csv";
	}
	
	
}
