package com.biotechvana.netools.models;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.HashMap;
import java.util.Map;

import org.json.JSONObject;

import com.biotechvana.netools.projects.Variable;
import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.JsonInclude;

/**
 * The Node class represents a node in a network with various attributes such as id, label, size, color, etc.
 * It supports property change listeners to notify changes in its properties.
 * It also provides methods to convert the node to a JSON object.
 */
@JsonAutoDetect(fieldVisibility = JsonAutoDetect.Visibility.ANY, getterVisibility = JsonAutoDetect.Visibility.NONE)
@JsonInclude(JsonInclude.Include.NON_NULL)
public class Node implements INode , IAttributable {

    // Nested static class with constants for default values
    public static class Defaults {
        public static final int BORDER_WIDTH = 1;
        public static final int BORDER_WIDTH_SELECTED = 2;
        public static final double OPACITY = 1.0;
        public static final String GROUP = "";
        public static final boolean HIDDEN = false;
        public static final String LABEL = "";
        public static final boolean LABEL_HIGHLIGHT_BOLD = true;
        public static final int LEVEL = 0;
        public static final double MASS = 1.0;
        public static final String SHAPE = "ellipse";
        public static final boolean PHYSICS = true;
        public static final int SIZE = 25;
        public static final String TITLE = "";
        public static final double VALUE = 1.0;
        public static final double X = Double.NaN;
        public static final double Y = Double.NaN;
        public static final String COLOR = "#97C2FC";
        public static final String COLOR_BORDER = "#2B7CE9";
        public static final String COLOR_BACKGROUND = "#D2E5FF";
        public static final String COLOR_HIGHLIGHT = "#D2E5FF";
        public static final String COLOR_HIGHLIGHT_BORDER = "#2B7CE9";
        public static final String COLOR_HIGHLIGHT_BACKGROUND = "#D2E5FF";
        public static final String COLOR_HOVER = "#D2E5FF";
        public static final String COLOR_HOVER_BORDER = "#2B7CE9";
        public static final String COLOR_HOVER_BACKGROUND = "#D2E5FF";
    }

    private Map<String, Object> viewerAttributes = new HashMap<>();
    private Map<String, Object> customAttributes = new HashMap<>();
    
    transient private Variable linkedVariable = null;
   
   transient private PropertyChangeSupport pcs = new PropertyChangeSupport(this);

    
    public Node() {
    	
    }
    
    // Constructor that takes only the id and sets the label to the id
    public Node(String id) {
        viewerAttributes.put("id", id);
        viewerAttributes.put("label", id);
    }

    // Constructor that takes both id and label
    public Node(String id, String label) {
        viewerAttributes.put("id", id);
        viewerAttributes.put("label", label);
    }

    
    @Override
    public int hashCode() {
    	return super.hashCode();
    	// return getId().hashCode();
    }
    
    // Getters and setters for all attributes with property change support

    @Override
	public String getId() {
        return (String) viewerAttributes.get("id");
    }

    @Override
	public void setId(String id) {
        String oldId = (String) viewerAttributes.get("id");
        if (id.equals(oldId)) return;
        viewerAttributes.put("id", id);
        pcs.firePropertyChange("id", oldId, id);
    }

    @Override
	public int getBorderWidth() {
        return (int) viewerAttributes.getOrDefault("borderWidth", Defaults.BORDER_WIDTH);
    }

    @Override
	public void setBorderWidth(int borderWidth) {
        int oldBorderWidth = (int) viewerAttributes.getOrDefault("borderWidth", Defaults.BORDER_WIDTH);
        if (borderWidth == oldBorderWidth) return;
        viewerAttributes.put("borderWidth", borderWidth);
        pcs.firePropertyChange("borderWidth", oldBorderWidth, borderWidth);
    }

    @Override
	public int getBorderWidthSelected() {
        return (int) viewerAttributes.getOrDefault("borderWidthSelected", Defaults.BORDER_WIDTH_SELECTED);
    }

    @Override
	public void setBorderWidthSelected(int borderWidthSelected) {
        int oldBorderWidthSelected = (int) viewerAttributes.getOrDefault("borderWidthSelected", Defaults.BORDER_WIDTH_SELECTED);
        if (borderWidthSelected == oldBorderWidthSelected) return;
        viewerAttributes.put("borderWidthSelected", borderWidthSelected);
        pcs.firePropertyChange("borderWidthSelected", oldBorderWidthSelected, borderWidthSelected);
    }

    @Override
	public double getOpacity() {
        return (double) viewerAttributes.getOrDefault("opacity", Defaults.OPACITY);
    }

    @Override
	public void setOpacity(double opacity) {
        double oldOpacity = (double) viewerAttributes.getOrDefault("opacity", Defaults.OPACITY);
        if (opacity == oldOpacity) return;
        viewerAttributes.put("opacity", opacity);
        pcs.firePropertyChange("opacity", oldOpacity, opacity);
    }

    @Override
	public String getGroup() {
        return (String) viewerAttributes.getOrDefault("group", Defaults.GROUP);
    }

    @Override
	public void setGroup(String group) {
        String oldGroup = (String) viewerAttributes.getOrDefault("group", Defaults.GROUP);
        if (group.equals(oldGroup)) return;
        viewerAttributes.put("group", group);
        pcs.firePropertyChange("group", oldGroup, group);
    }

    @Override
	public boolean isHidden() {
        return (boolean) viewerAttributes.getOrDefault("hidden", Defaults.HIDDEN);
    }

    @Override
	public void setHidden(boolean hidden) {
        boolean oldHidden = (boolean) viewerAttributes.getOrDefault("hidden", Defaults.HIDDEN);
        if (hidden == oldHidden) return;
        viewerAttributes.put("hidden", hidden);
        pcs.firePropertyChange("hidden", oldHidden, hidden);
    }

    @Override
	public String getLabel() {
        return (String) viewerAttributes.get("label");
    }

    @Override
	public void setLabel(String label) {
        String oldLabel = (String) viewerAttributes.get("label");
        if (label.equals(oldLabel)) return;
        viewerAttributes.put("label", label);
        pcs.firePropertyChange("label", oldLabel, label);
    }

    @Override
	public boolean isLabelHighlightBold() {
        return (boolean) viewerAttributes.getOrDefault("labelHighlightBold", Defaults.LABEL_HIGHLIGHT_BOLD);
    }

    @Override
	public void setLabelHighlightBold(boolean labelHighlightBold) {
        boolean oldLabelHighlightBold = (boolean) viewerAttributes.getOrDefault("labelHighlightBold", Defaults.LABEL_HIGHLIGHT_BOLD);
        if (labelHighlightBold == oldLabelHighlightBold) return;
        viewerAttributes.put("labelHighlightBold", labelHighlightBold);
        pcs.firePropertyChange("labelHighlightBold", oldLabelHighlightBold, labelHighlightBold);
    }

    @Override
	public int getLevel() {
        return (int) viewerAttributes.getOrDefault("level", Defaults.LEVEL);
    }

    @Override
	public void setLevel(int level) {
        int oldLevel = (int) viewerAttributes.getOrDefault("level", Defaults.LEVEL);
        if (level == oldLevel) return;
        viewerAttributes.put("level", level);
        pcs.firePropertyChange("level", oldLevel, level);
    }

    @Override
	public double getMass() {
        return (double) viewerAttributes.getOrDefault("mass", Defaults.MASS);
    }

    @Override
	public void setMass(double mass) {
        double oldMass = (double) viewerAttributes.getOrDefault("mass", Defaults.MASS);
        if (mass == oldMass) return;
        viewerAttributes.put("mass", mass);
        pcs.firePropertyChange("mass", oldMass, mass);
    }

    @Override
	public String getShape() {
        return (String) viewerAttributes.getOrDefault("shape", Defaults.SHAPE);
    }

    @Override
	public void setShape(String shape) {
        String oldShape = (String) viewerAttributes.getOrDefault("shape", Defaults.SHAPE);
        if (shape.equals(oldShape)) return;
        viewerAttributes.put("shape", shape);
        pcs.firePropertyChange("shape", oldShape, shape);
    }

    @Override
	public boolean isPhysics() {
        return (boolean) viewerAttributes.getOrDefault("physics", Defaults.PHYSICS);
    }

    @Override
	public void setPhysics(boolean physics) {
        boolean oldPhysics = (boolean) viewerAttributes.getOrDefault("physics", Defaults.PHYSICS);
        if (physics == oldPhysics) return;
        viewerAttributes.put("physics", physics);
        pcs.firePropertyChange("physics", oldPhysics, physics);
    }

    @Override
	public int getSize() {
        return (int) viewerAttributes.getOrDefault("size", Defaults.SIZE);
    }

    @Override
	public void setSize(int size) {
        int oldSize = (int) viewerAttributes.getOrDefault("size", Defaults.SIZE);
        if (size == oldSize) return;
        viewerAttributes.put("size", size);
        pcs.firePropertyChange("size", oldSize, size);
    }

    @Override
	public String getTitle() {
        return (String) viewerAttributes.getOrDefault("title", Defaults.TITLE);
    }

    @Override
	public void setTitle(String title) {
        String oldTitle = (String) viewerAttributes.getOrDefault("title", Defaults.TITLE);
        if (title.equals(oldTitle)) return;
        viewerAttributes.put("title", title);
        pcs.firePropertyChange("title", oldTitle, title);
    }

    @Override
	public double getValue() {
        return (double) viewerAttributes.getOrDefault("value", Defaults.VALUE);
    }

    @Override
	public void setValue(double value) {
        double oldValue = (double) viewerAttributes.getOrDefault("value", Defaults.VALUE);
        if (value == oldValue) return;
        viewerAttributes.put("value", value);
        pcs.firePropertyChange("value", oldValue, value);
    }

    @Override
	public double getX() {
        return (double) viewerAttributes.getOrDefault("x", Defaults.X);
    }

    @Override
	public void setX(double x) {
        double oldX = (double) viewerAttributes.getOrDefault("x", Defaults.X);
        if (x == oldX) return;
        viewerAttributes.put("x", x);
        pcs.firePropertyChange("x", oldX, x);
    }

    @Override
	public double getY() {
        return (double) viewerAttributes.getOrDefault("y", Defaults.Y);
    }

    @Override
	public void setY(double y) {
        double oldY = (double) viewerAttributes.getOrDefault("y", Defaults.Y);
        if (y == oldY) return;
        viewerAttributes.put("y", y);
        pcs.firePropertyChange("y", oldY, y);
    }

    @Override
	public String getColor() {
        return (String) viewerAttributes.getOrDefault("color", Defaults.COLOR);
    }

    @Override
	public void setColor(String color) {
        String oldColor = (String) viewerAttributes.getOrDefault("color", Defaults.COLOR);
        if (color.equals(oldColor)) return;
        viewerAttributes.put("color", color);
        pcs.firePropertyChange("color", oldColor, color);
    }

    @Override
	public String getColorBorder() {
        return (String) viewerAttributes.getOrDefault("colorBorder", Defaults.COLOR_BORDER);
    }

    @Override
	public void setColorBorder(String colorBorder) {
        String oldColorBorder = (String) viewerAttributes.getOrDefault("colorBorder", Defaults.COLOR_BORDER);
        if (colorBorder.equals(oldColorBorder)) return;
        viewerAttributes.put("colorBorder", colorBorder);
        pcs.firePropertyChange("colorBorder", oldColorBorder, colorBorder);
    }

    @Override
	public String getColorHighlight() {
        return (String) viewerAttributes.getOrDefault("colorHighlight", Defaults.COLOR_HIGHLIGHT);
    }

    @Override
	public void setColorHighlight(String colorHighlight) {
        String oldColorHighlight = (String) viewerAttributes.getOrDefault("colorHighlight", Defaults.COLOR_HIGHLIGHT);
        if (colorHighlight.equals(oldColorHighlight)) return;
        viewerAttributes.put("colorHighlight", colorHighlight);
        pcs.firePropertyChange("colorHighlight", oldColorHighlight, colorHighlight);
    }

    @Override
	public String getColorHighlightBorder() {
        return (String) viewerAttributes.getOrDefault("colorHighlightBorder", Defaults.COLOR_HIGHLIGHT_BORDER);
    }

    @Override
	public void setColorHighlightBorder(String colorHighlightBorder) {
        String oldColorHighlightBorder = (String) viewerAttributes.getOrDefault("colorHighlightBorder", Defaults.COLOR_HIGHLIGHT_BORDER);
        if (colorHighlightBorder.equals(oldColorHighlightBorder)) return;
        viewerAttributes.put("colorHighlightBorder", colorHighlightBorder);
        pcs.firePropertyChange("colorHighlightBorder", oldColorHighlightBorder, colorHighlightBorder);
    }

    @Override
	public String getColorHover() {
        return (String) viewerAttributes.getOrDefault("colorHover", Defaults.COLOR_HOVER);
    }

    @Override
	public void setColorHover(String colorHover) {
        String oldColorHover = (String) viewerAttributes.getOrDefault("colorHover", Defaults.COLOR_HOVER);
        if (colorHover.equals(oldColorHover)) return;
        viewerAttributes.put("colorHover", colorHover);
        pcs.firePropertyChange("colorHover", oldColorHover, colorHover);
    }

    @Override
	public String getColorHoverBorder() {
        return (String) viewerAttributes.getOrDefault("colorHoverBorder", Defaults.COLOR_HOVER_BORDER);
    }

    @Override
	public void setColorHoverBorder(String colorHoverBorder) {
        String oldColorHoverBorder = (String) viewerAttributes.getOrDefault("colorHoverBorder", Defaults.COLOR_HOVER_BORDER);
        if (colorHoverBorder.equals(oldColorHoverBorder)) return;
        viewerAttributes.put("colorHoverBorder", colorHoverBorder);
        pcs.firePropertyChange("colorHoverBorder", oldColorHoverBorder, colorHoverBorder);
    }

    // Method to convert Node object to JSONObject
    @Override
	public JSONObject toJson() {
        JSONObject jsonObject = new JSONObject();
        jsonObject.put("id", getId());
        jsonObject.put("label", getLabel());
        if (viewerAttributes.containsKey("borderWidth")) jsonObject.put("borderWidth", getBorderWidth());
        if (viewerAttributes.containsKey("borderWidthSelected")) jsonObject.put("borderWidthSelected", getBorderWidthSelected());
        if (viewerAttributes.containsKey("opacity")) jsonObject.put("opacity", getOpacity());
        if (viewerAttributes.containsKey("group")) jsonObject.put("group", getGroup());
        if (viewerAttributes.containsKey("hidden")) jsonObject.put("hidden", isHidden());
        if (viewerAttributes.containsKey("labelHighlightBold")) jsonObject.put("labelHighlightBold", isLabelHighlightBold());
        if (viewerAttributes.containsKey("level")) jsonObject.put("level", getLevel());
        if (viewerAttributes.containsKey("mass")) jsonObject.put("mass", getMass());
        if (viewerAttributes.containsKey("shape")) jsonObject.put("shape", getShape());
        if (viewerAttributes.containsKey("physics")) jsonObject.put("physics", isPhysics());
        if (viewerAttributes.containsKey("size")) jsonObject.put("size", getSize());
        if (viewerAttributes.containsKey("title")) jsonObject.put("title", getTitle());
        if (viewerAttributes.containsKey("value")) jsonObject.put("value", getValue());
        colorsToJson(jsonObject);
        if (viewerAttributes.containsKey("x")) jsonObject.put("x", getX());
        if (viewerAttributes.containsKey("y")) jsonObject.put("y", getY());
		if (customAttributes != null) {
			if (customAttributes.containsKey("Cluster")) {
				jsonObject.put("group", customAttributes.get("Cluster"));
			}
		}
        return jsonObject;
    }

    // Method to convert color properties to JSONObject
    private void colorsToJson(JSONObject jsonObject) {
        if ( !viewerAttributes.containsKey("colorBorder") &&
            !viewerAttributes.containsKey("colorHighlight") &&
            !viewerAttributes.containsKey("colorHighlightBorder") &&
            !viewerAttributes.containsKey("colorHover") &&
            !viewerAttributes.containsKey("colorHoverBorder")) {
            if (viewerAttributes.containsKey("color"))
                jsonObject.put("color", getColor());
            return;
        }

        JSONObject colorObject = new JSONObject();
        jsonObject.put("color", colorObject);
        if (viewerAttributes.containsKey("color")) {
            colorObject.put("background", getColor());
        }
        if (viewerAttributes.containsKey("colorBorder")) {
            colorObject.put("border", getColorBorder());
        }

        if (viewerAttributes.containsKey("colorHighlightBorder")) {
            JSONObject highlightObject = new JSONObject();
            if (viewerAttributes.containsKey("colorHighlight")) {
                highlightObject.put("background", getColorHighlight());
            }
            highlightObject.put("border", getColorHighlightBorder());
            colorObject.put("highlight", highlightObject);
        } else if (viewerAttributes.containsKey("colorHighlight")) {
                colorObject.put("highlight", getColorHighlight());
        }

        if (viewerAttributes.containsKey("colorHoverBorder")) {
            JSONObject hoverObject = new JSONObject();
            if (viewerAttributes.containsKey("colorHover")) {
                hoverObject.put("background", getColorHover());
            }
            hoverObject.put("border", getColorHoverBorder());
            colorObject.put("hover", hoverObject);
        } else if (viewerAttributes.containsKey("colorHover")) {
            colorObject.put("hover", getColorHover());
        }
    }

    // Methods to add and remove property change listeners
    public void addPropertyChangeListener(PropertyChangeListener listener) {
        pcs.addPropertyChangeListener(listener);
    }

    public void removePropertyChangeListener(PropertyChangeListener listener) {
        pcs.removePropertyChangeListener(listener);
    }

    // Methods for custom attributes
    @Override
    public Object getAttribute(String key) {
        return customAttributes.get(key);
    }

    @Override
    public void setAttribute(String key, Object value) {
        Object oldValue = customAttributes.get(key);
        if (value.equals(oldValue)) return;
        customAttributes.put(key, value);
        pcs.firePropertyChange(key, oldValue, value);
    }

    @Override
    public Map<String, Object> getAttributes() {
        return new HashMap<>(customAttributes);
    }
    
    public Variable getLinkedVariable() {
    	        return linkedVariable;
    }

	public void setLinkedVariable(Variable linkedVariable) {
		this.linkedVariable = linkedVariable;
	}


    // Main method for testing
    // Main method for testing
    public static void main(String[] args) {
    	INode node1 = new Node("1", "Node 1");
        node1.setBorderWidth(2);
        node1.setOpacity(0.8);
        node1.setGroup("Group 1");
        node1.setHidden(false);
        node1.setLabelHighlightBold(true);
        node1.setLevel(1);
        node1.setMass(1.5);
        node1.setShape("circle");
        node1.setPhysics(true);
        node1.setSize(30);
        node1.setTitle("Title 1");
        node1.setValue(10.0);
        node1.setX(100.0);
        node1.setY(200.0);
        node1.setColor("#FF0000");
        node1.setColorBorder("#00FF00");

        INode node2 = new Node("2", "Node 2");
        node2.setBorderWidth(1);
        node2.setOpacity(1.0);
        node2.setGroup("Group 2");
        node2.setHidden(true);
        node2.setLabelHighlightBold(false);
        node2.setLevel(2);
        node2.setMass(2.0);
        node2.setShape("square");
        node2.setPhysics(false);
        node2.setSize(25);
        node2.setTitle("Title 2");
        node2.setValue(20.0);
        node2.setX(300.0);
        node2.setY(400.0);
        node2.setColor("#00FF00");
        node2.setColorHighlight("#FFFF00");

        INode node3 = new Node();
        node3.setBorderWidth(1);
        node3.setOpacity(1.0);
        node3.setGroup("Group 3");
        node3.setHidden(false);
        node3.setLabelHighlightBold(true);
        node3.setLevel(3);
        node3.setMass(3.0);
        node3.setShape("triangle");
        node3.setPhysics(true);
        node3.setSize(35);
        node3.setTitle("Title 3");
        node3.setValue(30.0);
        node3.setX(500.0);
        node3.setY(600.0);
        node3.setColor("#0000FF");
        node3.setColorHover("#FF00FF");

        System.out.println("Node 1 JSON: " + node1.toJson().toString(4));
        System.out.println("Node 2 JSON: " + node2.toJson().toString(4));
        System.out.println("Node 3 JSON: " + node3.toJson().toString(4));
    }
}