package com.biotechvana.netools.internal;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.Reader;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Set;

import org.apache.commons.csv.CSVFormat;
import org.apache.commons.csv.CSVParser;
import org.apache.commons.csv.CSVRecord;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.e4.core.di.annotations.Optional;
import org.eclipse.e4.ui.di.UISynchronize;
import org.eclipse.e4.ui.model.application.MApplication;
import org.eclipse.e4.ui.progress.IProgressService;
import org.eclipse.e4.ui.services.IServiceConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.fieldassist.AutoCompleteField;
import org.eclipse.jface.fieldassist.ComboContentAdapter;
import org.eclipse.jface.fieldassist.TextContentAdapter;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.ScrolledForm;
import org.eclipse.ui.forms.widgets.Section;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.biotechvana.commons.SharedImages;
import com.biotechvana.netools.IVariablesView;
import com.biotechvana.netools.IWhiteListView;
import com.biotechvana.netools.projects.Arc;
import com.biotechvana.netools.projects.Dataset;
import com.biotechvana.netools.projects.IProjectsManager;
import com.biotechvana.netools.projects.NetworkDesign;
import com.biotechvana.netools.projects.Project;
import com.biotechvana.netools.projects.Variable;
import com.biotechvana.netools.ui.dialogs.ImportListDialog;

import jakarta.annotation.PostConstruct;
import jakarta.inject.Inject;
import jakarta.inject.Named;

public class WhiteListView implements IWhiteListView {
	public static final Logger logger = LoggerFactory.getLogger(WhiteListView.class);
	FormToolkit toolkit;
	ScrolledForm detailsForm;

	Dataset selected_dataset;
	Project activeProject;
	NetworkDesign activeDesign;
	IProjectsManager projectsManager;

	@Inject
	IProgressService progressService;


	@Inject
	UISynchronize sync;


	List<Variable> variables = new ArrayList<Variable>();
	Composite mainComposite;
	TableViewer tableViewer;

	@Inject
	@Named(IServiceConstants.ACTIVE_SHELL) Shell activeShell;

	
	@Inject
	@Optional 
	public void setActiveDesign(@Named(IProjectsManager.ACTIVE_DESIGN) NetworkDesign networkDesign) {
		if (networkDesign == null) {
			activeDesign = null;
			activeProject = null;
			variables = new ArrayList<>();
		}
		else {
			this.activeDesign = networkDesign;
			this.activeProject = networkDesign.getProject();
			this.variables = activeProject.getVariables();
		}
		sync.asyncExec(
				() -> updateUI()
			);
	}
	


	/**
	 * Update GUI in must be called in the display thread
	 */
	private void updateUI() {
		if (detailsForm == null || detailsForm.isDisposed()) {
			return;
		}
		logger.debug("Updating UI for the View");
		if (activeDesign != null) {
			detailsForm.setText("Network Designer: " + activeDesign.getDesignName());
			tableViewer.setInput(activeDesign.getWhitelist());
		} else {
			detailsForm.setText("Network Designer - No Active Design");
			tableViewer.setInput(new ArrayList<>());
		}
		 setComboValues();

	}

	private void renderMainSection() {
		tableViewer = new TableViewer(mainComposite, SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL | SWT.FULL_SELECTION | SWT.BORDER);
		tableViewer.getTable().setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		tableViewer.getTable().setHeaderVisible(true);
		tableViewer.getTable().setLinesVisible(true);
		tableViewer.setContentProvider(new ArrayContentProvider());
		// tableViewer to display Arcs
		
		// WITHELIST COLUMNS
		// From column
		TableViewerColumn fromColumn = new TableViewerColumn(tableViewer, SWT.NONE);
		fromColumn.getColumn().setText("From");
		fromColumn.getColumn().setWidth(150);
		fromColumn.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(Object element) {
				Arc arc = (Arc) element;
				return  arc.getSource().getName();
			}
		});
		
		// To column
		TableViewerColumn toColumn = new TableViewerColumn(tableViewer, SWT.NONE);
		toColumn.getColumn().setText("To");
		toColumn.getColumn().setWidth(150);
		toColumn.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(Object element) {
				Arc arc = (Arc) element;
				return  arc.getTarget().getName();
			}
		});
		
		// Reason column
		TableViewerColumn reasonColumn = new TableViewerColumn(tableViewer, SWT.NONE);
		reasonColumn.getColumn().setText("Reason");
		reasonColumn.getColumn().setWidth(150);
		reasonColumn.setLabelProvider(new ColumnLabelProvider() {
			@Override 
			public String getText(Object element) {
				Arc arc = (Arc) element;
				return arc.getReason();
			}
		});
		
		if (activeDesign != null) {
			tableViewer.setInput(activeDesign.getWhitelist());
		} else {
			tableViewer.setInput(new ArrayList<>());
		}
		
	}

	



	

	@PostConstruct
	public void createPartControl(Composite parent, MApplication app, IProjectsManager projectsManager) { 
		app.getContext().set(IWhiteListView.class, this);
		this.projectsManager = projectsManager;
		toolkit = new FormToolkit(parent.getDisplay());

		detailsForm = toolkit.createScrolledForm(parent);
		detailsForm.setText("Network Designer");
		detailsForm.getBody().setLayout(new GridLayout(1, false));

		// Create filter section
		Section descSection = toolkit.createSection(detailsForm.getBody(), Section.TITLE_BAR);
		descSection.setText("Add/Remove edges from Whitelist ");
		descSection.setLayoutData(new GridData(SWT.FILL, SWT.TOP, true, false));

		Composite descComposite = toolkit.createComposite(descSection);
		descComposite.setLayout(new FillLayout());

		createDescSection(descComposite);

		descSection.setClient(descComposite);

		// Create variables section
		Section mainSection = toolkit.createSection(detailsForm.getBody(), Section.TITLE_BAR);
		mainSection.setText("Whitelist");
		mainSection.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));

		mainComposite = toolkit.createComposite(mainSection);
		mainComposite.setLayout(new GridLayout(1, false));
		renderMainSection();
		mainSection.setClient(mainComposite);
	}
	
	Combo comboVarFrom;
	Combo comboVarTo ;
	Text textReason;
	Button btnAdd;
	Button btnRemove;
	Button btnImport;
	Button btnDelete;
	
	private void createDescSection(Composite descComposite) {
		Composite addForm = toolkit.createComposite(descComposite);
		addForm.setLayout(new GridLayout(3, false));
		addForm.setLayoutData(new GridData(SWT.FILL, SWT.TOP, true, false));
		
		// from field
		toolkit.createLabel(addForm, "From: ");
        comboVarFrom = new Combo(addForm, SWT.BORDER | SWT.DROP_DOWN | SWT.SINGLE   | SWT.FLAT);
        comboVarFrom.setLayoutData(new GridData(SWT.FILL, SWT.TOP, true, false, 2, 1));
           
        // to field
        toolkit.createLabel(addForm, "To: ");
        comboVarTo = new Combo(addForm, SWT.BORDER | SWT.DROP_DOWN | SWT.SINGLE   | SWT.FLAT);
        comboVarTo.setLayoutData(new GridData(SWT.FILL, SWT.TOP, true, false, 2, 1));
        
        // reason field
        toolkit.createLabel(addForm, "Reason: ");
        textReason = new Text(addForm, SWT.BORDER | SWT.DROP_DOWN | SWT.SINGLE | SWT.FLAT);
        textReason.setLayoutData(new GridData(SWT.FILL, SWT.TOP, true, false, 2, 1));
        
        
        Composite buttonBar = toolkit.createComposite(addForm);
        buttonBar.setLayout(new GridLayout(5, false));
        buttonBar.setLayoutData(new GridData(SWT.FILL, SWT.TOP, true, false, 3, 1));
        
        // add add button
        btnAdd = toolkit.createButton(buttonBar, "Add", SWT.PUSH);
        btnAdd.setLayoutData(new GridData(SWT.RIGHT, SWT.TOP, true, false));
        btnAdd.setImage(SharedImages.ADD_ITEM);
        btnAdd.setEnabled(false);
        
        // add remove button
        btnRemove = toolkit.createButton(buttonBar, "Remove", SWT.PUSH);
        btnRemove.setLayoutData(new GridData(SWT.LEFT, SWT.TOP, true, false));
        btnRemove.setImage(SharedImages.REMOVE_ITEM);
        btnRemove.setEnabled(false);
        
        Label spacer = new Label(buttonBar, SWT.NONE);
        spacer.setLayoutData(new GridData(30, SWT.DEFAULT));
        
        // add import list button
        btnImport = toolkit.createButton(buttonBar, "Import List", SWT.PUSH);
        btnImport.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
        btnImport.setImage(SharedImages.ADD_ITEM);
        btnImport.setEnabled(true);
        
        // add delete list button
        btnDelete = toolkit.createButton(buttonBar, "Remove List", SWT.PUSH);
        btnDelete.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
        btnDelete.setImage(SharedImages.REMOVE_ITEM);
        btnDelete.setEnabled(true);
        
        SelectionAdapter updateAdapter =   new SelectionAdapter() {
   		 
			@Override
			public void widgetSelected(SelectionEvent e) {
				String from = comboVarFrom.getText();
				String to = comboVarTo.getText();
				//logger.debug("Adding edge from {} to {}", from, to);
				if (varMap.containsKey(from) && varMap.containsKey(to)) {
					System.out.printf("Adding edge from %s to %s\n", from, to);
					btnAdd.setEnabled(true);
					btnRemove.setEnabled(true);
				} else {
					btnAdd.setEnabled(false);
					btnRemove.setEnabled(false);
				}
			}
		};
		comboVarFrom.addSelectionListener(updateAdapter);
		comboVarTo.addSelectionListener(updateAdapter);
		
		btnAdd.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				System.out.println("Add button pressed");
				String from = comboVarFrom.getText();
				String to = comboVarTo.getText();
				String reason = textReason.getText();
				
				if (!from.equals("*") && !to.equals("*")) {
					activeDesign.addWhitelist(varMap.get(from), varMap.get(to), reason);
				} else if (from.equals("*")) {
					for (Variable v: variables) {
						if (v != varMap.get(to)) {
							activeDesign.addWhitelist(v, varMap.get(to), reason);
						}
					}
				} else if (to.equals("*") && (!from.equals(to))) {
					for (Variable v: variables) {
						if (varMap.get(from) != v) {
							activeDesign.addWhitelist(varMap.get(from), v, reason);
						}
					}
				}
							
				tableViewer.refresh();
			}
		});
		
		btnRemove.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				System.out.println("Remove button pressed");
				String from = comboVarFrom.getText();
				String to = comboVarTo.getText();
				
				if (!from.equals("*") && !to.equals("*")) {
					activeDesign.removeWhitelist(varMap.get(from), varMap.get(to));
				} else if (from.equals("*")) {
					for (Variable v: variables) {
						activeDesign.removeWhitelist(v, varMap.get(to));
					}
				} else if (to.equals("*")) {
					for (Variable v: variables) {
						activeDesign.removeWhitelist(varMap.get(from), v);
					}
				}
				
				tableViewer.refresh();
			}
		});
		
		
		btnImport.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				try {
					executeImport();
				} catch (InvocationTargetException e1) {
					// TODO Auto-generated catch block
					e1.printStackTrace();
				} catch (InterruptedException e1) {
					// TODO Auto-generated catch block
					e1.printStackTrace();
				}
				
				tableViewer.refresh();
			}
			
		});
		
		btnDelete.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				try {
					executeDeletion();
				} catch (InvocationTargetException e1) {
					// TODO Auto-generated catch block
					e1.printStackTrace();
				} catch (InterruptedException e1) {
					// TODO Auto-generated catch block
					e1.printStackTrace();
				}
				
				tableViewer.refresh();
			}
		});
		
		ModifyListener modifyAdapter = new ModifyListener() {
			
			@Override
			public void modifyText(ModifyEvent e) {
				String from = comboVarFrom.getText();
				String to = comboVarTo.getText();
				//logger.debug("Adding edge from %s to %s", from, to);
				if (varMap.containsKey(from) && varMap.containsKey(to)) {
					System.out.printf("Adding edge from %s to %s\n", from, to);
					btnAdd.setEnabled(true);
					btnRemove.setEnabled(true);
				} else {
					btnAdd.setEnabled(false);
					btnRemove.setEnabled(false);
				}
			}
		};
		
		
		comboVarTo.addModifyListener(modifyAdapter);
		comboVarFrom.addModifyListener(modifyAdapter);
		
        
        setComboValues();
       
	}
	HashMap<String, Variable> varMap = new HashMap<>();

	protected void setComboValues() {
		
		// fill from combo and to combo
		comboVarFrom.removeAll();
		comboVarTo.removeAll();
		varMap.clear();
		
		comboVarFrom.add("*");
		comboVarTo.add("*");
		varMap.put("*", null);
		for (Variable v : variables) {
			comboVarFrom.add(v.getName());
			comboVarTo.add(v.getName());
			varMap.put(v.getName(), v);
		}
		if (activeDesign != null) {
            // btnAdd.setEnabled(true);
            comboVarFrom.setEnabled(true);
            comboVarTo.setEnabled(true);
            
            new AutoCompleteField(comboVarFrom, new ComboContentAdapter(),  comboVarFrom.getItems());
            new AutoCompleteField(comboVarTo, new ComboContentAdapter(), comboVarTo.getItems());

        }
		else {
        	btnAdd.setEnabled(false);
        	btnRemove.setEnabled(false);
        	comboVarFrom.setEnabled(false);
        	comboVarTo.setEnabled(false);
        }
		
	}
	
	public void refreshView() {
		sync.asyncExec(
				() -> updateUI()
			);
	}
	
	
	
protected void executeImport() throws InvocationTargetException, InterruptedException {
    	
    	ImportListDialog dialog = new ImportListDialog(activeShell , activeProject);
        if (dialog.open() == ImportListDialog.OK) {
            // Retrieve the dataset details from the dialog
            String csvSeparator =   dialog.getCsvSeparator();
            char separator = csvSeparator.charAt(0);
            //String sampleColumn =   dialog.getSelectedSampleNameClm();
            File csvFile = dialog.getCsvFile();
                  
            System.out.println("CSV File: " + (csvFile != null ? csvFile.getAbsolutePath() : "None"));
            
            progressService.run(true, false, new IRunnableWithProgress() {

					@Override
					public void run(IProgressMonitor monitor)
							throws InvocationTargetException, InterruptedException {
	
				            Reader reader = null;
							try {
								reader = new FileReader(csvFile);
							} catch (FileNotFoundException e) {
								// TODO Auto-generated catch block
								e.printStackTrace();
							}
				            CSVParser csvParser = null;
							try {
								csvParser = new CSVParser(reader,
								        CSVFormat.DEFAULT.withDelimiter(separator).withFirstRecordAsHeader());
							} catch (IOException e) {
								// TODO Auto-generated catch block
								e.printStackTrace();
							}
				            List<String[]> data = new ArrayList<>();
				            List<String> headers = csvParser.getHeaderNames();
				            logger.info("Headers: " + headers);
				            for (CSVRecord csvRecord : csvParser) {
				                String[] row = new String[headers.size()];
				                for (int i = 0; i < headers.size(); i++) {
				                    row[i] = csvRecord.get(i);
				                }
				                data.add(row);
				            }
				            
				            List <String> fromList = new ArrayList<>();
				            List <String> toList = new ArrayList<>();
				            Variable from = null;
				            Variable to = null;
							for (String[] row : data) {
							    if (row.length > 0 && row[0] != null && !row[0].isEmpty()) {
							        fromList.add(row[0]);
							        from = activeProject.getVariableByName(row[0]);
							    }
							    if (row.length > 1 && row[1] != null && !row[1].isEmpty()) {
							        toList.add(row[1]);
							        to = activeProject.getVariableByName(row[1]);
							    }
							    activeDesign.addWhitelist(from, to);
							}
						
						if (monitor.isCanceled()) {
							// TODO :: handle cancellation
							// return;
						}
						projectsManager.saveProject(activeProject,monitor);
						
					}
					
				});
                   
        }
    }
    
    
    protected void executeDeletion() throws InvocationTargetException, InterruptedException {
    	
				if (activeDesign == null) {
					logger.error("No design selected");
					return;
				}
    	
    	logger.info("Deleting list " + activeDesign.getDesignName());
    	
    	// show a dialog to confirm the deletion
    	boolean confirmed = MessageDialog.openConfirm(activeShell, "Confirm Deletion", 
    			"Are you sure you want to delete the elements of the list ?");
    	if (!confirmed) {
    		return;
    	}
    	
		progressService.run(true, false, monitor -> {
			monitor.beginTask("Deleting list ", 1);
			
			activeDesign.getWhitelist().clear();
			
			projectsManager.saveProject(activeProject, monitor);
			monitor.done();
		});
    	
    }

}
