package com.biotechvana.netools.internal;

import java.io.File;
import java.lang.reflect.InvocationTargetException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.e4.core.di.annotations.Optional;
import org.eclipse.e4.core.di.extensions.EventTopic;
import org.eclipse.e4.ui.di.UIEventTopic;
import org.eclipse.e4.ui.di.UISynchronize;
import org.eclipse.e4.ui.model.application.MApplication;
import org.eclipse.e4.ui.progress.IProgressService;
import org.eclipse.e4.ui.services.IServiceConstants;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.swt.SWT;
import org.eclipse.swt.browser.Browser;
import org.eclipse.swt.browser.BrowserFunction;
import org.eclipse.swt.browser.ProgressAdapter;
import org.eclipse.swt.browser.ProgressEvent;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.TabFolder;
import org.eclipse.swt.widgets.TabItem;
import org.eclipse.ui.forms.events.HyperlinkAdapter;
import org.eclipse.ui.forms.widgets.Form;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.biotechvana.netools.models.NetworkModel;
import com.biotechvana.netools.projects.IProjectsManager;
import com.biotechvana.netools.projects.Network;
import com.biotechvana.netools.projects.NetworkBuild;
import com.biotechvana.netools.projects.Project;
import com.biotechvana.netools.projects.ProjectEvents;
import com.biotechvana.workflow.tracking.ExecStatus;

import jakarta.annotation.PostConstruct;
import jakarta.inject.Inject;
import jakarta.inject.Named;

/**
 * This class is responsible for displaying the network project view
 * Display all current network build and corresponding status, 
 * Network initialized from network build to load one of the network
 */
public class NetworkProjectView implements INetworkProjectView {
	public static final Logger logger = LoggerFactory.getLogger(NetworkProjectView.class);
	
	Browser browser;
	
	FormToolkit toolkit;
	Form detailsForm;
	
	
	private ScrolledComposite jobScrolledComposite;
	private Composite mainBody;

	Project activeProject;
	IProjectsManager projectsManager;

	@Inject
	IProgressService progressService;


	@Inject
	UISynchronize sync;

	@Inject
	MApplication app;

	TableViewer networksTableViewer;
	TableViewer buildsTableViewer;
	private TabFolder tabFolder;
	
	/*private final NetworkView networkView; // instància, no static

    public NetworkProjectView(NetworkView view) {
        this.networkView = view;
    }*/
	
	
	@Inject
	@Named(IServiceConstants.ACTIVE_SHELL)
	Shell shell;

	@Inject
	@Optional 
	public void setActiveProject(@Named(IProjectsManager.ACTIVE_PROJECT) Project project) {
		activeProject = project;
		sync.asyncExec(
				() -> updateUI()
			);
	}

	@Inject
	void subscribeNetworkBuildAddedToProject(@Optional @EventTopic(ProjectEvents.NETWORK_BUILD_ADDED) NetworkBuild build ) {
		sync.asyncExec(() -> updateUI());
	}
	
	@Inject
	void subscribeNetworkBuildUpdatedToProject(@Optional @UIEventTopic(ProjectEvents.NETWORK_BUILD_UPDATED) NetworkBuild build ) {
		if (detailsForm == null || detailsForm.isDisposed()) {
			return;
		}
		if (buildsTableViewer != null && build!=null) {
			buildsTableViewer.update(build, null);
		}
    }

	@Inject
	void subscribeNetworkBuildDeletedToProject(@Optional @UIEventTopic(ProjectEvents.NETWORK_BUILD_DELETED) NetworkBuild build ) {
		if (detailsForm == null || detailsForm.isDisposed()) {
			return;
		}
		if (buildsTableViewer != null && build!=null) {
			buildsTableViewer.refresh();
		}
    }
	
	
	@Inject
	void subscribeNetworkAddedToProject(@Optional @UIEventTopic(ProjectEvents.NETWORK_ADDED) Network network ) {
		if (detailsForm == null || detailsForm.isDisposed()) {
			return;
		}
		if (networksTableViewer != null && activeProject!=null) {
			networksTableViewer.setInput( activeProject.getNetworks()  );
		}
    }
	@Inject
	void subscribeNetworkUpdatedToProject(@Optional @UIEventTopic(ProjectEvents.NETWORK_UPDATED) Network network ) {
		if (detailsForm == null || detailsForm.isDisposed()) {
			return;
		}
		if (networksTableViewer != null && activeProject!=null) {
			networksTableViewer.setInput( activeProject.getNetworks()  );
		}
    }
	@Inject
	void subscribeNetworkDeletedProject(@Optional @UIEventTopic(ProjectEvents.NETWORK_DELETED) Network network ) {
		if (detailsForm == null || detailsForm.isDisposed()) {
			return;
		}
		if (networksTableViewer != null && activeProject!=null) {
			networksTableViewer.setInput( activeProject.getNetworks()  );
		}
    }
	
	/**
	 * Update GUI in must be called in the display thread
	 */
	private void updateUI() {
		if (detailsForm == null || detailsForm.isDisposed()) {
			return;
		}
		logger.debug("Updating UI for the NetworkProjectView");
		if (activeProject == null) {
			detailsForm.setText("Networks");
			detailsForm.setMessage(" - No project is selected - ");
			buildsTableViewer.setInput(null);
			networksTableViewer.setInput(null);
			
			return;
		}
		detailsForm.setText("Networks");
		detailsForm.setMessage(" - " + activeProject.getProjectName() + " - ");
		buildsTableViewer.setInput(activeProject.getNetworkBuilds());
		networksTableViewer.setInput( activeProject.getNetworks()  );
		// TODO :: networksTableViewer.setInput(activeProject.getNetworks()); 


	}

	

	



	

	@PostConstruct
	public void createPartControl(Composite parent, MApplication app, IProjectsManager projectsManager) { 
		app.getContext().set(INetworkProjectView.class, this);
		this.projectsManager = projectsManager;
		toolkit = new FormToolkit(parent.getDisplay());

		detailsForm = toolkit.createForm(parent);
		detailsForm.setText("Structure Learning Projects");
		detailsForm.setMessage("---");
		detailsForm.getBody().setLayout(new FillLayout());
		detailsForm.addMessageHyperlinkListener(new HyperlinkAdapter());
		toolkit.decorateFormHeading(detailsForm);
		
		jobScrolledComposite = new ScrolledComposite( detailsForm.getBody(),  SWT.H_SCROLL | SWT.V_SCROLL);
		mainBody =  toolkit.createComposite(jobScrolledComposite);
		mainBody.setLayout(new GridLayout(1, false));
		createMainBody();
		jobScrolledComposite.setMinSize(mainBody.computeSize(SWT.DEFAULT, SWT.DEFAULT));
	    jobScrolledComposite.setContent(mainBody);
	    jobScrolledComposite.setExpandVertical(true);
	    jobScrolledComposite.setExpandHorizontal(true);
	    
	    
	    browser = new Browser(detailsForm, SWT.NONE);
	    browser.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		//browser.setVisible(false);
		//browser.setLayoutData(new GridData(0, 0));
		
	    HtmlLoader.load(browser, "Exporter.html");
	    //Path html = Paths.get("C:/Users/ivanc/Projects/NetTools/netools/com.biotechvana.netools/bundles/com.biotechvana.netools.ui/src/com/biotechvana/netools/internal/resources");
	    //browser.setUrl(html.toUri().toString());
	    
	    /*browser.addProgressListener(new ProgressAdapter() {
	    	  @Override public void completed(ProgressEvent e) {
	    	    Object ua = browser.evaluate("return navigator.userAgent;");
	    	    System.out.println("UA: " + ua);
	    	    System.out.println("JVM arch: " + System.getProperty("sun.arch.data.model") + "-bit");
	    	  }
	    	});*/

	}
	
	private void addBrowserFunction() {
		// Register BrowserFunction for each event
		new BrowserFunction(browser, "onClick") {
			@Override
			public Object function(Object[] arguments) {
				logger.debug("Click event received in Java with arguments: " + arguments[0]);
				System.out.println("Click event received in Java with arguments: " + arguments[0]);
				return null;
			}
		};
	}

	private void createMainBody() {
		// mainBody has a grid layout with 1 column
		
		Label label = new Label(mainBody, SWT.NONE);
		label.setText("Project Details");

		// Add a tab folder to the mainBody
		tabFolder = new TabFolder(mainBody, SWT.NONE);
		tabFolder.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));

		// Create Networks tab
		TabItem networksTab = new TabItem(tabFolder, SWT.NONE);
		networksTab.setText("Networks");
		Composite networksComposite = new Composite(tabFolder, SWT.NONE);
		networksComposite.setLayout(new GridLayout(1, false));
		networksTab.setControl(networksComposite);
		createNetworksTab(networksComposite);

		// Create Builds tab
		TabItem buildsTab = new TabItem(tabFolder, SWT.NONE);
		buildsTab.setText("Builds");
		Composite buildsComposite = new Composite(tabFolder, SWT.NONE);
		buildsComposite.setLayout(new GridLayout(1, false));
		buildsTab.setControl(buildsComposite);
		createBuildsTab(buildsComposite);

		// Add selection listener for tabs
		tabFolder.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				handleTabSelection(tabFolder.getSelectionIndex());
			}
		});
	}

	private void createNetworksTab(Composite parent) {
		// Add a label describing the table
		Label networksLabel = new Label(parent, SWT.NONE);
		networksLabel.setText("List of Networks in the Current Project:");

		// Create table viewer
		networksTableViewer = new TableViewer(parent, SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL | SWT.FULL_SELECTION | SWT.BORDER);
		
		// Configure the table
		networksTableViewer.getTable().setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		networksTableViewer.getTable().setHeaderVisible(true);
		networksTableViewer.getTable().setLinesVisible(true);

		// Create columns
		TableViewerColumn nameColumn = new TableViewerColumn(networksTableViewer, SWT.NONE);
		nameColumn.getColumn().setText("Network Name");
		nameColumn.getColumn().setWidth(200);
		nameColumn.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(Object element) {
				// get the network name
				Network network = (Network) element;
				return network.getNetworkName();
				// return "Network Name";
			}
			
		});

		TableViewerColumn statusColumn = new TableViewerColumn(networksTableViewer, SWT.NONE);
		statusColumn.getColumn().setText("Status");
		statusColumn.getColumn().setWidth(100);
		statusColumn.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(Object element) {
				// get the network status
				
				Network network = (Network) element;
				if (network.getModel() == null && network.getStatus() == ExecStatus.Finished) {
					return "OK";
				}
				if ( network.getStatus() != ExecStatus.Finished ) {
					return "!!";
				}
				return "Ready";
			}
			@Override
			public Image getImage(Object element) {
				Network network = (Network) element;
				if (network.getModel() == null && network.getStatus() == ExecStatus.Finished) {
					return null;
				}
				return ViewUtils.getExecStatusImage(network.getStatus());
				//return ViewUtils.getExecStatusImage(ExecStatus.Finished);
			}
		});

		// Create context menu
		MenuManager menuManager = new MenuManager();
		Menu menu = menuManager.createContextMenu(networksTableViewer.getTable());
		networksTableViewer.getTable().setMenu(menu);

		// Add menu items
		menuManager.add(new Action("View Network") {
			@Override
			public void run() {
				logger.debug("View Network action triggered");
				Network selectedNetwork = getSelectedNetwork();
				if (selectedNetwork != null) {
					
					try {
						progressService.run(true, false, monitor -> {
							try {
								selectedNetwork.loadNetworkModel(monitor);
							} catch (Exception e) {
								// TODO Auto-generated catch block
								e.printStackTrace();
							}
							
							if (selectedNetwork.getModel() == null) {
								selectedNetwork.createNetworkModel(monitor);
							}
							// TODO :: open a new view to display the network
							sync.asyncExec(() -> {
								// open the network view
								app.getContext().set(NetworkModel.class, selectedNetwork.getModel());
							});
							//app.getContext().set(NetworkModel.class, selectedNetwork.getModel());
						});
					} catch (InvocationTargetException | InterruptedException e) {
						// TODO Auto-generated catch block
						e.printStackTrace();
					}
					
					
					
					
				}
				else {
					logger.debug("No network selected");
					// TODO :: add message or dialog feedback to the user
				}
			}
		});
		
		/*browser.addProgressListener(new ProgressAdapter() {
	        @Override
	        public void completed(ProgressEvent event) {
	            System.out.println("Exporter.html loaded.");
	        }
	    });*/
	    
	    //addBrowserFunction();
		
		/*menuManager.add(new Action("Export Network") {
		    @Override
		    public void run() {
		        browser.getDisplay().asyncExec(() -> {
		        	String networkJson = (String) app.getContext().get("networkJson");
		        	System.out.println(networkJson);
		            boolean ok = browser.execute("exportNetworkToHighQualityPNG(" + networkJson + ");");
		            System.out.println("JS executed? " + ok);
		        });
		    }
		});*/
		
		/*menuManager.add(new Action("Export Network") {
			@Override
			public void run() {
				logger.debug("Export Network action triggered");
				
				try {
					sync.asyncExec(() -> {
						browser.execute("exportNetworkToHighQualityPNG();");
					});

				} catch (Exception e) {
					e.printStackTrace();
				}
			}
			
		});*/
		
		menuManager.add(new Separator()); 
		
		menuManager.add(new Action("Reset") {
			@Override
			public void run() {
				// TODO: Implement load action
				logger.debug("ReLoad network action triggered");
				Network selectedNetwork = getSelectedNetwork();
				
				if (selectedNetwork != null) {
					
					// confirm to reload the network
					boolean confirm = MessageDialog.openConfirm(shell, "Confirm Reload",
							"Are you sure you want to reload the network " + selectedNetwork.getNetworkName() + ", this will reset the network to the initial state ?");
					if (!confirm) {
						return;
					}
					
					try {
						progressService.run(true, false, monitor -> {
							selectedNetwork.createNetworkModel(monitor);
							app.getContext().set(NetworkModel.class, selectedNetwork.getModel());
						});
					} catch (InvocationTargetException | InterruptedException e) {
						// TODO Auto-generated catch block
						e.printStackTrace();
					}
					
					
					
					
				}
				else {
					logger.debug("No network selected");
					// TODO :: add message or dialog feedback to the user
				}
			}
		});

//		menuManager.add(new Action("Refresh") {
//			@Override
//			public void run() {
//				// TODO: Implement refresh action
//				logger.debug("Refresh network action triggered");
//			}
//		});

		menuManager.add(new Action("Delete") {
			@Override
			public void run() {
				// TODO: Implement delete action
				logger.debug("Delete network action triggered");
				Network selectedNetwork = getSelectedNetwork();
				if (selectedNetwork != null) {
					Project project = selectedNetwork.getProject();
					// show confirmation
					boolean confirm = MessageDialog.openConfirm(shell, "Confirm Deletion",
							"Are you sure you want to delete the network " + selectedNetwork.getNetworkName() + "?");
					if (!confirm) {
						return;
					}
					try {
						progressService.run(true, false, monitor -> {
							monitor.beginTask("Deleting network " + selectedNetwork.getNetworkName(),
									IProgressMonitor.UNKNOWN);
							project.removeNetwork(selectedNetwork, monitor);
							projectsManager.saveProject(project, monitor);
							monitor.done();
						}

						);
					} catch (InvocationTargetException | InterruptedException e) {
						// TODO Auto-generated catch block
						e.printStackTrace();
					}
				}
				else {
					logger.debug("No network selected");
					// TODO :: add message or dialog feedback to the user
				}
			}
		});

//		menuManager.add(new Action("Clone") {
//			@Override
//			public void run() {
//				// TODO: Implement clone action
//				logger.debug("Clone network action triggered");
//			}
//		});

		// Set input (empty for now)
		networksTableViewer.setContentProvider(ArrayContentProvider.getInstance());
		if(activeProject != null) {
			networksTableViewer.setInput( activeProject.getNetworks()  );
		}
	}

	private void createBuildsTab(Composite parent) {
		// Add a label describing the table
		Label buildsLabel = new Label(parent, SWT.NONE);
		buildsLabel.setText("List of Network Builds in the Active Project:");

		// Create table viewer for builds
		buildsTableViewer = new TableViewer(parent, SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL | SWT.FULL_SELECTION | SWT.BORDER);
		
		// Configure the table
		buildsTableViewer.getTable().setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		buildsTableViewer.getTable().setHeaderVisible(true);
		buildsTableViewer.getTable().setLinesVisible(true);

		// Create columns
		TableViewerColumn buildNameColumn = new TableViewerColumn(buildsTableViewer, SWT.NONE);
		buildNameColumn.getColumn().setText("Build Name");
		buildNameColumn.getColumn().setWidth(200);
		buildNameColumn.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(Object element) {
				return ((NetworkBuild) element).getNetworkName();
			}
		});

		TableViewerColumn buildStatusColumn = new TableViewerColumn(buildsTableViewer, SWT.NONE);
		buildStatusColumn.getColumn().setText("Status");
		buildStatusColumn.getColumn().setWidth(100);
		buildStatusColumn.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(Object element) {
				return ((NetworkBuild) element).getBuildStatus().toString();
			}
			@Override
			public Image getImage(Object element) {
				return ViewUtils.getExecStatusImage(((NetworkBuild) element).getBuildStatus());
			}
		});

		// Create context menu
		MenuManager menuManager = new MenuManager();
		Menu menu = menuManager.createContextMenu(buildsTableViewer.getTable());
		buildsTableViewer.getTable().setMenu(menu);

		// Add menu items
		menuManager.add(new Action("Refresh") {
			@Override
			public void run() {
				NetworkBuild selectedBuild = getSelectedBuild();
				if (selectedBuild != null) {
					refreshNetworkBuild(selectedBuild);
				}
			}
		});

		menuManager.add(new Action("Load Network") {
			@Override
			public void run() {
				NetworkBuild selectedBuild = getSelectedBuild();
				if (selectedBuild != null) {
					loadNetworkBuild(selectedBuild);
				}
			}
		});

		menuManager.add(new Action("Stop") {
			@Override
			public void run() {
				NetworkBuild selectedBuild = getSelectedBuild();
				if (selectedBuild != null) {
					stopNetworkBuild(selectedBuild);
				}
			}
		});

		menuManager.add(new Action("Delete") {
			@Override
			public void run() {
				List<NetworkBuild> selectedBuild = getSelectedBuilds();
				if (selectedBuild != null) {
					deleteNetworkBuild(selectedBuild);
				}
			}
		});
		buildsTableViewer.setContentProvider(ArrayContentProvider.getInstance());
		// Set input
		if (activeProject != null) {
			
			buildsTableViewer.setInput(activeProject.getNetworkBuilds());
		}
	}

	// Helper method to get the selected build
	private NetworkBuild getSelectedBuild() {
		IStructuredSelection selection = (IStructuredSelection) buildsTableViewer.getSelection();
		if (!selection.isEmpty()) {
			return (NetworkBuild) selection.getFirstElement();
		}
		return null;
	}
	private List<NetworkBuild> getSelectedBuilds() {
		IStructuredSelection selection = (IStructuredSelection) buildsTableViewer.getSelection();
	    List<NetworkBuild> selectedBuilds = new ArrayList<>();
	    for (Object element : selection.toList()) {
	        if (element instanceof NetworkBuild) {
	            selectedBuilds.add((NetworkBuild) element);
	        }
	    }
	    return selectedBuilds;
		
	}
	private Network  getSelectedNetwork() {
		IStructuredSelection selection = (IStructuredSelection)  networksTableViewer.getSelection();
		if (!selection.isEmpty()) {
			return (Network) selection.getFirstElement();
		}
		return null;
	}
	
	// Stub methods for menu actions
	private void refreshNetworkBuild(NetworkBuild build) {
		logger.debug("Refreshing build: " + build.getNetworkName());
		// TODO: Implement refresh logic
		projectsManager.resfreshNetworkBuild(build);
		
	}

	private void loadNetworkBuild(NetworkBuild build) {
		String networkName = build.getNetworkName();
		logger.debug("Loading network from build: " + networkName);
		if (build.getProject().hasNetwork(networkName)) {
			logger.debug("Network " + networkName + " already loaded");
			// show a confirmation dialog to reload and reset the network
			// show a confirmation dialog to reload and reset the network
			boolean confirm = MessageDialog.openConfirm(shell, 
			                                            "Confirm Reload/Reset", 
			                                            "Network " + networkName + " already loaded. Do you want to reload and reset the network?");
			if (!confirm) {
			    return;
			}
		}
		// TODO: Implement network loading logic
		// check if we already have the network loaded before
		try {
			progressService.run(true, true, monitor -> {
				build.createNetwork(monitor);
			});
		} catch (InvocationTargetException | InterruptedException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
	}

	private void stopNetworkBuild(NetworkBuild build) {
		logger.debug("Stopping build: " + build.getNetworkName());
		// TODO: Implement stop logic
	}

	private void deleteNetworkBuild(List<NetworkBuild> selectedBuilds) {
		if (selectedBuilds.isEmpty()) {
			logger.debug("No build selected for deletion");
			return;
		}
		if( selectedBuilds.size() == 1) {
			NetworkBuild selectedBuild = selectedBuilds.get(0);
			logger.debug("Deleting build: " + selectedBuild.getNetworkName());
			
			
			// show confirmation
			boolean confirm = MessageDialog.openConfirm(shell, "Confirm Deletion",
					"Are you sure you want to delete the network build " + selectedBuild.getNetworkName() + "?");
			if (!confirm) {
				return;
			}
			try {
				progressService.run(true, false, 
						monitor -> {
							monitor.beginTask("Deleting network build " + selectedBuild.getNetworkName(), IProgressMonitor.UNKNOWN);
							projectsManager.deleteNetworkBuild(selectedBuild);
							monitor.done();
						}
						
					);
			} catch (InvocationTargetException | InterruptedException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
            return;
        }
		logger.debug("Deleting multiple builds");
		// show confirmation
		boolean confirm = MessageDialog.openConfirm(shell, "Confirm Deletion",
				"Are you sure you want to delete the selected network builds?");
		if (!confirm) {
			return;
		}
		try {
            progressService.run(true, false, 
                    monitor -> {
                    	monitor.beginTask("Deleting selected network builds", selectedBuilds.size());
                        for (NetworkBuild build : selectedBuilds) {
                            projectsManager.deleteNetworkBuild(build);
                            monitor.worked(1);
                        }
                        monitor.done();
                    }
                );
        } catch (InvocationTargetException | InterruptedException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
		
		
	}
	
	

	private void handleTabSelection(int tabIndex) {
		logger.debug("Tab selection changed to index: " + tabIndex);
		switch (tabIndex) {
			case 0: // Networks tab
				// Handle networks tab selection
				
				break;
			case 1: // Builds tab
				// Handle builds tab selection
				break;
		}
	}





}
