package com.biotechvana.netools.projects;

import java.io.File;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.InputStreamReader;
import java.net.URI;
import java.net.URISyntaxException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.csv.CSVFormat;
import org.apache.commons.csv.CSVParser;
import org.apache.commons.vfs2.FileObject;
import org.apache.commons.vfs2.FileSystemException;
import org.apache.commons.vfs2.FileSystemManager;
import org.apache.commons.vfs2.FileSystemOptions;
import org.apache.commons.vfs2.Selectors;
import org.apache.commons.vfs2.VFS;
import org.apache.commons.vfs2.provider.ftp.FtpFileSystemConfigBuilder;
import org.apache.commons.vfs2.provider.sftp.SftpFileSystemConfigBuilder;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.e4.core.di.annotations.Creatable;
import org.eclipse.e4.core.di.annotations.Optional;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.biotechvana.commons.ftpservice.IFTPManager;
import com.biotechvana.commons.model.UserLoginInfo;
import com.biotechvana.users.IUsersService;
import com.fasterxml.jackson.annotation.JsonAutoDetect.Visibility;
import com.fasterxml.jackson.annotation.PropertyAccessor;
import com.fasterxml.jackson.core.exc.StreamWriteException;
import com.fasterxml.jackson.databind.DatabindException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.json.JsonMapper;

import jakarta.inject.Inject;
import jakarta.inject.Named;

@Creatable
public class RemoteStorage {
	private static final Logger logger = LoggerFactory.getLogger(RemoteStorage.class);

	public  static final String DATASETS_DIRECTORY = "datasets";
	public  static final String PROJECTS_DIRECTORY = "projects";
	public  static final String UPLOADS_DIRECTORY = "uploads";
	public  static final String NETWORKS_DIRECTORY = "networks";
	public  static final String DESIGNS_DIRECTORY = "designs";
	public  static final String TASKS_DIRECTORY = "tasks";








	private FileSystemManager fileSystemManager;
	private FileSystemOptions SFTPopts;
	private FileSystemOptions FTPopts;
	private UserLoginInfo activeLogin;

	private FileObject remoteCWD;
	private FileObject appRoot;
	FileObject projectsFolder;
	public RemoteStorage() throws FileSystemException  {
		this.fileSystemManager = VFS.getManager();
		SFTPopts = new FileSystemOptions();
		//FIXME
		SftpFileSystemConfigBuilder.getInstance().setIdentities(SFTPopts, new File[0]);
		SftpFileSystemConfigBuilder.getInstance().setSessionTimeout(SFTPopts, Duration.ofSeconds(5));

		FTPopts = new FileSystemOptions();
		FtpFileSystemConfigBuilder.getInstance().setPassiveMode(FTPopts, true);
	}

	@Inject
	public void setActiveLogin(@Optional @Named(IUsersService.ACTIVE_LOGIN) UserLoginInfo activeLogin ) {
		if(this.activeLogin != activeLogin) {
			// TODO :: about to change
			// TODO :: Close previous connection 
			if (activeLogin == null) {
				if (projectsManager != null) {
					projectsManager.userChanging();
				}
			}

			logger.info("Active login changed to {}", activeLogin);
			this.activeLogin = activeLogin;
			setConnectionSettings();
			if (projectsManager != null) {
				projectsManager.userChanged();
			}
		}

	}

	private void setConnectionSettings() {
		if (this.isConnected()) {
			this.close();
		}
		if(activeLogin == null) {
			this.close();
			return;
		}
		if(this.login()) {



		}

	}

	private boolean login() {
		if(activeLogin!= null)
			return login(activeLogin);
		return false;
	}

	private boolean login(UserLoginInfo activeLogin) {
		try {

			String hostname = activeLogin.getHostInfo().getServerURL();
			String username = activeLogin.getUserName();
			String password = activeLogin.getPassword();
			String scheme = activeLogin.getHostInfo().getScheme();
			if (hostname.isEmpty() || username.isEmpty() || password.isEmpty()) {
				this.close();
				return false;
			}
			int connPort = 22; // Default ssh
			switch (scheme) {
			case IFTPManager.SCHEME_FTP:
				connPort = activeLogin.getHostInfo().getFtpPort();
				break;
			case IFTPManager.SCHEME_SFTP:
				connPort = activeLogin.getHostInfo().getSshPort();
				break;
			case IFTPManager.SCHEME_FTPS:
				connPort = activeLogin.getHostInfo().getFtpPort();
				break;
			default:
				scheme = IFTPManager.SCHEME_SFTP;
				connPort = activeLogin.getHostInfo().getSshPort();
				break;
			}
			String userHome = activeLogin.getHostInfo().getUserHomePath();
			SftpFileSystemConfigBuilder.getInstance().setUserDirIsRoot(SFTPopts, userHome.isEmpty());

			FileSystemOptions opts;
			if(IFTPManager.SCHEME_SFTP.equals(scheme)) {
				opts = SFTPopts;
			}else {
				opts = FTPopts;
			}
			// TODO :: find alternative way to set password instead of URI format
			URI uri = new URI(scheme,
					username+":"+password,hostname,connPort,userHome,null,null);

			// fileSystemManager and updateManager is two differnt manager one handling transfers and the other for resolving files
			remoteCWD = fileSystemManager.resolveFile(uri.toString(), opts);
			appRoot = remoteCWD.resolveFile("NETOOLS");
			if (!appRoot.exists()) {
				appRoot.createFolder();
				// System.out.println("Created 'app_root' directory in " + remoteCWD.getName().getPath());
				logger.info("Created 'NETOOLS' directory in {}", remoteCWD.getName().getPath());
			}
			projectsFolder = appRoot.resolveFile(PROJECTS_DIRECTORY);
			if (!projectsFolder.exists()) {
				projectsFolder.createFolder();
				// System.out.println("Created 'app_root' directory in " + remoteCWD.getName().getPath());
				logger.info("Created 'projects' directory in {}", appRoot.getName().getPath());
			}
		} catch (FileSystemException | URISyntaxException e) {
			e.printStackTrace();
			return false;
		}
		this.activeLogin = activeLogin;
		return true;
	}

	private boolean close() {
		if (remoteCWD != null) {
			try {
				remoteCWD.close();

				VFS.reset();
				this.fileSystemManager = VFS.getManager();


			} catch (FileSystemException e) {
				e.printStackTrace();
			} finally {
				remoteCWD = null;
				//				if(eventsBroker!= null) {
				//					eventsBroker.post(FTPEvents.FTP_CONN_CLOSED, null);
				//				}
			}

		}

		return true;

	}

	boolean isConnected() {
		return remoteCWD != null;

	}

	/**
	 * Create project folder and save project info
	 * @param activeProject
	 */
	public void createProject(Project activeProject) {
		String projectName = activeProject.getProjectName();
		String projectDescription = activeProject.getProjectDescription();
		String projectID = activeProject.getProjectID();
		// create project folder
		FileObject projectFolder;
		try {
			projectFolder = projectsFolder.resolveFile(projectID);
			if (!projectFolder.exists()) {
				projectFolder.createFolder();
				// System.out.println("Created 'app_root' directory in " + remoteCWD.getName().getPath());
				logger.info("Created 'projects' directory in {}", projectsFolder.getName().getPath());
			}
			// create tasks folder
			FileObject tasksFolder = projectFolder.resolveFile(TASKS_DIRECTORY);
			if (!tasksFolder.exists()) {
				tasksFolder.createFolder();
				// System.out.println("Created 'app_root' directory in " +
				// remoteCWD.getName().getPath());
				logger.info("Created 'tasks' directory in {}", projectFolder.getName().getPath());
			}
			// create networks folder
			FileObject networksFolder = projectFolder.resolveFile(NETWORKS_DIRECTORY);
			if (!networksFolder.exists()) {
				networksFolder.createFolder();
				// System.out.println("Created 'app_root' directory in " +
				// remoteCWD.getName().getPath());
				logger.info("Created 'networks' directory in {}", projectFolder.getName().getPath());
			}
			saveProject(activeProject);
			// FileObject projectInfo = projectFolder.resolveFile("project.info");
			// serialize project info
			// Project.serializeProject(activeProject, projectInfo.getContent().getOutputStream());
			//			// create datasets folder
			//			FileObject datasetsFolder = projectFolder.resolveFile("datasets");
			//			if (!datasetsFolder.exists()) {
			//				datasetsFolder.createFolder();
			//				// System.out.println("Created 'app_root' directory in " +
			//				// remoteCWD.getName().getPath());
			//				logger.info("Created 'datasets' directory in {}", projectFolder.getName().getPath());
			//			}


		} catch (FileSystemException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}


	}

	public void saveProject(Project project) {
		IProgressMonitor monitor = new NullProgressMonitor();
		saveProject(project, monitor);

	}
	
	/**
	 * Save project strucure into a .project.json
	 * @param project
	 * @param monitor
	 */
	public void saveProject(Project project, IProgressMonitor monitor) {
		String projectID = project.getProjectID();
		// create project folder
		boolean backups = false;
		boolean error = false;
		FileObject projectFolder;
		try {
			projectFolder = projectsFolder.resolveFile(projectID);
			if (!projectFolder.exists()) {
				projectFolder.createFolder();
				// System.out.println("Created 'app_root' directory in " + remoteCWD.getName().getPath());
				logger.info("Created {} directory in {}", PROJECTS_DIRECTORY,   projectsFolder.getName().getPath());
			}
			FileObject projectInfo = projectFolder.resolveFile(".project.json");
			// create a backup of the project.info file if it exists
			if (projectInfo.exists()) { 
				FileObject projectInfoBackup = projectFolder.resolveFile(".project.json.bak");
				if (projectInfoBackup.exists()) {
					projectInfoBackup.delete();
				}
				projectInfoBackup.copyFrom(projectInfo, Selectors.SELECT_ALL);
			    backups = true;
			}
			
				
			
			// Serialize to JSON
			ObjectMapper mapper = JsonMapper.builder().configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false).build();
			// mapper ignore fields in json that are not the model ignoreUnknown
			
			// mapper.setVisibility(PropertyAccessor.FIELD, Visibility.ANY);
			mapper.writeValue(projectInfo.getContent().getOutputStream(), project);

		} catch (FileSystemException e) {
			// TODO Auto-generated catch block
			error = true;
			e.printStackTrace();
		} catch (StreamWriteException e) {
			// TODO Auto-generated catch block
			error = true;
			e.printStackTrace();
		} catch (DatabindException e) {
			// TODO Auto-generated catch block
			error = true;
			e.printStackTrace();
		} catch (IOException e) {
			// TODO Auto-generated catch block
			error = true;
			e.printStackTrace();
		}
		if (error) {
            logger.error("Error saving project {}", project.getProjectName());
            // restore backup
            if (backups) {
            	try {
    				projectFolder = projectsFolder.resolveFile(projectID);
    				FileObject projectInfoBackup = projectFolder.resolveFile(".project.json.bak");
    				FileObject projectInfo = projectFolder.resolveFile(".project.json");
    				// delete the project.info file if  it exists
    				if (projectInfo.exists()) {
    					projectInfo.delete();
    				}
    				
    				if (projectInfoBackup.exists()) {
    					projectInfo .copyFrom(projectInfoBackup, Selectors.SELECT_ALL);
    				}
    				throw new RuntimeException("Error saving project " + project.getProjectName());
    			} catch (FileSystemException e) {
    				// TODO Auto-generated catch block
    				e.printStackTrace();
    				// error here is unrecoverable
    				throw new RuntimeException("Error saving project " + project.getProjectName(), e);
    			}
            } else {
            	throw new RuntimeException("Error saving project " + project.getProjectName());
            }
            
       
		}
		
	}
	
	/**
	 * Save dataset: write data file and variables file
	 * @param dataset
	 * @param monitor
	 */
	public void saveDataset(Dataset dataset, IProgressMonitor monitor) {
		// get dataset folder
		Project project = dataset.getProject();
		String datasetID = dataset.getDatasetID();
		FileObject datasetFolder;
		try {
			datasetFolder = projectsFolder
					.resolveFile(project.getProjectID() + "/" + DATASETS_DIRECTORY + "/" + datasetID);
			if (!datasetFolder.exists()) {
				datasetFolder.createFolder();
				// System.out.println("Created 'app_root' directory in " +
				// remoteCWD.getName().getPath());
				logger.info("Created 'datasets' directory in {}", projectsFolder.getName().getPath());
			}
			// get last dtfile name and save at with the dataset name 
			FileObject datasetFile = datasetFolder.resolveFile(dataset.getDataFile());
			FileObject sourceFile = datasetFolder.resolveFile(dataset.getLastDataFile());
			if (sourceFile == null) {
				throw new RuntimeException("Data file not found");
			}
			if (!sourceFile.exists()) {
				throw new RuntimeException("Data file not found");
			}
			datasetFile.copyFrom(sourceFile, Selectors.SELECT_ALL);
			// save variables 
			FileObject variablesFile = datasetFolder.resolveFile(dataset.getVariablesFile());
			// get writestremm and let dataset write to it
			dataset.writeVariables(variablesFile.getContent().getOutputStream());
			// close the stream
			variablesFile.getContent().close();
			
		} catch (FileSystemException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		
	}
	
	public void saveNetworkDesign(NetworkDesign newtworkDesign, IProgressMonitor monitor) {
		// save network design to remote storage
		Project project = newtworkDesign.getProject();
		String designID = newtworkDesign.getDesignID();
		FileObject designFolder;
		try {
			designFolder = projectsFolder
					.resolveFile(project.getProjectID() + "/" + DESIGNS_DIRECTORY + "/" + designID);
			if (!designFolder.exists()) {
				designFolder.createFolder();
				// System.out.println("Created 'app_root' directory in " +
				// remoteCWD.getName().getPath());
				logger.info("Created 'designs' directory in {}", projectsFolder.getName().getPath());
			}
			// write whilelist to whitelist.csv
			FileObject whitelistFile = designFolder.resolveFile("whitelist.csv");
			newtworkDesign.writeWhitelist(whitelistFile.getContent().getOutputStream());
			// close the stream
			whitelistFile.getContent().close();
			
			// write blacklist to blacklist.csv
			FileObject blacklistFile = designFolder.resolveFile("blacklist.csv");
			newtworkDesign.writeBlacklist(blacklistFile.getContent().getOutputStream());
			// close the stream
			blacklistFile.getContent().close();

		} catch (FileSystemException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
	}
	

	/**
	 * Load project from remote storage, if project does not exist return null
	 * 
	 * @param projectID
	 * @return
	 */
	public Project loadProject(String projectID) {
		FileObject projectFolder;
		try {
			projectFolder = projectsFolder.resolveFile(projectID);
			if (!projectFolder.exists()) {
				return null;
			}
			FileObject projectInfo = projectFolder.resolveFile(".project.json");
			// Serialize to JSON
			ObjectMapper mapper = JsonMapper.builder().configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false).build();;
			// mapper.setVisibility(PropertyAccessor.FIELD, Visibility.ANY);
			Project project = mapper.readValue(projectInfo.getContent().getInputStream(), Project.class);
			return project;
		} catch (FileSystemException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		return null;

	}

	public void loadProjects(List<Project> projects) {
		// get all folders in projects folder
		FileObject[] projectFolders;
		try {
			projectFolders = projectsFolder.getChildren();
			for (FileObject projectFolder : projectFolders) {
				try {
					// only load folders that have project.info file
					// check first if folder is a directory
					if (!projectFolder.isFolder()) {
						continue;
					}
					logger.info("Resolving project file {}", projectFolder.getName().getPath());
					FileObject projectInfo = projectFolder.resolveFile("project.info");
					logger.info("Loading project {}", projectFolder.getName().getPath());
					Project project = Project.deserializeProject(projectInfo.getContent().getInputStream());
					projects.add(project);
				} catch (Exception e) {
					// TODO Auto-generated catch block
					logger.error("Error loading project {}", projectFolder.getName().getPath());
					e.printStackTrace();
				}

			}
			// for testing only
//			if (projects.isEmpty()) {
//				// load test projects
//				projects.addAll(testProjects());
//			}


		} catch (FileSystemException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}

	}

	/**
	 * Load projectsMap from remote storage , mapping projectID to project name
	 */
	public List<ProjectEntry> loadProjects() {
		// projectInfo file is at projectsFolder/.projects.info
		FileObject projectsInfo;
		try {
			projectsInfo = projectsFolder.resolveFile(".projects.info");
			if (!projectsInfo.exists()) {
				logger.info("No projects info file found");
				
				// return null;
			}
			try(ObjectInputStream in = new ObjectInputStream(projectsInfo.getContent().getInputStream())) {
				List<ProjectEntry> projectEntries = (List<ProjectEntry>) in.readObject();

				return projectEntries;
			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			} catch (ClassNotFoundException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}

		} catch (FileSystemException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		return null;

	}

	public void saveProjects(List<ProjectEntry> projectEntries) {
		// TODO Auto-generated method stub
		// save list to remote storage in projects.info
		FileObject projectsInfo;
		try {
			logger.info("Saving projects list on remote storage");
			projectsInfo = projectsFolder.resolveFile(".projects.info");
			try(ObjectOutputStream out = new ObjectOutputStream(projectsInfo.getContent().getOutputStream())){
				out.writeObject(projectEntries);
			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}


			//			logger.info("Readding projects");
			//			try(ObjectInputStream in = new ObjectInputStream(projectsInfo.getContent().getInputStream())) {
			//				List<Project> projects2 = (List<Project>) in.readObject();
			//			} catch (IOException e) {
			//				// TODO Auto-generated catch block
			//				e.printStackTrace();
			//			} catch (ClassNotFoundException e) {
			//				// TODO Auto-generated catch block
			//				e.printStackTrace();
			//			}
			//			logger.info("Readding projects done");
		} catch (FileSystemException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}

	}

	public String uploadDataset(File csvFile, String projectID, String datasetName, String fileName,
			IProgressMonitor monitor) {

		// set the progress monitor to upload the file

		monitor.subTask("Uploading file " + datasetName);
		// create dataset folder if not exists
		FileObject projectFolder;
		try {
			projectFolder = projectsFolder.resolveFile(projectID);
			if (!projectFolder.exists()) {
				projectFolder.createFolder();
				// System.out.println("Created 'app_root' directory in " +
				// remoteCWD.getName().getPath());
				logger.info("Created 'projects' directory in {}", projectsFolder.getName().getPath());
			}
			FileObject datasetsFolder = projectFolder.resolveFile(DATASETS_DIRECTORY);
			if (!datasetsFolder.exists()) {
				datasetsFolder.createFolder();
				// System.out.println("Created 'app_root' directory in " +
				// remoteCWD.getName().getPath());
				logger.info("Created {} directory in {}",DATASETS_DIRECTORY, projectFolder.getName().getPath());
			}
			FileObject datasetFolder = datasetsFolder.resolveFile(datasetName);
			if (!datasetFolder.exists()) {
				datasetFolder.createFolder();
				// System.out.println("Created 'app_root' directory in " +
				// remoteCWD.getName().getPath());
				logger.info("Created 'datasets' directory in {}", datasetsFolder.getName().getPath());
			}
			FileObject datasetFile = datasetFolder.resolveFile(fileName);
			datasetFile.copyFrom(fileSystemManager.resolveFile(csvFile.getAbsolutePath()), Selectors.SELECT_ALL);
			return datasetFile.getName().getPath();

		} catch (FileSystemException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		return null;

	}

	public String uploadDataset(File csvFile, String projectID,
			IProgressMonitor monitor) {

		// set the progress monitor to upload the file
		String datasetName = csvFile.getName();
		String fileName = csvFile.getName();
		monitor.subTask("Uploading file " + datasetName);
		// create dataset folder if not exists
		FileObject projectFolder;
		try {
			projectFolder = projectsFolder.resolveFile(projectID);
			if (!projectFolder.exists()) {
				projectFolder.createFolder();
				// System.out.println("Created 'app_root' directory in " +
				// remoteCWD.getName().getPath());
				logger.info("Created 'projects' directory in {}", projectsFolder.getName().getPath());
			}
			FileObject uploadsFolder = projectFolder.resolveFile("uploads");
			if (!uploadsFolder.exists()) {
				uploadsFolder.createFolder();
				// System.out.println("Created 'app_root' directory in " +
				// remoteCWD.getName().getPath());
				logger.info("Created 'uploads' directory in {}", projectFolder.getName().getPath());
			}

			FileObject datasetFile = uploadsFolder.resolveFile(fileName);
			datasetFile.copyFrom(fileSystemManager.resolveFile(csvFile.getAbsolutePath()), Selectors.SELECT_ALL);
			return datasetFile.getPath().toString();

		} catch (FileSystemException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		return null;

	}




//	private List<Project> testProjects() {
//		List<Project> projects = new ArrayList<Project>();
//		HashMap<String, String> projectMaps = new HashMap<String, String>();
//		Project project = new Project("Test Project", "Test Project Description");
//
//		createProject(project);
//		projects.add(project);
//		saveProject(project	);
//		projectMaps.put(project.getProjectID(), project.getProjectName());
//
//		project = new Project("Test Project 2", "Test Project Description 2");
//		createProject(project);
//		projects.add(project);
//		saveProject(project	);
//		projectMaps.put(project.getProjectID(), project.getProjectName());
//
//		saveProjects(projectMaps);
//
//		return projects;
//
//
//	}

	public String getProjectsBaseDirectory() {
		if (projectsFolder == null) {
			throw new IllegalStateException("Projects directory not set");
		}
		return projectsFolder.getName().getPath();
	}
	public String getAppBaseDirectory() {
		if (appRoot == null) {
			throw new IllegalStateException("App root directory not set");
		}
		return appRoot.getName().getPath();
	}

	IProjectsManager  projectsManager;

	public void setProjectsManager(IProjectsManager projectsManagerImpl) {
		this.projectsManager = projectsManagerImpl;

	}

	/**
	 * Validate file exists on remote storage
	 * @param remoteFilePath
	 * @return true if file exists, false otherwise
	 */
	public boolean validateFile(String remoteFilePath) {
		FileObject file;
		try {
			file = remoteCWD.resolveFile(remoteFilePath);
			return file.exists();
		} catch (FileSystemException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		return false;
	}

	public String resolveFile(String remoteFilePath) {
		FileObject file;
		try {
			file = remoteCWD.resolveFile(remoteFilePath);
			return file.getName().getPath();
		} catch (FileSystemException e) {
			throw new RuntimeException("Error resolving file " + remoteFilePath, e);
		}

	}
	
	/**
	 * Read variable data from a csv file
	 * @param fileName name of the file to read:  variablesFile is csv file with two columns : "Variable","DataType","Distribution","VariableType","HasMissingData"
	 * @return List of variables
	 */
	public List<Variable> readVariableData(String fileName) {
		FileObject file;
		List<Variable> variables = new ArrayList<>();
		try {
			file = remoteCWD.resolveFile(fileName);
			if (file.exists()) {
				try (Reader reader = new InputStreamReader(file.getContent().getInputStream());
					 CSVParser csvParser = new CSVParser(reader, CSVFormat.DEFAULT.withDelimiter(',').withFirstRecordAsHeader())) {
					csvParser.forEach(record -> {
						Variable variable = new Variable();
						variable.setName(record.get("Variable"));
						variable.setDataType(DatasetType.valueOf(record.get("DataType").toUpperCase()));
						variable.setDistribution(Distribution.valueOf(record.get("Distribution").toUpperCase()));
						variable.setVariableType(VariableType.valueOf(record.get("VariableType").toUpperCase()));
						variable.setHasMissingData(Boolean.parseBoolean(record.get("HasMissingData")));
						variables.add(variable);
					});
				}
			} else {
				logger.error("File {} does not exist", fileName);
			}
		} catch (FileSystemException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
		return variables;
	}
	
	
	public Reader openForReading(String fileName) {
		FileObject file;
		try {
			file = remoteCWD.resolveFile(fileName);
			if (file.exists()) {
				return new InputStreamReader(file.getContent().getInputStream());
			} else {
				logger.error("File {} does not exist", fileName);
			}
		} catch (FileSystemException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
		return null;
	}
	
	public OutputStream openForWriting(String fileName) {
		FileObject file;
		try {
			file = remoteCWD.resolveFile(fileName);
			
			return file.getContent().getOutputStream(); 
			
		} catch (FileSystemException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
		return null;
	}

	
	public void deleteProject(Project project) {
		String projectID = project.getProjectID();
        deleteProject(projectID);
    
		
	}

	public void deleteProject(String projectID) {
		FileObject projectFolder;
        try {
            projectFolder = projectsFolder.resolveFile(projectID);
            projectFolder.delete(Selectors.SELECT_ALL);
        } catch (FileSystemException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
            throw new RuntimeException("Error deleting project " + projectID, e);
        }
       
 	
	}

	
	
	/**
	 * Delete dataset from remote storage
	 * @param selectedDataset
	 * @param monitor
	 */
	public void deleteDataset(Dataset selectedDataset, IProgressMonitor monitor) {
		Project project = selectedDataset.getProject();
		String datasetID = selectedDataset.getDatasetID();
		// remove dataset from project
		// get dataset folder
		FileObject datasetFolder;
		try {
			datasetFolder = projectsFolder
					.resolveFile(project.getProjectID() + "/" + DATASETS_DIRECTORY + "/" + datasetID);
			datasetFolder.delete(Selectors.SELECT_ALL);
		} catch (FileSystemException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		
	}

	public String createNetworkDirectory(NetworkBuild networkBuild, IProgressMonitor monitor) {
		return createNetworkDirectory(networkBuild.getProject(), networkBuild.getBuildID(), monitor);
	}
	public String createNetworkDirectory(Project project,String networkID, IProgressMonitor monitor) {
		
		// create network directory
		FileObject projectFolder;
		try {
            projectFolder = projectsFolder.resolveFile(project.getProjectID());
            if (!projectFolder.exists()) {
                projectFolder.createFolder();
                // System.out.println("Created 'app_root' directory in " + remoteCWD.getName().getPath());
                logger.info("Created 'projects' directory in {}", projectsFolder.getName().getPath());
            }
            FileObject networksFolder = projectFolder.resolveFile(NETWORKS_DIRECTORY);
            if (!networksFolder.exists()) {
                networksFolder.createFolder();
                // System.out.println("Created 'app_root' directory in " +
                // remoteCWD.getName().getPath());
                logger.info("Created 'networks' directory in {}", projectFolder.getName().getPath());
            }
            FileObject networkFolder = networksFolder.resolveFile(networkID);
            if (!networkFolder.exists()) {
                networkFolder.createFolder();
                // System.out.println("Created 'app_root' directory in " +
                // remoteCWD.getName().getPath());
                logger.info("Created 'network' directory in {}", networksFolder.getName().getPath());
            }
           
            return networkFolder.getName().getPath();
            
        } catch (FileSystemException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } 
		return null;
	}

	public void deleteDirectory(String networkDirectory) {
		FileObject networkFolder;
        try {
            networkFolder = remoteCWD.resolveFile(networkDirectory);
			if (!networkFolder.exists()) {
				logger.error("deleteDirectory : Network directory {} does not exist", networkDirectory);
				return;
			}
            networkFolder.delete(Selectors.SELECT_ALL);
        } catch (FileSystemException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
            logger.error("Error deleting network directory {}", networkDirectory);
        }
        
    
		
	}
	
	public void copyFile(String sourceNetworkFile, String targetNetworkFile)
	{
		copyFile(sourceNetworkFile, targetNetworkFile, new NullProgressMonitor());
	}

	public void copyFile(String sourceNetworkFile, String targetNetworkFile, IProgressMonitor monitor) {
		FileObject sourceFile;
        FileObject targetFile;
        try {
            sourceFile = remoteCWD.resolveFile(sourceNetworkFile);
            targetFile = remoteCWD.resolveFile(targetNetworkFile);
            targetFile.copyFrom(sourceFile, Selectors.SELECT_ALL);
        } catch (FileSystemException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
        
    
		
	}

	


	
	

	

}
