package com.biotechvana.netools.projects;

import jakarta.annotation.PostConstruct;
import jakarta.inject.Inject;

import java.io.File;
import java.lang.Thread.State;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.SubMonitor;
import org.eclipse.e4.core.di.annotations.Optional;
import org.eclipse.e4.core.services.events.IEventBroker;
import org.eclipse.e4.ui.di.UISynchronize;
import org.eclipse.e4.ui.model.application.MApplication;
import org.eclipse.e4.ui.progress.IProgressService;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.biotechvana.netools.jobs.NetworkJob;
import com.biotechvana.netools.projects.jobs.NetworkBuildJob;
import com.biotechvana.netools.projects.jobs.PostUpdateJob;
import com.biotechvana.workflow.IValidationResult;
import com.biotechvana.workflow.SubmissionHistoryManager;
import com.biotechvana.workflow.WorkflowJob;
import com.biotechvana.workflow.ext.SubmissionHistoryUpdator;
import com.biotechvana.workflow.manager.IWorkflowManager;
import com.biotechvana.workflow.submissionhistory.SubmissionHistoryEntry;
import com.biotechvana.workflow.tracking.ExecStatus;

public class ProjectsManagerImpl implements IProjectsManager {
	private static final Logger logger = LoggerFactory.getLogger(ProjectsManagerImpl.class);

	@Inject
	RemoteStorage remoteStorage;

	@Inject
	IProgressService progressService;

	IStatusLineManager manager;

	@Inject
	void setManager(@Optional IStatusLineManager manager) {
		this.manager = manager;
	}

	//	@Inject
	//	void setProgressService(@Optional IProgressService progressService) {
	//		this.progressService = progressService;
	//	}

	List<Project> projects = new ArrayList<Project>();
	/**
	 * map projectID to projectName
	 */
	// Map<String, String> projectsIDMap = new HashMap<String,String>();
	List<ProjectEntry> projectEntries = new ArrayList<ProjectEntry>();

	/**
	 * map projectID to project 
	 */
	HashMap<String, Project> projectsMap = new HashMap<String,Project>();

	@Inject
	IWorkflowManager workflowManager;

	@Inject
	private IEventBroker eventsBroker;

	@Inject
	MApplication app;


	SubmissionHistoryManager submissionHistoryManager;

	@Inject
	void setSubmissionHistoryManager(@Optional SubmissionHistoryManager submissionHistoryManager) {
		this.submissionHistoryManager = submissionHistoryManager;

	}

	@Inject UISynchronize uiSync;
	/**
	 * 	 The active project is the project that is currently selected in the UI
	 */
	Project activeProject = null;

	// tracking of active project
	HashMap<String, DTFile> processingTasks = new HashMap<String,DTFile>();
	HashMap<String, NetworkBuild> networkTasks = new HashMap<String,NetworkBuild>();

	//	public IEventBroker getEventBroker ()
	//	{
	//		return eventsBroker;
	//	
	//	}

	//	@Inject @Optional
	//    public  void setEventBroker(IEventBroker eventsBroker) {
	//    	this.eventsBroker = eventsBroker;
	//    }

	@Override
	public boolean canCreateProject() {
		// TODO Auto-generated method stub
		return remoteStorage.isConnected();
	}


	/** 
	 * Save the projects and manager status to the remote storage
	 */
	@Override
	public void saveProjects() throws Exception {

		remoteStorage.saveProjects(projectEntries);
	}

	@Override
	public void loadProjects() throws Exception {
		// clear previous projects if any
		if (!projects.isEmpty()) {
			projects.clear();
		}
		projectEntries.clear();


		// populate the projects list from the remote storage
		projectEntries.addAll(remoteStorage.loadProjects());
		// remoteStorage.loadProjects(projects);
		// for the initial project loading, we need to set the manager 
		// for each project
		// just to make sure that the manager is set
		for (ProjectEntry projectEntry : projectEntries) {
			logger.info("Loading project {}", projectEntry.getProjectName());
			Project project = remoteStorage.loadProject(projectEntry.getProjectID());
			if (project != null) {
				logger.info("Loaded project {}", project.getProjectName());
				projects.add(project);
				projectsMap.put(projectEntry.getProjectID(), project);
			}

		}

		for (Project p : projects) {
			p.setProjectsManager(this);
		}

		// testing networks 
		//		Project testProject = projects.get(0);
		//		Network.createNetwork(testProject, "Network1", "network1");
		//		Network.createNetwork(testProject, "Network2", "network2");
		//		saveProject(testProject);


		// setActiveProject(projects.get(0));



	}

	@Override
	public void createProject(Project activeProject) {
		createProject(activeProject, new NullProgressMonitor());
	}
	@Override
	public void deleteProject(Project project) {
		try
		{
			// remove selectedProjectEntry from projectEntries
			String projectID = project.getProjectID();
			// find projectEntry with projectID
			ProjectEntry projectEntry = null;
			for (ProjectEntry entry : projectEntries) {
				if (entry.getProjectID().equals(projectID)) {
					projectEntry = entry;
					break;
				}
			}
			projectEntries.remove(projectEntry);
			remoteStorage.deleteProject(project);
			projects.remove(project);
			projectsMap.remove(project.getProjectID());
			saveProjects();
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
			throw new RuntimeException(e);
		}


	}

	@Override
	public void deleteProjectFromEntry(ProjectEntry selectedProjectEntry) {
		try
		{

			// remove selectedProjectEntry from projectEntries
			String projectID = selectedProjectEntry.getProjectID();
			// find projectEntry with projectID
			ProjectEntry projectEntry = null;
			for (ProjectEntry entry : projectEntries) {
				if (entry.getProjectID().equals(projectID)) {
					projectEntry = entry;
					break;
				}
			}
			projectEntries.remove(projectEntry);

			// delete folder from remote storage

			remoteStorage.deleteProject(selectedProjectEntry.getProjectID());
			// remove the project from the list if it exists
			Project project = projectsMap.get(selectedProjectEntry.getProjectID());
			if (project != null) {
				projects.remove(project);
				projectsMap.remove(project.getProjectID());
			}
			// save projects list
			saveProjects();
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
			throw new RuntimeException(e);
		}
	}


	@Override
	public void createProject(Project newProject, IProgressMonitor monitor) {
		// TODO :: validation first 

		newProject.setProjectsManager(this);
		// create a projectEntry and add it to the projects list
		ProjectEntry projectEntry = new ProjectEntry(newProject.getProjectID(), newProject.getProjectName(), newProject.getProjectDescription(), "");
		projectEntries.add(projectEntry);
		remoteStorage.createProject(newProject);

		projects.add(newProject);
		saveProject(newProject, monitor);
		try {
			saveProjects();
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
			throw new RuntimeException(e);
		}
	}

	@Override
	public List<ProjectEntry> getProjectEntries() {
		List<ProjectEntry> projectEntries = new ArrayList<>();
		projectEntries.addAll(this.projectEntries);
		return projectEntries;
	}

	SubmissionHistoryUpdator  updator;
	@PostConstruct
	public void setup() {
		if (progressService == null) {
			logger.warn("Progress service is null");
		}
		updator = new SubmissionHistoryUpdator();
		// Load the projects from the remote storage
		remoteStorage.setProjectsManager(this);
		init();
		updator.start();
	}



	public void init() {
		if (!remoteStorage.isConnected()) {
			return;
		}


		if (uiSync == null) {
			logger.warn("UISync is null");
			try {
				loadProjects();
				// log list of projects
				logger.info("Projects loaded from remote storage:");
				for(Project p : projects) {
					logger.info("Project: {}", p.getProjectName());
				}
			} catch (Exception e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}

		}
		else {
			uiSync.syncExec(() -> {
				logger.info("UI thread");

				try {

					progressService.run(true, false, 
					//progressService.busyCursorWhile( 

							new IRunnableWithProgress() {

								@Override
								public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
									// TODO Auto-generated method stub
									try {
										loadProjects();
										// log list of projects
										logger.info("Projects loaded from remote storage:");
										for(Project p : projects) {
											logger.info("Project: {}", p.getProjectName());
										}
									} catch (Exception e) {
										// TODO Auto-generated catch block
										e.printStackTrace();
									}
								}
							}

							);
				} catch (InvocationTargetException | InterruptedException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}

			});
		}



	}


	/**
	 * upload raw data table to a dataset, 
	 * @param csvFile : the csv file to be uploaded [local]
	 * @param newDataset : the dataset to which the file will be uploaded
	 * @param monitor : progress monitor
	 * @return the remote path of the uploaded file
	 */
	@Override
	public String uploadDataset(File csvFile, Dataset newDataset, IProgressMonitor monitor) {
		if (monitor == null) {
			// create a new progress monitor
			monitor = new NullProgressMonitor();
		}
		String datasetName = newDataset.getDatasetName();
		Project activeProject = newDataset.getProject();
		String projectID = activeProject.getProjectID();
		String fileName = "00_" + datasetName + ".csv";
		SubMonitor subMonitor  = SubMonitor.convert(monitor, IProgressMonitor.UNKNOWN);
		subMonitor.setTaskName("Uploading file");
		String remotePath = remoteStorage.uploadDataset(csvFile , projectID, datasetName , fileName, subMonitor);
		subMonitor.worked(1);

		return remotePath;
	}

	@Override
	public boolean uploadDataset(Dataset newDataset, File csvFile, IProgressMonitor monitor) {
		return uploadDataset(newDataset, csvFile, ",", null, monitor);
	}


	@Override
	public boolean uploadDataset(Dataset newDataset, File csvFile, String csvSeparator, String sampleNameColumn,
			IProgressMonitor monitor) {
		if (monitor == null) {
			// create a new progress monitor
			monitor = new NullProgressMonitor();
		}
		Project activeProject = newDataset.getProject();
		String datasetName = newDataset.getDatasetName();

		// convert to SubMonitor and set total number of work units
		SubMonitor subMonitor = SubMonitor.convert(monitor,2);

		SubMonitor subMonitor1  = SubMonitor.convert(subMonitor, IProgressMonitor.UNKNOWN);

		subMonitor1.setTaskName("Creating dataset " + datasetName);

		String projectName = activeProject.getProjectName();
		String projectID = activeProject.getProjectID();
		// append 00_ to datasetName 
		String fileName = "00_" + datasetName + ".csv";
		// upload the csv file to the remote storage
		String remotePath = remoteStorage.uploadDataset(csvFile , projectID, datasetName , fileName, monitor);
		logger.debug("Remote path: {}", remotePath);
		subMonitor1.worked(1);

		subMonitor1  = SubMonitor.convert(subMonitor, IProgressMonitor.UNKNOWN);
		subMonitor1.setTaskName("Submitting to workflow");
		RawDTFile rawDataFile =  new RawDTFile();
		newDataset.addDTFile(rawDataFile);
		WorkflowJob job = rawDataFile.getWorkflowJob();
		job.setWorkflowManager(workflowManager);
		IValidationResult validationResult = workflowManager.runWorkflowJob(job);
		if (validationResult.isOK()) {
			processingTasks.put(rawDataFile.getLastJobId(), rawDataFile);

		}
		else {
			logger.error("Failed to submit to workflow: {}", validationResult.getMessage());

		}
		boolean result = validationResult.isOK();// rawDataFile.sumbitToWorkflow(workflowManager);


		subMonitor1.worked(1);
		// assuming all validations are done, we can now save the dataset to the
		// database
		//activeProject.addDataset(newDataset);
		monitor.done();
		return result;
	}

	@Override
	public Project getActiveProject() {
		//		// for testing
		//		if (activeProject == null) {
		//			setActiveProject(projects.get(0));
		//		}

		return activeProject; 
	}


	@Override
	public void setSelectedDataset(Dataset selectedDataset) {
		logger.debug("Selected dataset: {}", selectedDataset.getDatasetName());
		app.getContext().set(IProjectsManager.SELECTED_DATASET, selectedDataset);
	}


	public void setActiveProjectFromEntry(ProjectEntry selectedProject) {
		String projectID = selectedProject.getProjectID();
		Project project = projectsMap.get(projectID);
		if (project == null) {
			logger.info("Loading project {}", projectID);
			project = remoteStorage.loadProject(projectID);
			if (project == null) {
				throw new RuntimeException("Failed to load project: " + projectID);
			}
			project.setProjectsManager(this);
			logger.info("Loaded project {}", project.getProjectName());
			projects.add(project);
			projectsMap.put(projectID, project);

		}
		setActiveProject(project);
	}

	public void setActiveProject(Project project) {

		// notify the event broker that the active project is going to change
		if(activeProject != null) {
			logger.debug("Active project is changing from {}", activeProject.getProjectName());
			eventsBroker.post(ProjectEvents.ACTIVE_PROJECT_CHANGING, activeProject);
		}


		this.activeProject = project;
		logger.debug("Active project is now {}", project == null ? "null" : project.getProjectName());
		// it is possible that the active project is null
		eventsBroker.post(ProjectEvents.ACTIVE_PROJECT_CHANGED, activeProject);
		app.getContext().set(IProjectsManager.ACTIVE_PROJECT, activeProject);
		app.getContext().set(IProjectsManager.ACTIVE_DESIGN, null);

		// clear the processing tasks
		processingTasks.clear();
		if (activeProject != null) {
			// need to collect the processing tasks from previous sessions
			for (DTFile dtFile : activeProject.getProcessingTasks()) {
				if (dtFile.getExecStatus() != ExecStatus.Finished) {
					processingTasks.put(dtFile.getLastJobId(), dtFile);
				}
				// processingTasks.put(dtFile.getLastJobId(), dtFile);
			}

		}
		networkTasks.clear();
		if (activeProject != null) {
			// need to collect the processing tasks from previous sessions
			for (NetworkBuild networkBuild : activeProject.getNetworkBuilds()) {
				if(networkBuild.getBuildStatus() != ExecStatus.Finished && networkBuild.getLastJobId() != null) // null mean it is not submitted yet
					networkTasks.put(networkBuild.getLastJobId(), networkBuild);
			}

		}

	}

	// EVENTS part
	@Override
	public void datasetAddedToProject(Dataset dataset) {
		// notify the event broker
		logger.debug("Dataset added to project: {}", dataset.getDatasetName());
		eventsBroker.post(ProjectEvents.DATASET_ADDED, dataset);
	}

	@Override
	public void projectNameChanged(Project project) {

		logger.debug("Project name changed: {}", project.getProjectName());
		eventsBroker.post(ProjectEvents.PROJECT_NAME_CHANGED, project);
	}

	@Override
	public void projectDescriptionChanged(Project project) {
		// TODO Auto-generated method stub
		logger.debug("Project description changed: {}", project.getProjectDescription());
		eventsBroker.post(ProjectEvents.PROJECT_DESCRIPTION_CHANGED, project);

	}




	@Override
	public void dtFileAddedToDataset(DTFile file) {
		logger.debug("DTFile added to dataset: {}", file.getFileName());
		eventsBroker.post(ProjectEvents.DTFILE_ADDED, file);

	}
	@Override
	public void dtFileUpdated(DTFile dtFile) {
		logger.debug("DTFile updated: {}", dtFile.getFileName());
		eventsBroker.post(ProjectEvents.DTFILE_UPDATED, dtFile);
		eventsBroker.post(ProjectEvents.DATASET_UPDATED, dtFile.getDataset());

	}

	@Override
	public String getBaseFolder() {
		// TODO Auto-generated method stub
		return remoteStorage.getProjectsBaseDirectory();
	}

	@Override
	public void saveProject(Project activeProject) {
		saveProject(activeProject, new NullProgressMonitor());

	}

	/**
	 * save project structure to remote storage and save datasets by default
	 */
	@Override
	public void saveProject(Project activeProject,   IProgressMonitor monitor) {
		saveProject(activeProject, true, monitor);
	}

	/**
	 * save project structure to remote storage and save datasets if saveDatasets is true
	 */
	@Override
	public void saveProject(Project activeProject, boolean saveDatasets,   IProgressMonitor monitor) {
		remoteStorage.saveProject(activeProject, monitor);
		activeProject.getProjectManager().getActiveDesign();
		// save project structure to remote storage
		// remoteStorage.saveProject(activeProject);
		// for each dataset save the dataset
		if (saveDatasets) {
			for (Dataset dataset : activeProject.getDatasets()) {
				if (dataset.getDTFiles().size() > 0) {
					remoteStorage.saveDataset(dataset, monitor);
				}
			}
		}
		for (NetworkDesign newtworkDesign : activeProject.getNetworkDesigns()) {
			remoteStorage.saveNetworkDesign(newtworkDesign, monitor);
		}

	}


	@Override
	public void userChanging() {
		logger.debug("User is changing");
		// TODO :: need to check if we need to save before we change user
	}


	@Override
	public void userChanged() {
		init();

	}


	class SubmissionHistoryUpdator {
		private final Logger logger = LoggerFactory.getLogger(SubmissionHistoryUpdator.class);
		public static final int HISTORY_REFRESH_INTERVAL_MILLIS = 30 * 1000; // 30 seconds

		Thread updateThread;
		int updateInterval = HISTORY_REFRESH_INTERVAL_MILLIS; 
		Boolean toRun = true;
		boolean initialized = false;

		public SubmissionHistoryUpdator()
		{
			updateThread = getUpdatorThread();
			updateThread.setDaemon(true);
		}
		IProgressMonitor monitor =  null;
		private Thread getUpdatorThread() {
			// TODO Auto-generated method stub
			synchronized (toRun) {
				toRun = true;
			}
			return new Thread(new Runnable() {

				@Override
				public void run() {
					// TODO Auto-generated method stub
					while(toRun) {


						// UI Notify progress pre Update

						// Login update

						// UI notify post update
						try {
							//logger.debug("Updating Submission History");
							if (submissionHistoryManager != null && ( processingTasks.size() > 0 || networkTasks.size() > 0  ))
							{
								synchronized (submissionHistoryManager) {

									uiSync.syncExec(new Runnable() {

										@Override
										public void run() {
											if(manager != null) {
												monitor = manager.getProgressMonitor();
												monitor.beginTask("Updating", IProgressMonitor.UNKNOWN);
											}

										}
									});

									update();	


									uiSync.syncExec(new Runnable() { 
										@Override
										public void run() {
											if(monitor != null) {
												monitor.done();
												monitor = null;
											}
										}
									});
								}
							}







						} catch (Exception e) {
							// I was intrubuted during running ?? must terminate 
							System.out.println("Someone wake me up !!");
							e.printStackTrace();


						}

						// cool for some time
						try {
							Thread.sleep(updateInterval);
						} catch (Exception e) {
							// TODO Auto-generated catch block
							e.printStackTrace();
							logger.debug("Someone wake me up !! SubmissionHistoryUpdator");
							if(updateThread.isInterrupted()) {
								System.out.println("Someone wake me up !!");
							}
						}
					}
					updateThread = null;

				}
			});
		}


		public void initalize() {
			if(!initialized) {
				submissionHistoryManager.loadCurrentUserHistory(false, true); 
				initialized = true;
			}
		}


		public void update() {
			initalize();
			// logic to update history and project tasks
			logger.debug("Updating submission history");
			submissionHistoryManager.sync();
			logger.debug("Submission History Updated");
			logger.debug("Updating projects tasks");

			// we need to keep track of processingTasks and see which one is finished or which one has errors
			Iterator<Map.Entry<String, DTFile>> taskIterator = processingTasks.entrySet().iterator();
			while (taskIterator.hasNext()) {
				Map.Entry<String, DTFile> entry = taskIterator.next();
				String jobId = entry.getKey();
				DTFile dtFile = entry.getValue();
				SubmissionHistoryEntry trackHistory = submissionHistoryManager.getEntry(jobId);
				logger.debug("Found Entry {} ", trackHistory); // trackHistory.isSucceed();
				// TODO :: need to expand the logic here for updating and tracking tasks
				// associated with the project
				if (trackHistory.isNotRunning()) {
					trackHistory.getStatus();
					PostUpdateJob updateJob = new PostUpdateJob(ProjectsManagerImpl.this, dtFile, trackHistory);
					updateJob.setUser(false);
					updateJob.schedule();
					taskIterator.remove();

				}
			}
//			Set<String> keys = processingTasks.keySet();
//			for (String jobId : keys) {
//				DTFile dtFile = processingTasks.get(jobId);
//				SubmissionHistoryEntry trackHistory = submissionHistoryManager.getEntry(jobId);
//				logger.debug("Found Entry {} ", trackHistory); // trackHistory.isSucceed();
//				// TODO :: need to expand the logic here for updating and tracking tasks associated with the project
//				if(trackHistory.isNotRunning()) {
//					trackHistory.getStatus();
//					PostUpdateJob updateJob = new PostUpdateJob(ProjectsManagerImpl.this,dtFile,trackHistory) ;
//					updateJob.setUser(false);
//					updateJob.schedule();
//					processingTasks.remove(jobId);
//
//				}
//			}
//			keys = networkTasks.keySet();
//			for (String jobId : keys) {
//				NetworkBuild networkBuild = networkTasks.get(jobId);
//				if (updateNetworkBuild(networkBuild,jobId) )
//				{
//					networkTasks.remove(jobId);
//				}
//			}
			Iterator<Map.Entry<String, NetworkBuild>> iterator = networkTasks.entrySet().iterator();
			while (iterator.hasNext()) {
			    Map.Entry<String, NetworkBuild> entry = iterator.next();
			    String jobId = entry.getKey();
			    NetworkBuild networkBuild = entry.getValue();
			    if (updateNetworkBuild(networkBuild, jobId)) {
			        iterator.remove();
			    }
			}

		}





		public void start() {
			if(updateThread == null) {
				updateThread = getUpdatorThread();
			}
			// TODO Auto-generated method stub
			if(updateThread.getState() == State.NEW) 
			{
				updateThread.start();

			}
			else if (updateThread.getState() == State.TIMED_WAITING) {
				updateThread.interrupt();
			} else  if (updateThread.getState() == State.WAITING ){
				// it is in waiting
				updateThread.notify();
			}


		}
		public void stop() {
			// 
			synchronized (this) {
				toRun = false;
			}

		}

	}

	private boolean updateNetworkBuild(NetworkBuild networkBuild, String jobId) {
		SubmissionHistoryEntry trackHistory = submissionHistoryManager.getEntry(jobId);
		logger.debug("Found Entry {} ", trackHistory); // trackHistory.isSucceed();
		// TODO :: need to expand the logic here for updating and tracking tasks associated with the project
		if(trackHistory != null && trackHistory.isNotRunning()) {
			trackHistory.getStatus();
			PostUpdateJob updateJob = new PostUpdateJob(ProjectsManagerImpl.this,networkBuild,trackHistory) ;
			updateJob.setUser(false);
			updateJob.schedule();
			return true;
		}
		return false;

	}
	@Override
	public IValidationResult submitDTFileJob(DTFile dtFile, WorkflowJob job,  SubMonitor subMonitor) {
		job.setWorkflowManager(workflowManager);
		IValidationResult validationResult = workflowManager.runWorkflowJob(job);
		if (validationResult.isOK()) {
			processingTasks.put(dtFile.getLastJobId(), dtFile);

		}
		else {
			logger.error("Failed to submit to workflow: {}", validationResult.getMessage());

		}
		boolean result = validationResult.isOK();// rawDataFile.sumbitToWorkflow(workflowManager);
		return validationResult;

	}

	@Override
	public IValidationResult submitNetworkJob(NetworkBuild networkBuild, WorkflowJob job, IProgressMonitor monitor) {
		job.setWorkflowManager(workflowManager);
		IValidationResult validationResult = workflowManager.runWorkflowJob(job);
		if (validationResult.isOK()) {
			networkTasks.put(networkBuild.getLastJobId(), networkBuild);

		}
		else {
			logger.error("Failed to submit to workflow: {}", validationResult.getMessage());

		}
		boolean result = validationResult.isOK();// rawDataFile.sumbitToWorkflow(workflowManager);
		return validationResult;

	}

	/**
	 * validate the file in the dataset exists
	 */
	@Override
	public boolean validateFile(Dataset dataset, String filename) {
		String remoteDatasetPath = dataset.getBaseDirectory();
		String remoteFilePath = remoteDatasetPath + "/" + filename;
		return remoteStorage.validateFile(remoteFilePath);

	}


	@Override
	public String resolveFile(Dataset dataset, String dataFile) {
		String remoteDatasetPath = dataset.getBaseDirectory();
		String remoteFilePath = remoteDatasetPath + "/" + dataFile;
		return remoteStorage.resolveFile(  remoteFilePath);
	}


	/**
	 * return current remote storage
	 */
	@Override
	public RemoteStorage getRemoteStorage() {
		// TODO Auto-generated method stub
		return remoteStorage;
	}


	@Override
	public void datasetRemovedFromProject(Dataset selectedDataset) {

		logger.debug("Removing dataset " + selectedDataset.getDatasetName());
		eventsBroker.post(ProjectEvents.DATASET_REMOVED, selectedDataset);

	}


	/**
	 * delete the dataset from the remote storage
	 */
	@Override
	public void deleteDataset(Dataset selectedDataset, IProgressMonitor monitor) {
		// TODO Auto-generated method stub
		remoteStorage.deleteDataset(selectedDataset, monitor);


	}


	@Override
	public String createNetworkDirectory(NetworkBuild networkBuild, IProgressMonitor monitor) {
		// TODO Auto-generated method stub
		return remoteStorage.createNetworkDirectory(networkBuild, monitor);

	}
	@Override
	public String createNetworkDirectory(Project project, String baseName, IProgressMonitor monitor) {
		// TODO Auto-generated method stub
		return remoteStorage.createNetworkDirectory(project,baseName, monitor);

	}

	@Override
	public void networkBuildAddedToProject(NetworkBuild networkBuild) {
		eventsBroker.post(ProjectEvents.NETWORK_BUILD_ADDED, networkBuild);		
	}


	@Override
	public void networkBuildUpdated(NetworkBuild networkBuild) {
		logger.debug("Network build updated: {}", networkBuild.getNetworkName());
		eventsBroker.post(ProjectEvents.NETWORK_BUILD_UPDATED, networkBuild);		

	}


	@Override
	public void resfreshNetworkBuild(NetworkBuild build) {
		// detect if the network build is already submitted
		if (build.getLastJobId() != null) {
			String jobId = build.getLastJobId();
			if(updateNetworkBuild(build, jobId))
			{
				if (networkTasks.containsKey(jobId))
					networkTasks.remove(jobId); 
			}
		} else {
			// need to submit the network build again
			NetworkBuildJob job = new NetworkBuildJob(build);
			job.setUser(true);
			job.schedule();
		}

	}

	@Override
	public void deleteNetworkBuild(NetworkBuild build, IProgressMonitor monitor) {
		logger.debug("Deleting network build: {}", build.getNetworkName());
		String networksDirectory = build.getProject().getNetworksDirectory();
		// sync submissionHistoryManager if needed
		synchronized (submissionHistoryManager) {
			updator.initalize();
		}


		for (String networkTaskId : build.getNetworkTasks()) {

			//  get the submission history entry
			SubmissionHistoryEntry enrty = submissionHistoryManager.getEntry(networkTaskId);
			if (enrty != null)
			{
				submissionHistoryManager.deleteEntry(enrty);
			}

			String networkDirectory = build.getProject().getTasksDirectory(networkTaskId);
			remoteStorage.deleteDirectory( networkDirectory);
		}
		build.getProject().removeNetworkBuild(build);
		saveProject(build.getProject(),monitor);
	}
	@Override
	public void deleteNetworkBuild(NetworkBuild build) {

		deleteNetworkBuild(build, new NullProgressMonitor());

	}


	@Override
	public void networkBuildDeleted(NetworkBuild build) {
		logger.debug("Network build deleted: {}", build.getNetworkName());
		eventsBroker.post(ProjectEvents.NETWORK_BUILD_DELETED, build);

	}


	@Override
	public void networkAddedToProject(Network network) {
		logger.debug("Network added to project: {}", network.getNetworkName());
		eventsBroker.post(ProjectEvents.NETWORK_ADDED, network);
	}


	@Override
	public void networkDeleted(Network selectedNetwork) {
		logger.debug("Network deleted: {}", selectedNetwork.getNetworkName());
        eventsBroker.post(ProjectEvents.NETWORK_DELETED, selectedNetwork);
    
		
	}


	@Override
	public void deleteNetwork(Network selectedNetwork, IProgressMonitor monitor) {
		String networkPath = selectedNetwork.getNetworkDirectory();
		remoteStorage.deleteDirectory(networkPath);
		
	}


	@Override
	public void networkUpdated(Network network) {
		logger.debug("Network updated: {}", network.getNetworkName());
		eventsBroker.post(ProjectEvents.NETWORK_UPDATED, network);
		
	}


	@Override
	public void networkDesignAddedToProject(NetworkDesign design) {
		logger.debug("Network design added to project: {}", design.getDesignName());
		eventsBroker.post(ProjectEvents.NETWORK_DESIGN_ADDED, design);
	}
	
	
	
	@Override
	public void setActiveDesign(NetworkDesign newDesign) {
		logger.debug("Active design changed: {}", newDesign.getDesignName());
		app.getContext().set(IProjectsManager.ACTIVE_DESIGN, newDesign);
		eventsBroker.post(ProjectEvents.ACTIVE_DESIGN_CHANGED, newDesign);
		
	}
	
	public NetworkDesign getActiveDesign() {
		return (NetworkDesign) app.getContext().get(IProjectsManager.ACTIVE_DESIGN);
	}


	@Override
	public void setSelectedDesgin(NetworkDesign networkDesign) {
		logger.debug("Selected design: {}", networkDesign.getDesignName());
        app.getContext().set(IProjectsManager.SELECTED_DESIGN, networkDesign);
        
    
		
	}
















}
