package com.biotechvana.netools.projects;

import java.io.File;
import java.io.OutputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import com.biotechvana.workflow.tracking.ExecStatus;
import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.JsonBackReference;
import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIdentityInfo;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonManagedReference;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.ObjectIdGenerators;

@JsonIdentityInfo(generator = ObjectIdGenerators.PropertyGenerator.class, property = "datasetID" , scope = Dataset.class)
@JsonAutoDetect(fieldVisibility = JsonAutoDetect.Visibility.ANY, getterVisibility = JsonAutoDetect.Visibility.NONE)

public class Dataset implements Serializable , IDataTable {
	private static final long serialVersionUID = 1L;

	private String datasetID;
	private String datasetName;
	private DatasetType datasetType;
	
	@JsonBackReference//("project-dataset")
	private Project project;

	// a list of Variables and corresponding distributions
	private List<Variable> variables = new ArrayList<Variable>();
	
	protected Map<String,Sample> samplesMap = new java.util.HashMap<String,Sample>();
	
	protected List<Sample> samples = new ArrayList<Sample>();
	
	/**
	 *  A list of processed data files
	 */
	@JsonManagedReference
	private List<DTFile> dataFiles =new ArrayList<DTFile>();
	
	@JsonCreator
	public Dataset(@JsonProperty("datasetName") String datasetName, @JsonProperty("datasetType")DatasetType datasetType) {
		// trim the dataset name
		datasetName = datasetName.trim();
		// Sanitize the project name to remove spaces and special characters to create a unique project ID
		// first replace white spaces with underscore
		this.datasetID = datasetName.replaceAll("\\s+", "_");
		this.datasetID = this.datasetID.replaceAll("[^a-zA-Z0-9_]", "");
		this.datasetName = datasetName;
		this.datasetType = datasetType;
	}
	
	public String getDatasetName() {
		return datasetName;
	}
	public DatasetType getDatasetType() {
		return datasetType;
	}

	public void setProject(Project project) {
		this.project = project;
		
	}
	
	@JsonIgnore
	public String getBaseDirectory() {
		// TODO Auto-generated method stub
		return project.getDatasetDirectory() + "/" + datasetID;
	}
	public Project getProject() {
		return project;
    
	}
	
	public void addDTFile(DTFile file) {
		dataFiles.add(file);
		file.setDataset(this);
		project.getProjectManager().dtFileAddedToDataset(file);
		
	}
	
	public List<DTFile> getDTFiles() {
		// return a copy of the list
		return new ArrayList<DTFile>(dataFiles);
	}

	public String getDatasetID() {
		return datasetID;
    
	}

	@Override
	public int getNumberOfSamples() {
		return samples.size();
	}
	

	@Override
	public int getNumberOfVariables() {
		// use last dtfile to get the number of variables
		return variables.size() ;
	}

	@Override
	public ExecStatus getExecStatus() {
		// use last dtfile to get the exec status
		if (dataFiles.isEmpty())
			return null;
		return dataFiles.get(dataFiles.size()-1).getExecStatus();
		
	}

	public void addVariable(Variable variable) {
		this.addVariable(variable, null);
		
	}
	public void addVariable(Variable variable, DTFile dtFile) {
		// variable is originated from dtFile
		variables.add(variable);
	}

	public void addSample(Sample sample, DTFile dtFile) {
		// variable is originated from dtFile
		samples.add(sample);
        samplesMap.put(sample.getName(), sample);
		
	}

	public Collection<? extends Variable> getVariables() {
		return variables;
    }

	public String getDataFile() {
		// last dtfile is the data file
		// return dataFiles.get(dataFiles.size()-1).getFilePath();
		return getBaseDirectory() + "/" + datasetID + ".csv";
	}

	public String getLastDataFile() {
		// last dtfile is the data file
		return dataFiles.get(dataFiles.size()-1).getFilePath();
	}
	
	public String getVariablesFile() {
		// TODO :: need to implement
		// for the moemnt return 01_{datasetID}_variables.csv
		return getBaseDirectory() + "/" + datasetID + "_variables.csv";
	}

	// write the variables to a csv file with the following format 
	// "Variable", "Distribution", VariableType
	public void writeVariables(OutputStream outputStream) {
		 // for each variable write the variable name, distribution and type
		// write the header
			try {
				outputStream.write("\"VariableID\",\"VariableName\",\"Distribution\",\"DataType\",\"VariableType\",\"IsSelected\"\n".getBytes());
				for (Variable variable : variables) {
					String datasetType = variable.getDataType().getKey();
					String variableType = variable.getVariableType().getKey();
					String distribution = variable.getDistribution().getKey();
					outputStream.write(
							("\""+ variable.getId() +    "\",\"" +  variable.getName() + "\",\"" + distribution + "\",\"" + datasetType +  "\",\"" +variableType +     "\"," +  variable.isSelected()  +  "\n")
									.getBytes());
				}
			} catch (Exception e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
    
		
	}
	

	

	
	
	
	
}
