package com.biotechvana.netools.projects;

import java.io.IOException;
import java.io.OutputStream;
import java.io.Reader;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.csv.CSVFormat;
import org.apache.commons.csv.CSVParser;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.SubMonitor;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.widgets.Shell;
import org.jgrapht.graph.DefaultDirectedWeightedGraph;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.biotechvana.netools.jobs.NetworkJob;
import com.biotechvana.netools.models.DirectedWeightedNetwork;
import com.biotechvana.netools.models.Edge;
import com.biotechvana.netools.models.Node;
import com.biotechvana.netools.models.json.DDWGraph;
import com.biotechvana.netools.models.json.DDWGraphDeserializer;
import com.biotechvana.netools.models.json.DDWGraphSerializer;
import com.biotechvana.netools.models.json.DefaultDirectedWeightedGraphMixin;
import com.biotechvana.workflow.IValidationResult;
import com.biotechvana.workflow.submissionhistory.SubmissionHistoryEntry;
import com.biotechvana.workflow.tracking.ExecStatus;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.json.JsonMapper;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.biotechvana.netools.models.Node;


public class BNLearnNetworkBuild extends NetworkBuild {
	private static final Logger logger = LoggerFactory.getLogger(BNLearnNetworkBuild.class);
    // Paramaters for building the network
    private int maxParents = 4;

    // bootstrap options
    private boolean bootstrap = true;
    private int nBootstrap = 100;
    private NetworkScore score = NetworkScore.BIC;
    /**
     * bootstrap score threshold for the network
     */
    private Double bootstrapScoreThreshold = 0.3;
    private boolean adaptiveBootstrapScoreSelection = true;
	private boolean runModelSelection = true;
	private Double bicScoreThreshold = 0.0;
	private Double pvalueThreshold = 0.1;

    public int getMaxParents() {
        return maxParents;
    }

    public void setMaxParents(int maxParents) {
        this.maxParents = maxParents;
    }

    public boolean isBootstrap() {
        return bootstrap;
    }

    public void setBootstrap(boolean bootstrap) {
        this.bootstrap = bootstrap;
    }

    public int getNBootstrap() {
        return nBootstrap;
    }

    public void setNBootstrap(int nBootstrap) {
        this.nBootstrap = nBootstrap;
    }

    public NetworkScore getScore() {
        return score;
    }

    public void setScore(NetworkScore score) {
        this.score = score;
    }

    public Double getBootstrapScoreThreshold() {
        return bootstrapScoreThreshold;
    }

    public void setScoreThreshold(Double scoreThreshold) {
        this.bootstrapScoreThreshold = scoreThreshold;
    }
    
    
	public void setRunModelSelection(boolean isSelected) {
		this.runModelSelection = isSelected;
		
	}
	
	// if we run model selection we need to create two networks,  baseNetworkID will be the network before model selection
	String baseNetworkID = null;
	public IValidationResult buildProject(IProgressMonitor monitor) {
		
		setExecStatus(ExecStatus.Scheduled);
		// create the network directory
		String projectID = project.getProjectID();
		
		if (this.runModelSelection ) {
			baseNetworkID = networkName+"_base";
			//project.getProjectManager().createNetworkDirectory(project,baseNetworkName, monitor);
			
		}
		networkID = project.getUniqueNetworkID(networkName);
		String finalNetworkDirectory =  project.getProjectManager().createNetworkDirectory(project,networkID , monitor);
		if (baseNetworkID != null) {
			// create second network directory
			baseNetworkID = project.getUniqueNetworkID(baseNetworkID);
			project.getProjectManager().createNetworkDirectory(project,baseNetworkID , monitor);
		}
		
		
		String networkTaskBaseDirectory = project.getTasksDirectory()    ;//project.getNetworksDirectory();
		// write final dataset to network dir
		// project.writeDatasets(monitor);
		// let the project write all datasets files
		List<String> datasetsDataFiles = new ArrayList<String>();
		List<String> datasetsVariablesFiles = new ArrayList<String>();
		for (Dataset dataset : project.getDatasets()) {
			datasetsDataFiles.add(dataset.getDataFile());
			datasetsVariablesFiles.add(dataset.getVariablesFile());
		}
		
		NetworkDesign design = project.getProjectManager().getActiveDesign();
//		if (this.networkDesignID != null) {
//			design = project.getNetworkDesign(this.networkDesignID);
//		}
		
		// get all paths to dataset in the project data.csv and variables.csv
		// prepare the job set inputs file and variable files
		nRestart += 1;
		if (nRestart > 1) {
			// if we have more than one restart we need to add the restart number to the
			// task name
			lastTaskName = baseTaskName + "_" + (nRestart);
		}
		else {
			lastTaskName = baseTaskName;
			
		}
		
		
		NetworkJob job = new NetworkJob();
		job.setJobName(lastTaskName);
		job.setProjectId(project.getProjectID());
		job.setNetworkName(buildID);
		job.setNetworkDirectory(finalNetworkDirectory);
		job.setNetworkMethod(NETWORK_METHOD_BNLEAN);
		job.setBaseDirectory(networkTaskBaseDirectory);
		job.setNetworkScore(score.toString());
		job.setBootstrap(bootstrap);
		
		if (design != null) {
			job.setWLFilename(design.getWhitelistFileName());
			job.setBLFilename(design.getBlacklistFileName());
		}
		
		if (bootstrap) {
			
			job.setNBootstrap(nBootstrap);
		}
		job.setAdaptiveBootstrapScoreSelection(adaptiveBootstrapScoreSelection);
		// job.setNBootstrap(nBootstrap);
		job.setRunModelSelection(runModelSelection);
		if (runModelSelection) {
			if (bootstrap && !adaptiveBootstrapScoreSelection) {
				job.setBootstrapScoreThreshold(bootstrapScoreThreshold);
			}
			
			// this.bicScoreThreshold
			job.setBICScoreThreshold(bicScoreThreshold);
			// this.pvalueThreshold
			job.setPValueThreshold(pvalueThreshold);
		}
		
		

		job.setInputFiles(datasetsDataFiles);
		job.setInputVariablesFiles(datasetsVariablesFiles);
		
		// submit the job
		IValidationResult result = project.getProjectManager().submitNetworkJob(this, job, monitor);
		if (result.isOK()) {
			setExecStatus(ExecStatus.Running);
		} else {
			setExecStatus(ExecStatus.Failed);
		}
		
		// if the design is not null, then write the whilelist and black list the netowkr directory
		return result;
	}

	@Override
	public void postUpdate(SubmissionHistoryEntry trackHistory , IProgressMonitor monitor) {
		// for bootstrap and network selection we will create two networks
		// one network for the boostrap and the other for the final network
		if (trackHistory !=  null ) {
			if (trackHistory.getStatus() ==  ExecStatus.Failed) {
				setExecStatus(ExecStatus.Failed);
				return;
			}
		}
		if (trackHistory.getStatus() ==  ExecStatus.Running) {
			// still running
			return;
		}
		if (trackHistory.getStatus() == ExecStatus.Finished) {
			
			
			
			String buildDirectory = project.getTasksDirectory(lastTaskName);
			String networkDirectory = project.getNetworkDirectory(networkID);
			boolean isValid = true;
			
			String targetEdgesFile = networkDirectory + "/" + networkID + ".edges.csv";
			String targetNodesFile = networkDirectory + "/" + networkID + ".nodes.csv";
			
			
			String sourceNodesFile = buildDirectory + "/" + buildID + ".nodes.csv";
			// those files in case we run model selection
			String sourceEdgesFile = buildDirectory + "/" + buildID + ".edges.csv";
			
			if (this.runModelSelection ) {
				String baseNetworkDir = project.getNetworkDirectory(baseNetworkID);
				String targetBaseEdgesFile = baseNetworkDir + "/" + baseNetworkID + ".edges.csv";
				String targetbaseNodesFile = baseNetworkDir + "/" + baseNetworkID + ".nodes.csv";
				
				
				String sourceBaseEdgesFile = buildDirectory + "/" + buildID + ".base.edges.csv";
				String sourceBaseNodesFile = buildDirectory + "/" + buildID + ".base.nodes.csv";
				if (this.bootstrap ) {
					// for bootstrap we need to create two network 
					// first netwok is {networkName}_bootstrap which is undirected network will all edges from the bootstrap,
					// the second network is the final network with the selected edges based on the cutoff value 
					sourceBaseEdgesFile = buildDirectory + "/" + buildID + ".bootstrap.edges.csv";
					sourceBaseNodesFile = buildDirectory + "/" + buildID + ".bootstrap.nodes.csv";
				} 

				isValid = isValid && project.getProjectManager().getRemoteStorage().validateFile(sourceBaseEdgesFile);
				isValid = isValid && project.getProjectManager().getRemoteStorage().validateFile(sourceBaseNodesFile);
				// copy the network file to the network directory
				project.getProjectManager().getRemoteStorage().copyFile(sourceBaseEdgesFile, targetBaseEdgesFile);
				// copy the node file to the network directory
				project.getProjectManager().getRemoteStorage().copyFile(sourceBaseNodesFile, targetbaseNodesFile);
				
				
				
			}
			else {
				if (this.bootstrap) {
					// for bootstrap
					sourceEdgesFile = buildDirectory + "/" + buildID + ".bootstrap.edges.csv";
					sourceNodesFile = buildDirectory + "/" + buildID + ".bootstrap.nodes.csv";
				} 
				else {
					sourceEdgesFile = buildDirectory + "/" + buildID + ".base.edges.csv";
                    sourceNodesFile = buildDirectory + "/" + buildID + ".base.nodes.csv";
				}
			}
			
			

			isValid = isValid && project.getProjectManager().getRemoteStorage().validateFile(sourceEdgesFile);
			isValid = isValid && project.getProjectManager().getRemoteStorage().validateFile(sourceNodesFile);
			
			
			
			// copy the network file to the network directory
			project.getProjectManager().getRemoteStorage().copyFile(sourceEdgesFile, targetEdgesFile);
			// copy the node file to the network directory
			project.getProjectManager().getRemoteStorage().copyFile(sourceNodesFile, targetNodesFile);
			
			
			
			if (isValid ) {
				// create a Network and add it to the project collection of networks
				//Network.createNetwork(project, networkName, project.getNetworkDirectory(networkName) , monitor);
				createNetwork(monitor);
				setExecStatus(ExecStatus.Finished);
				
				// save project so far
				//project.getProjectManager().saveProject(project, false, monitor);
			} else {
				setExecStatus(ExecStatus.Failed);
			}
			
			
			
		}
		
	}

	private boolean loadNetwork(String nodesFile, String edgesFile) {
		DirectedWeightedNetwork network = new DirectedWeightedNetwork();	

		try (Reader reader = project.getProjectManager().getRemoteStorage().openForReading(nodesFile);
			CSVParser csvParser = new CSVParser(reader, CSVFormat.DEFAULT.withDelimiter(',').withFirstRecordAsHeader())) {
			// read the network file , file has at least two columns "ID","Cluster","InDegree","OutDegree","DegreeCentrality","ClosenessCentrality","BetweennessCentrality"

			csvParser.getRecords().forEach(record -> {
				String id = record.get("ID");
				logger.debug("Node : " + id);
				Variable variable = project.getVariable(id);
				// String label = record.get("Label");
				// logger.debug("Node : " + id + " with label (" + label + ")");
				Node newNode = network.addNode(id);
				// if the record has a cluster column, then add it to the Node
				if (record.isMapped("Cluster") ) {
					String cluster = record.get("Cluster");					
					newNode.setAttribute("Cluster", cluster);
				}
				if (variable != null) {
					//newNode.setAttribute("Variable", variable);
					
					newNode.setLabel(variable.getName());
					newNode.setAttribute("Type", variable.getDataType().toString());
					newNode.setAttribute("Distribution", variable.getDistribution().toString());
				}
				
			});
		} catch (IOException e1) {
			// TODO Auto-generated catch block
			e1.printStackTrace();
		}

		try (Reader reader = project.getProjectManager().getRemoteStorage().openForReading(edgesFile);
				CSVParser csvParser = new CSVParser(reader, CSVFormat.DEFAULT.withDelimiter(',').withFirstRecordAsHeader())) {
			// read the network file , file has at least two columns "From","To" and a score 
			csvParser.getRecords().forEach(record -> {
				String id = record.get("ID");
				String from = record.get("From");
				String to = record.get("To");
				String score = record.get("Score");
				logger.debug("Edge : " + from + " -> " + to + " with score (" + score + ")");
				
				// Add edge to network with the score as weight
				try {
					double weightScore = Double.parseDouble(score);
					network.addEdge(from, to, weightScore);
				} catch (NumberFormatException e) {
					logger.error("Invalid score format for edge " + id + ": " + score, e);
				}
			});
			
			
			// testing saveing and loading the network
			// save the network
			String networkPath  = project.getNetworkDirectory(networkName) + "/" + networkName + ".network.json";
			logger.debug("Saving network to " + network);
			OutputStream outputStream = project.getProjectManager().getRemoteStorage().openForWriting(networkPath);
			
			
//			SimpleModule module = new SimpleModule();
//			
//			module.addSerializer( DDWGraph .class, new DDWGraphSerializer());
//			module.addDeserializer(DefaultDirectedWeightedGraph.class, new DDWGraphDeserializer());

			
			ObjectMapper mapper = JsonMapper.builder().configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false).build();
			
			mapper.addMixIn(DefaultDirectedWeightedGraph.class, DefaultDirectedWeightedGraphMixin.class);
			mapper.writeValue(outputStream, network);
			// load the network
			 DirectedWeightedNetwork network2 = mapper.readValue(project.getProjectManager().getRemoteStorage().openForReading(networkPath), DirectedWeightedNetwork.class);
			
			return true;
		}
		catch (Exception e) {
			logger.debug("Saving network to " + network);
			logger.error("Can not read network file", e);
		}
		return false;
	}

	@Override
	public void createNetwork(IProgressMonitor monitor) {
		
		
		String networkDirectory = project.getNetworkDirectory(this.networkID);
		String targetEdgesFile = networkDirectory + "/" + this.networkID + ".edges.csv";
        String targetNodesFile = networkDirectory + "/" + this.networkID + ".nodes.csv";
        // check if the files are there
		if (!project.getProjectManager().getRemoteStorage().validateFile(targetEdgesFile)) {
			logger.error("Can not find network file " + targetEdgesFile);
			throw new RuntimeException("Can not find network file " + targetEdgesFile);
		}
		if (!project.getProjectManager().getRemoteStorage().validateFile(targetNodesFile)) {
			logger.error("Can not find network file " + targetNodesFile);
			throw new RuntimeException("Can not find network file " + targetNodesFile);
		}
		
		if (project.hasNetwork(this.networkID)) {
			logger.debug("Network " + this.networkID + " already loaded");
			Network network = project.getNetwork(this.networkID);
			// reset the network
		}
		else {
			// create a Network and add it to the project collection of networks
			Network.createNetwork(project, networkName, this.networkID ,true, monitor);
		}
		
		
		if (this.baseNetworkID != null) {
			String baseNetworkDirectory = project.getNetworkDirectory(this.baseNetworkID);
			String targetBaseEdgesFile = baseNetworkDirectory + "/" + this.baseNetworkID + ".edges.csv";
			String targetbaseNodesFile = baseNetworkDirectory + "/" + this.baseNetworkID + ".nodes.csv";
			// check if the files are there
			if (!project.getProjectManager().getRemoteStorage().validateFile(targetBaseEdgesFile)) {
				logger.error("Can not find network file " + targetBaseEdgesFile);
				throw new RuntimeException("Can not find network file " + targetBaseEdgesFile);
			}
			if (!project.getProjectManager().getRemoteStorage().validateFile(targetbaseNodesFile)) {
				logger.error("Can not find network file " + targetbaseNodesFile);
				throw new RuntimeException("Can not find network file " + targetbaseNodesFile);
			}
			if (project.hasNetwork(this.baseNetworkID)) {
				logger.debug("Network " + this.baseNetworkID + " already loaded");
				Network network = project.getNetwork(this.baseNetworkID);
				// reset the network
			} else {
				// create a Network and add it to the project collection of networks
				Network.createNetwork(project, this.baseNetworkID, this.baseNetworkID, false, monitor);
			}
		}
		
		
		
		project.getProjectManager().saveProject(project, false, monitor);
        
		// loadNetwork(targetNodesFile, targetEdgesFile);
    	
	}

	
	public void setAdaptiveBootstrapScoreSelection(boolean isSelected) {
		this.adaptiveBootstrapScoreSelection = isSelected;		
	}

	public boolean getUseAdaptiveBootstrapScoreSelection() {
		return adaptiveBootstrapScoreSelection;
	}

	public boolean getRunModelSelection() {
		return runModelSelection;
	}

	public void setPValueThreshold(double pValueThreshold2) {
		this.pvalueThreshold = pValueThreshold2;
		
	}

	public void setBicScoreThreshold(double bicScoreThreshold2) {
		this.bicScoreThreshold = bicScoreThreshold2;
        
		
	}

	public double getBicScoreThreshold() {
		// TODO Auto-generated method stub
		return bicScoreThreshold;
	}

	public double getPValueThreshold() {
		// TODO Auto-generated method stub
		return pvalueThreshold;
	}

	public void setBuildID(String buildId) {
		this.buildID = buildId;
		
	}





	
	
}
