package com.biotechvana.netools.models;

import org.apache.commons.csv.CSVFormat;
import org.apache.commons.csv.CSVParser;
import org.eclipse.core.runtime.IProgressMonitor;
import org.jgrapht.Graph;
import org.jgrapht.graph.DefaultDirectedWeightedGraph;
import org.jgrapht.graph.DefaultEdge;
import org.jgrapht.graph.SimpleGraph;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.biotechvana.netools.models.json.DefaultDirectedWeightedGraphMixin;
import com.biotechvana.netools.projects.BNLearnNetworkBuild;
import com.biotechvana.netools.projects.Network;
import com.biotechvana.netools.projects.Project;
import com.biotechvana.netools.projects.RemoteStorage;
import com.biotechvana.netools.projects.Variable;
import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.JsonIdentityInfo;
import com.fasterxml.jackson.annotation.JsonSubTypes;
import com.fasterxml.jackson.annotation.JsonTypeInfo;
import com.fasterxml.jackson.annotation.ObjectIdGenerators;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.json.JsonMapper;

import java.io.IOException;
import java.io.OutputStream;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import com.fasterxml.jackson.annotation.JsonInclude;

@JsonTypeInfo(
	    use = JsonTypeInfo.Id.NAME,
	    include = JsonTypeInfo.As.PROPERTY,
	    property = "type"
	)
	@JsonSubTypes({
	    @JsonSubTypes.Type(value = SimpleGraph.class, name = "SimpleGraph"),
	    @JsonSubTypes.Type(value = DirectedWeightedNetwork.class, name = "DirectedWeightedGraph")
	   // @JsonSubTypes.Type(value = C.class, name = "C")
	})
@JsonIdentityInfo(generator = ObjectIdGenerators.PropertyGenerator.class, property = "networkID")
@JsonAutoDetect(fieldVisibility = JsonAutoDetect.Visibility.ANY, getterVisibility = JsonAutoDetect.Visibility.NONE)
@JsonInclude(JsonInclude.Include.NON_NULL)
abstract public class NetworkModel {
	private static final Logger logger = LoggerFactory.getLogger(NetworkModel.class);

	static HashMap<String, String> KnownEdgeModelTypes = new HashMap<String, String>();
	static HashMap<String, String> KnownNodeModelTypes = new HashMap<String, String>();
	
	
	// EDGE ATTRIBUTES
	static final String EDGE_CLM_ID = "id"; // String
	static final String EDGE_CLM_SOURCE = "source"; // String
	static final String EDGE_CLM_TARGET = "target"; // String
	static final String EDGE_CLM_LABEL = "label"; // String
	// weight
	static final String EDGE_CLM_WEIGHT = "weight"; // double
	static final String EDGE_CLM_STRENGTH = "strength"; // double
	static final String EDGE_CLM_PROPORTION = "proportion"; // double
	static final String EDGE_CLM_STRENGTH_SCORE = "strength.score"; // double
	static final String EDGE_CLM_P_VALUES = "p.values"; // double
	static final String EDGE_CLM_BOOTSTRAP_CORE = "bootstrap_core"; // boolean
	static final String EDGE_CLM_P_VALUE_SCORE = "p.value.score"; // double
	static final String EDGE_CLM_FREQUENCY = "frequency"; // double
	
	
	// NODE ATTRIBUTES
	static final String NODE_CLM_ID = "id"; // String
	static final String NODE_CLM_LABEL = "label"; // String
	static final String NODE_CLM_TYPE = "Type"; // categorical
	static final String NODE_CLM_VARIABLE_TYPE = "VariableType"; // categorical
	static final String NODE_CLM_DISTRIBUTION = "Distribution"; // categorical
	static final String NODE_CLM_VARIABLE_NAME = "VariableName"; // String
	static final String NODE_CLM_DATA_TYPE = "DataType"; // categorical
	static final String NODE_CLM_CLUSTER = "Cluster"; // categorical
	static final String NODE_CLM_IS_SELECTED = "IsSelected"; // boolean
	
	
	// possible model types used of parsing and creating filters
	public static final String MODEL_TYPE_DOUBLE = "double";
	public static final String MODEL_TYPE_STRING = "string";
	public static final String MODEL_TYPE_BOOLEAN = "boolean";
	public static final String MODEL_TYPE_INTEGER = "integer";
	// for categorical variables
	public static final String MODEL_TYPE_CATEGORICAL = "categorical";
	
	static {
		// {"edgeID":"X11_X8","sourceID":"X11","targetID":"X8","color":"#e5a50a","width":0.253055749624453,"arrows":"to","weight":0.253055749624453,"opacity":0.0,
		// "customAttributes":{"strength":"-4.85571417021765","proportion":"0.5","strength.score":"0.0101858320815105","p.values":"0.075237734534697","bootstrap_core":"FALSE","p.value.score":"0","frequency":"NA"}}
		KnownEdgeModelTypes.put(EDGE_CLM_ID, MODEL_TYPE_STRING);
		KnownEdgeModelTypes.put(EDGE_CLM_SOURCE, MODEL_TYPE_STRING);
		KnownEdgeModelTypes.put(EDGE_CLM_TARGET, MODEL_TYPE_STRING);
		KnownEdgeModelTypes.put(EDGE_CLM_LABEL, MODEL_TYPE_STRING);
		KnownEdgeModelTypes.put(EDGE_CLM_WEIGHT, MODEL_TYPE_DOUBLE);
		KnownEdgeModelTypes.put(EDGE_CLM_STRENGTH, MODEL_TYPE_DOUBLE);
		KnownEdgeModelTypes.put(EDGE_CLM_PROPORTION, MODEL_TYPE_DOUBLE);
		KnownEdgeModelTypes.put(EDGE_CLM_STRENGTH_SCORE, MODEL_TYPE_DOUBLE);
		KnownEdgeModelTypes.put(EDGE_CLM_P_VALUES, MODEL_TYPE_DOUBLE);
		KnownEdgeModelTypes.put(EDGE_CLM_BOOTSTRAP_CORE, MODEL_TYPE_BOOLEAN);
		KnownEdgeModelTypes.put(EDGE_CLM_P_VALUE_SCORE, MODEL_TYPE_DOUBLE);
		KnownEdgeModelTypes.put(EDGE_CLM_FREQUENCY, MODEL_TYPE_DOUBLE);
		
		KnownNodeModelTypes.put(NODE_CLM_ID, MODEL_TYPE_STRING);
		KnownNodeModelTypes.put(NODE_CLM_LABEL, MODEL_TYPE_STRING);
		KnownNodeModelTypes.put(NODE_CLM_TYPE, MODEL_TYPE_CATEGORICAL);
		KnownNodeModelTypes.put(NODE_CLM_VARIABLE_TYPE, MODEL_TYPE_CATEGORICAL);
		KnownNodeModelTypes.put(NODE_CLM_DISTRIBUTION, MODEL_TYPE_CATEGORICAL);
		KnownNodeModelTypes.put(NODE_CLM_VARIABLE_NAME, MODEL_TYPE_STRING);
		KnownNodeModelTypes.put(NODE_CLM_DATA_TYPE, MODEL_TYPE_CATEGORICAL);
		KnownNodeModelTypes.put(NODE_CLM_CLUSTER, MODEL_TYPE_CATEGORICAL);
		KnownNodeModelTypes.put(NODE_CLM_IS_SELECTED, MODEL_TYPE_BOOLEAN);
		
	}
	
	protected String networkID;
	protected String networkName;
	
	// transient protected Graph<Node, Edge> graph;
    protected GraphType graphType;
    
    public NetworkModel(GraphType graphType)  {
    	this.graphType = graphType;
    }
    
//    Set<Node> nodes = new HashSet<Node>();
    // Set<Edge> edges = new HashSet<Edge>();
//    HashMap<String,HashMap<String,Edge>> edges =  new HashMap<String,HashMap<String,Edge>>();
    
    // transient Map<String,Edge> edgesMap = new HashMap<String,Edge>();
    
	protected abstract Graph< Node,  Edge> getGraph() ;
    
	
	LinkedHashSet<String> nodesModel = new LinkedHashSet<String>();
	Set<String> edgesModel = new HashSet<String>();
	
	transient Network network;
	
	
	public void setNetwork(Network network) {
		this.network = network;
	}

  
    public Node addNode(String nodeId) {
        Node node = new Node(nodeId);
        getGraph().addVertex(node);
        return node;
    }
    
//    Set<Node> vertexSet() {
//    	return  nodes;
//    }

	

	public Edge addEdge(Node fromNode, Node toNode) {

		if (fromNode != null && toNode != null) {
			Edge edge = new Edge();
			edge.setSource(fromNode.getId());
			edge.setTarget(toNode.getId());
			edge.setId(fromNode.getId() + "_" + toNode.getId());
//			HashMap<String, Edge> map = edges.get(fromNode.getId());
//			if (map == null) {
//				map = new HashMap<String, Edge>();
//				edges.put(fromNode.getId(), map);
//			}
//			map.put(toNode.getId(), edge);
			getGraph().addEdge(fromNode, toNode, edge);
			return edge;
		}
		return null;
		
	}
	
	

	public void setEdgeWeight(Edge edge, double weight) {
		edge.setWeight(weight);
	}
    


    public Node getNodeById(String nodeId) {
        return getGraph().vertexSet().stream().filter(node -> node.getId().equals(nodeId)).findFirst().orElse(null);
    }
    
    public Edge getEdgeById(String edgeId) {
        return getGraph().edgeSet().stream().filter(edge -> edge.getId().equals(edgeId)).findFirst().orElse(null);
    }

    
    
    public Edge addEdge(String fromNodeId, String toNodeId) {
        Node fromNode = getNodeById(fromNodeId);
        Node toNode = getNodeById(toNodeId);
        return addEdge(fromNode, toNode);
    }
    
    
    public JSONObject getJsonData() {
        JSONArray nodesArray = new JSONArray();
        JSONArray edgesArray = new JSONArray();

        // Export nodes with attributes
        for (INode node : getGraph().vertexSet()) {
            JSONObject nodeJson = new JSONObject();
//            nodeJson.put("id", node.getId());
//            nodeJson.put("label", node.getLabel());
//            nodeJson.put("color", node.getColor());
//            nodeJson.put("shape", node.getShape());
//            nodeJson.put("x", node.getPosition().getX());
//            nodeJson.put("y", node.getPosition().getY());
//            nodeJson.put("size", node.getSize());
//            nodeJson.put("borderWidth", node.getBorderWidth());
//            nodeJson.put("borderWidthSelected", node.getBorderWidthSelected());
//            nodeJson.put("font", node.getFont());
//            nodeJson.put("mass", node.getMass());
//            nodeJson.put("physics", node.getPhysics());
            nodesArray.put(node.toJson());
        }

        // Export edges with attributes
        for (Edge edge :  getGraph().edgeSet()) {
            JSONObject edgeJson = new JSONObject();
            
            
            // edgeJson.put("from", getGraph(). getEdgeSource(edge).getId());
            // edgeJson.put("to",  getGraph().getEdgeTarget(edge).getId());
            // wighted graph
            edgeJson.put("value", edge.getWeight());
            edge.toJson(edgeJson);
            
            
            edgesArray.put(edgeJson);
        }

        // Combine nodes and edges into a JSON object
        JSONObject visJsData = new JSONObject();
        visJsData.put("nodes", nodesArray);
        visJsData.put("edges", edgesArray);

        return visJsData;
    }
    
    public String exportToVisJsFormat() {
        JSONArray nodesArray = new JSONArray();
        JSONArray edgesArray = new JSONArray();

        // Export nodes with attributes
        for (INode node : getGraph().vertexSet()) {
            JSONObject nodeJson = new JSONObject();
//            nodeJson.put("id", node.getId());
//            nodeJson.put("label", node.getLabel());
//            nodeJson.put("color", node.getColor());
//            nodeJson.put("shape", node.getShape());
//            nodeJson.put("x", node.getPosition().getX());
//            nodeJson.put("y", node.getPosition().getY());
//            nodeJson.put("size", node.getSize());
//            nodeJson.put("borderWidth", node.getBorderWidth());
//            nodeJson.put("borderWidthSelected", node.getBorderWidthSelected());
//            nodeJson.put("font", node.getFont());
//            nodeJson.put("mass", node.getMass());
//            nodeJson.put("physics", node.getPhysics());
            nodesArray.put(node.toJson());
        }

        // Export edges with attributes
        for (Edge edge :  getGraph().edgeSet()) {
            JSONObject edgeJson = new JSONObject();
            
            
            // edgeJson.put("from", getGraph(). getEdgeSource(edge).getId());
            // edgeJson.put("to",  getGraph().getEdgeTarget(edge).getId());
            // wighted graph
            edgeJson.put("value", edge.getWeight());
            edge.toJson(edgeJson);
            
            
            edgesArray.put(edgeJson);
        }

        // Combine nodes and edges into a JSON object
        JSONObject visJsData = new JSONObject();
        visJsData.put("nodes", nodesArray);
        visJsData.put("edges", edgesArray);

        return visJsData.toString();
    }




	public String exportToVisJsFormatSimple() {
        JSONArray nodesArray = new JSONArray();
        JSONArray edgesArray = new JSONArray();

        // Export nodes with only id and label
        for (INode node :  getGraph().vertexSet()) {
            JSONObject nodeJson = new JSONObject();
            nodeJson.put("id", node.getId());
            nodeJson.put("label", node.getLabel());
            nodesArray.put(nodeJson);
        }

        // Export edges without additional attributes
        for (Edge edge :  getGraph().edgeSet()) {
            JSONObject edgeJson = new JSONObject();
            edgeJson.put("from",  getGraph().getEdgeSource(edge).getId());
            edgeJson.put("to",  getGraph().getEdgeTarget(edge).getId());
            edgesArray.put(edgeJson);
        }

        // Combine nodes and edges into a JSON object
        JSONObject visJsData = new JSONObject();
        visJsData.put("nodes", nodesArray);
        visJsData.put("edges", edgesArray);

        return visJsData.toString();
    }

	
	/**
	 * Create a network from a nodes and edges file, set network name and project
	 * @param networkName : name of the network
	 * @param nodesFile : file with nodes on remote storage
	 * @param edgesFile : file with edges on remote storage
	 * @param project : project to which the network belongs
	 * @return
	 */
	public static NetworkModel createNetwork(String networkName, String nodesFile, String edgesFile,
			Project project,
			IProgressMonitor monitor) {
		DirectedWeightedNetwork network = new DirectedWeightedNetwork();	
		network.networkName = "test";
		network.graphType = GraphType.DIRECTED;
		
		


		
		return network;
	}
	
	// save network to json format
	public boolean saveNetwork(IProgressMonitor monitor) {
		if (network == null || network.getProject() == null) {
			 throw new RuntimeException("Project is not set for the network");
		}
		if (networkName == null) {
			throw new RuntimeException("Network name is not set for the network");
		}
		// testing saveing and loading the network
		// save the network
		String networkPath = network.getNetworkDirectory();
		networkPath  =  networkPath + "/" + networkName + ".network.json";
		logger.debug("Saving network to " + network);
		Project project = network.getProject();
		OutputStream outputStream = project.getProjectManager().getRemoteStorage().openForWriting(networkPath);	
		ObjectMapper mapper = JsonMapper.builder().configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false).build();	
		mapper.addMixIn(DefaultDirectedWeightedGraph.class, DefaultDirectedWeightedGraphMixin.class);
		try {
			mapper.writeValue(outputStream, this);
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
			throw new RuntimeException("Can not save network", e);
		}
//		// load the network
//		 DirectedWeightedNetwork network2 = mapper.readValue(project.getProjectManager().getRemoteStorage().openForReading(networkPath), DirectedWeightedNetwork.class);
//		
		return true;
	}
	
	// load network from json format
	public static NetworkModel loadNetwork(Network network) {
		String networkPath = network.getNetworkDirectory();
		//networkPath = "\\NETOOLS\\projects\\testproject\\networks\\test_network";
		networkPath  =  networkPath + "/" + network.getNetworkName() + ".network.json";
		//networkPath  =  networkPath + "\\" + network.getNetworkName() + ".network.json";
		RemoteStorage remoteStorage = network.getProject().getProjectManager().getRemoteStorage();
		ObjectMapper mapper = JsonMapper.builder().configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false).build();
		mapper.addMixIn(DefaultDirectedWeightedGraph.class, DefaultDirectedWeightedGraphMixin.class);
		NetworkModel networkModel;
		try {
			networkModel = mapper.readValue(remoteStorage.openForReading(networkPath), DirectedWeightedNetwork.class);
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
			throw new RuntimeException("Can not read network", e);
		}
		networkModel.setNetwork(network);
		return networkModel;
	}

	public static NetworkModel createNetwork(Network network, String nodesFile, String edgesFile,boolean directed,
			IProgressMonitor monitor) {
		DirectedWeightedNetwork networkModel = new DirectedWeightedNetwork();
		networkModel.networkName = network.getNetworkName();
		networkModel.networkID = network.getNetworkID();
		if (directed)
			networkModel.graphType = GraphType.DIRECTED;
		else
			networkModel.graphType = GraphType.UNDIRECTED;
		networkModel.setNetwork(network);
		RemoteStorage remoteStorage = network.getProject().getProjectManager().getRemoteStorage();
		try (Reader reader = remoteStorage.openForReading(nodesFile);
			CSVParser csvParser = new CSVParser(reader, CSVFormat.DEFAULT.withDelimiter(',').withFirstRecordAsHeader())) {
			// read the network file , file has at least two columns "ID","Cluster","InDegree","OutDegree","DegreeCentrality","ClosenessCentrality","BetweennessCentrality"
			// VariableID, VariableName, Distribution, DataType, VariableType, IsSelected, cluster
			csvParser.getRecords().forEach(record -> {
				String id = record.get("VariableID");
				logger.debug("Node : " + id);
				Variable variable = network.getProject().getVariable(id);
				// String label = record.get("Label");
				// logger.debug("Node : " + id + " with label (" + label + ")");
				Node newNode = networkModel.addNode(id);
				// if the record has a cluster column, then add it to the Node
				// for each attribute in the record, add it to the
				for (String key : record.toMap().keySet()) {
					if (key.equals("VariableID"))
						continue;
					//newNode.setAttribute(key, record.get(key));
					Object parsedValue = parseValue(key, record.get(key), "node");
					newNode.setAttribute(key, parsedValue);
				}
//				if (record.isMapped("Cluster") ) {
//					String cluster = record.get("Cluster");					
//					newNode.setAttribute("Cluster", cluster);
//				}
				if (variable != null) {
					newNode.setLinkedVariable(variable);	
					
					newNode.setLabel(variable.getName());
					newNode.setAttribute("Type", variable.getDataType().toString());
					newNode.setAttribute("Distribution", variable.getDistribution().toString());
				}
				
			});
		} catch (IOException e1) {
			// TODO Auto-generated catch block
			e1.printStackTrace();
			throw new RuntimeException("Can not read nodes file", e1);
		}

		try (Reader reader = remoteStorage.openForReading(edgesFile);
				CSVParser csvParser = new CSVParser(reader, CSVFormat.DEFAULT.withDelimiter(',').withFirstRecordAsHeader())) {
			// read the network file , file has at least two columns "From","To" and a score 
			csvParser.getRecords().forEach(record -> {
				// if id in the record, then use it
				String id = null;
				if (!record.isMapped("id")) {
                    logger.error("Edge record does not have an id");
				}
				else {
					id = record.get("id");
				}
				
				String from = record.get("from");
				String to = record.get("to");
				String score = record.get("score");
				logger.debug("Edge : " + from + " -> " + to + " with score (" + score + ")");
				if (id == null) {
					id = from + "_" + to;
				}
				// Add edge to network with the score as weight
				try {
					double weightScore = Double.parseDouble(score);
					Edge edge = networkModel.addEdge(from, to, weightScore);
					if (directed) {
						edge.setArrows("to");
					}
					edge.setWeight(weightScore);
					edge.setWidth(weightScore);
					// // for each attribute in the record, add it to the edge
					for (String key : record.toMap().keySet()) {
                        if (key.equals("from") || key.equals("to") || key.equals("score") || key.equals("id"))
                            continue;
                        Object parsedValue = parseValue(key,record.get(key), "edge");
                        edge.setAttribute(key, parsedValue);
					}
				} catch (NumberFormatException e) {
					logger.error("Invalid score format for edge " + id + ": " + score, e);
				}
			});
			
			

		}
		catch (Exception e) {
			logger.debug("Saving network to " + network);
			logger.error("Can not read network file", e);
			throw new RuntimeException("Can not read edges ", e);
		}
    	
		
		
		
		return networkModel;
	}

	// convert the string to the type based on the attribute 
	public static Object parseValue(String key, String value, String element) {
		String type = null;
		if (element.equals("node")) {
			type = KnownNodeModelTypes.get(key);
		} else if (element.equals("edge")) {
			type = KnownEdgeModelTypes.get(key);
		}
		
		if (type == null) {
			return value;
		}
		switch (type) {
		case MODEL_TYPE_DOUBLE:
			try {
				return Double.parseDouble(value);
			} catch (NumberFormatException e) {
				logger.error("Invalid double format for key " + key + ": " + value, e);
				return null;
			}
		case MODEL_TYPE_INTEGER:
			try {
				return Integer.parseInt(value);
			} catch (NumberFormatException e) {
				logger.error("Invalid integer format for key " + key + ": " + value, e);
				return null;
			}
		case MODEL_TYPE_BOOLEAN:
			if (value.equalsIgnoreCase("true") || value.equalsIgnoreCase("false") ||
					value.equalsIgnoreCase("1") || value.equalsIgnoreCase("0") 
					|| value.equalsIgnoreCase("yes") || value.equalsIgnoreCase("no")
					) {
				return Boolean.parseBoolean(value);
			} else {
				logger.error("Invalid boolean format for key " + key + ": " + value);
				return null;
			}
		case MODEL_TYPE_STRING:
			return value;
		default:
			return value;
		}
	}
	
	public Set<String> getNodeModel() {
		// fill this.nodesModel
		// for each node in the graph
		// get the node attributes
		this.nodesModel.clear();
		// All nodes has id
		this.nodesModel.add("id");
		// all nodes has label
		this.nodesModel.add("label");
		for (Node node : getGraph().vertexSet()) {
			Map<String, Object> attributes = node.getAttributes();
			for (String key : attributes.keySet()) {
				if (!key.equals("id") && !key.equals("label") && !nodesModel.contains(key)) {
					nodesModel.add(key);
				}
			}
		}
		return this.nodesModel;
	}

	public Set<Node> getNodes() {
		return getGraph().vertexSet();
	}

	public  String[] getEdgedModel() {

		edgesModel.clear();
//		edgesModel.add("id");
//		
//		edgesModel.add("label");
//		edgesModel.add("source");
//		edgesModel.add("target");
//		edgesModel.add("weight");
		for (Edge edge : getGraph().edgeSet()) {
			Map<String, Object> attributes = edge.getAttributes();
			for (String key : attributes.keySet()) {
				if (!edgesModel.contains(key)) {
					edgesModel.add(key);
				}
			}
		}
		String[] edgesHeader = new String[edgesModel.size() + 5];
		edgesHeader[0] = "id";
		edgesHeader[1] = "label";
		edgesHeader[2] = "source";
		edgesHeader[3] = "target";
		edgesHeader[4] = "weight";
		Set<String > basicEdgesModel = new HashSet<String>();
		basicEdgesModel.add("id");
		basicEdgesModel.add("label");
		basicEdgesModel.add("source");
		basicEdgesModel.add("target");
		basicEdgesModel.add("weight");
		// add the rest 
		int i = 5;
		for (String key : edgesModel) {
			if (basicEdgesModel.contains(key))
				continue;
            edgesHeader[i] = key;
            i++;
		}
		
		
		return edgesHeader;
	}

	public Set<Edge> getEdges() {
		return getGraph().edgeSet();
	}
	
	public String[] getEdgedModelType() {
		String[] edgesModelHeader = getEdgedModel();
		String[] propertyTypes = new String[edgesModelHeader.length];
		for (int j = 0; j < edgesModelHeader.length; j++) {
			String key = edgesModelHeader[j];
			if (KnownEdgeModelTypes.containsKey(key)) {
				propertyTypes[j] = KnownEdgeModelTypes.get(key);
				if (propertyTypes[j] == null) {
					// default to string
					propertyTypes[j] = MODEL_TYPE_STRING;
				}
			}
		}
		
		return propertyTypes;
	}
	
	public String[] getNodeModelType() {
		Set<String> nodesModelHeader = getNodeModel();
		List<String> nodesModelHeaderList = new ArrayList<>(nodesModelHeader);
		String[] propertyTypes = new String[nodesModelHeaderList.size()];
		for (int j = 0; j < nodesModelHeader.size(); j++) {
			String key = nodesModelHeaderList.get(j);
			if (KnownNodeModelTypes.containsKey(key)) {
				propertyTypes[j] = KnownNodeModelTypes.get(key);
				if (propertyTypes[j] == null) {
					// default to string
					propertyTypes[j] = MODEL_TYPE_STRING;
				}
			}
		}
		
		return propertyTypes;
	}
	    
}
