package com.biotechvana.netools.internal;

import java.util.ArrayList;
import java.util.List;
import java.util.Arrays;

import org.eclipse.e4.core.di.annotations.Optional;
import org.eclipse.e4.core.di.extensions.EventTopic;
import org.eclipse.e4.ui.di.UIEventTopic;
import org.eclipse.e4.ui.di.UISynchronize;
import org.eclipse.e4.ui.model.application.MApplication;
import org.eclipse.e4.ui.progress.IProgressService;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.forms.widgets.Form;
import org.eclipse.ui.forms.widgets.FormText;
import org.eclipse.ui.forms.widgets.FormToolkit;

import com.biotechvana.commons.SharedImages;
import com.biotechvana.netools.IVariablesView;
import com.biotechvana.netools.projects.Dataset;
import com.biotechvana.netools.projects.DatasetType;
import com.biotechvana.netools.projects.Distribution;
import com.biotechvana.netools.projects.IProjectsManager;
import com.biotechvana.netools.projects.Project;
import com.biotechvana.netools.projects.ProjectEvents;
import com.biotechvana.netools.projects.Variable;
import com.biotechvana.netools.projects.VariableType;

import jakarta.annotation.PostConstruct;
import jakarta.inject.Inject;
import jakarta.inject.Named;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.ControlListener;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.forms.widgets.ScrolledForm;
import org.eclipse.ui.forms.widgets.Section;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Combo;

import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;

public class VariablesView implements IVariablesView {
	public static final Logger logger = LoggerFactory.getLogger(VariablesView.class);
	FormToolkit toolkit;
	 Form detailsForm;
	 Composite mainBody;
	 ScrolledComposite jobScrolledComposite;

	Dataset selected_dataset;
	Project activeProject;
	IProjectsManager projectsManager;

	@Inject
	IProgressService progressService;


	@Inject
	UISynchronize sync;


	List<Variable> variables = new ArrayList<Variable>();
	Composite variablesComposite;
	Composite variablesCompositeBody;
	TableViewer tableViewer;

	Group variableEditGroup;
	Variable currentSelectedVariable;

	@Inject
	@Optional 
	public void setActiveProject(@Named(IProjectsManager.ACTIVE_PROJECT) Project project) {
		activeProject = project;
		selected_dataset = null;
		sync.asyncExec(
				() -> updateUI()
			);
	}

	@Inject
	@Optional
	public void setSelectedDataset(@Named(IProjectsManager.SELECTED_DATASET) Dataset dataset) {
		selected_dataset = dataset;
		sync.asyncExec(
				() -> updateUI()
			);
	
	}
	@Inject
	@Optional
	void subscribeItemUpdated(@UIEventTopic(ProjectEvents.DATASET_UPDATED) Dataset dataset) {
		if (selected_dataset != null && dataset.getDatasetID().equals(selected_dataset.getDatasetID())) {
			selected_dataset = dataset;
			updateUI();
		}
	}
	


	private void updateUI() {
		if (detailsForm == null || detailsForm.isDisposed()) {
			return;
		}
		logger.debug("Updating UI for Variables View");
		updateVariables();
		// renderVariables();


	}

	private void renderVariablesSection() {
		if (variablesCompositeBody != null && !variablesCompositeBody.isDisposed()) {
			variablesCompositeBody.dispose();
		}


		variablesCompositeBody = toolkit.createComposite(variablesComposite);
		variablesCompositeBody.setLayout(new GridLayout(3, true));
		variablesCompositeBody.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		//		Label noVariablesLabel = toolkit.createLabel(variablesCompositeBody, "No variables");
		//		noVariablesLabel.setLayoutData(new GridData(SWT.CENTER, SWT.CENTER, true, true));
		//
		//		if (variables.isEmpty()) {
		//			
		//		} else {
		//			noVariablesLabel.setText("Number of variables: " + variables.size());
		//		}

		tableViewer = new TableViewer(variablesCompositeBody, SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL | SWT.FULL_SELECTION | SWT.BORDER);
		tableViewer.setContentProvider(ArrayContentProvider.getInstance());

		Table table = tableViewer.getTable();
		table.setHeaderVisible(true);
		table.setLinesVisible(true);
		table.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true , 2, 1	));



		createColumns();

		tableViewer.setInput(variables);

		tableViewer.addSelectionChangedListener(new ISelectionChangedListener() {
			@Override
			public void selectionChanged(SelectionChangedEvent event) {
				handleSelectionChange((IStructuredSelection) event.getSelection());
			}
		});

		variablesComposite.layout(true, true);

		// create an edit form 
		// create a group 
		variableEditGroup = new Group(variablesCompositeBody, SWT.NONE);
		variableEditGroup.setLayout(new GridLayout(2, false));
		variableEditGroup.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		variableEditGroup.setText("Edit Selected Variable");
	}

	private void createColumns() {
		String[] titles = { "ID", "Name", "Type", "Distribution", "Missing Data", "Selected" };
		int[] bounds = { 100, 200, 150, 150, 100, 100 };

		// ID column
		TableViewerColumn col = createTableViewerColumn(titles[0], bounds[0]);
		col.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(Object element) {
				Variable var = (Variable) element;
				return var.getId();
			}
		});

		// Name column
		col = createTableViewerColumn(titles[1], bounds[1]);
		col.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(Object element) {
				Variable var = (Variable) element;
				return var.getName();
			}
		});

		// Type column
		col = createTableViewerColumn(titles[2], bounds[2]);
		col.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(Object element) {
				Variable var = (Variable) element;
				return var.getVariableType().getKey();
			}
		});

		// Distribution column
		col = createTableViewerColumn(titles[3], bounds[3]);
		col.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(Object element) {
				Variable var = (Variable) element;
				return var.getDistribution().getValue();
			}
		});

		// Has Missing Data column
		col = createTableViewerColumn(titles[4], bounds[4]);
		col.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(Object element) {
				Variable var = (Variable) element;
				return var.isHasMissingData() ? "Yes" : "No";
			}
		});

		// Selected column
		col = createTableViewerColumn(titles[5], bounds[5]);
		col.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(Object element) {
				Variable var = (Variable) element;
				return var.isSelected() ? "Yes" : "No";
			}
		});
	}

	private TableViewerColumn createTableViewerColumn(String title, int bound) {
		final TableViewerColumn viewerColumn = new TableViewerColumn(tableViewer, SWT.NONE);
		final TableColumn column = viewerColumn.getColumn();
		column.setText(title);
		column.setWidth(bound);
		column.setResizable(true);
		column.setMoveable(true);
		return viewerColumn;
	}

	private void updateVariables() {
		// TODO Auto-generated method stub
		if (selected_dataset != null) {
			// add all variables from the dataset
			variables.clear();
			currentSelectedVariable = null;
			variables.addAll(selected_dataset.getVariables());

		}
		else {
			variables.clear();
			currentSelectedVariable = null;
		}
		tableViewer.setInput(variables);
		// dispose any controls in variableEditGroup
		for (Control child : variableEditGroup.getChildren()) {
			child.dispose();
		}
		variablesComposite.layout(true, true);
	}

	private void handleSelectionChange(IStructuredSelection selection) {
		if (selection.isEmpty()) {
			variableEditGroup.setText("Edit Selected Variable");
			currentSelectedVariable = null;
			return;
		}
		Variable selectedVariable = (Variable) selection.getFirstElement();
		if (selectedVariable == currentSelectedVariable) {
			return;
		}
		// for refresh saving do this 
		if (currentSelectedVariable != null) {
			tableViewer.refresh(currentSelectedVariable);
		}
		currentSelectedVariable = selectedVariable;

		// dispose any controls in variableEditGroup
		for (Control child : variableEditGroup.getChildren()) {
			child.dispose();
		}


		if (currentSelectedVariable != null) {
			createEditVariableComposite(variableEditGroup, selectedVariable);
		}

		variablesComposite.layout(true, true);
	}

	private void createEditVariableComposite(Group variableGroup, Variable variable) {

		variableGroup.setText("Edit : "  + variable.getName());

		// Variable ID
		Label idLabel = toolkit.createLabel(variableGroup, "ID:");
		Text idText = toolkit.createText(variableGroup, variable.getId(), SWT.BORDER | SWT.READ_ONLY);
		idText.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
		idText.setEnabled(false);
		// Variable Name
		Label nameLabel = toolkit.createLabel(variableGroup, "Name:");
		Text nameText = toolkit.createText(variableGroup, variable.getName(), SWT.BORDER);
		nameText.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
		nameText.addListener(SWT.Modify, e -> variable.setName(nameText.getText()));


		// Variable Type
		//		Label typeLabel = toolkit.createLabel(variableGroup, "Type:");
		//		Combo typeCombo = new Combo(variableGroup, SWT.DROP_DOWN | SWT.READ_ONLY);
		//		typeCombo.setItems(Arrays.stream(VariableType.values()).map(VariableType::getValue).toArray(String[]::new));
		//		typeCombo.setText(variable.getVariableType().getValue());
		//		typeCombo.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
		//		

		// Distribution
		Label distLabel = toolkit.createLabel(variableGroup, "Distribution:");
		Combo distCombo = new Combo(variableGroup, SWT.DROP_DOWN | SWT.READ_ONLY);
		distCombo.setItems(Arrays.stream(Distribution.values()).map(Distribution::getValue).toArray(String[]::new));
		distCombo.setText(variable.getDistribution().getValue());
		distCombo.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
		distCombo.addListener(SWT.Selection, e -> variable.setDistribution(Distribution.valueOf(distCombo.getText().toUpperCase())));

		if (variable.getDataType() == DatasetType.EXPERIMENT  || variable.getDataType() == DatasetType.GENERAL 
				|| variable.getDataType() == DatasetType.OUTCOME 
				|| variable.getDataType() == DatasetType.COVARIATE ) {

			distCombo.setEnabled(true);
		}
		else {
			distCombo.setEnabled(false);
		}

		// Has Missing Data
		Label missingDataLabel = toolkit.createLabel(variableGroup, "Has Missing Data:");
		Button missingDataCheckbox = toolkit.createButton(variableGroup, "", SWT.CHECK );
		missingDataCheckbox.setEnabled(false);
		missingDataCheckbox.setSelection(variable.isHasMissingData());

		// Is Selected
		Label selectedLabel = toolkit.createLabel(variableGroup, "Selected:");
		Button selectedCheckbox = toolkit.createButton(variableGroup, "", SWT.CHECK);
		selectedCheckbox.setSelection(variable.isSelected());
		selectedCheckbox.addListener(SWT.Selection, e -> variable.setSelected(selectedCheckbox.getSelection()));


		// // Create a FormText widget
		FormText formText = toolkit.createFormText(variableGroup, true);
		formText.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true, 2, 1));

		String formHelpText = getFormHelpText(variable);

		formText.setText(formHelpText, true, false);
	}

	private String getFormHelpText(Variable variable) {
		StringBuilder helpText = new StringBuilder();
		helpText.append("<form>");
		helpText.append("<p><b>Variable Type:</b> ").append(variable.getVariableType().getKey()).append("</p>");
		helpText.append("<p><b>Distribution:</b> ").append(variable.getDistribution().getValue()).append("</p>");
		helpText.append("<p><b>Description:</b></p>");

		switch (variable.getVariableType()) {
		case DISCRETE:
			helpText.append("<p>Discrete variables are countable in a finite amount of time. Examples include the number of students in a class, the number of cars in a parking lot, etc.</p>");
			break;
		case CONTINUOUS:
			helpText.append("<p>Continuous variables are measurable and can take any value within a range. Examples include height, weight, temperature, etc.</p>");
			break;
		}

		switch (variable.getDistribution()) {
		case NORMAL:
			helpText.append("<p>Normal distribution is a continuous probability distribution that is symmetrical on both sides of the mean, so the right side of the center is a mirror image of the left side.</p>");
			break;
		case BINOMIAL:
			helpText.append("<p>Binomial distribution is a discrete probability distribution of the number of successes in a sequence of n independent experiments, each asking a yes-no question.</p>");
			break;
		case POISSON:
			helpText.append("<p>Poisson distribution is a discrete frequency distribution that gives the probability of a number of independent events occurring in a fixed time.</p>");
			break;
		case MULTINOMIAL:
			helpText.append("<p>Multinomial distribution is a generalization of the binomial distribution. It describes outcomes of multi-nomial scenarios unlike binomial which describes binary scenarios.</p>");
			break;
		case NB:
			helpText.append("<p>Negative Binomial distribution is a discrete probability distribution of the number of successes in a sequence of independent and identically distributed Bernoulli trials before a specified number of failures occurs.</p>");
			break;
		case ZINB:
			helpText.append("<p>Zero Inflated Negative Binomial distribution is used for count data that has an excess of zero counts.</p>");
			break;
//		case UNIFORM:
//			helpText.append("<p>Uniform distribution is a type of probability distribution in which all outcomes are equally likely; each variable has the same probability that it will be the outcome.</p>");
//			break;
//		case EXPONENTIAL:
//			helpText.append("<p>Exponential distribution is a continuous probability distribution that describes the time between events in a Poisson point process.</p>");
//			break;
		case GAMMA:
			helpText.append("<p>Gamma distribution is a two-parameter family of continuous probability distributions. It is used to model waiting times.</p>");
			break;
//		case BETA:
//			helpText.append("<p>Beta distribution is a family of continuous probability distributions defined on the interval [0, 1], parameterized by two positive shape parameters.</p>");
//			break;
//		case CAUCHY:
//			helpText.append("<p>Cauchy distribution is a continuous probability distribution. It is also known as the Lorentz distribution.</p>");
//			break;
//		case CHI_SQUARE:
//			helpText.append("<p>Chi-Square distribution is a continuous probability distribution of the sum of the squares of k independent standard normal random variables.</p>");
//			break;
//		case F:
//			helpText.append("<p>F distribution is a continuous probability distribution that arises frequently as the null distribution of a test statistic, especially in variance analysis.</p>");
//			break;
		case LOGNORMAL:
			helpText.append("<p>Log-Normal distribution is a continuous probability distribution of a random variable whose logarithm is normally distributed.</p>");
			break;
		case NOMINAL:
			helpText.append("<p>Nominal distribution is a type of data that is used to label variables without providing any quantitative value.</p>");
			break;
		default:
			break;
		}

		helpText.append("</form>");
		return helpText.toString();
	}

	@PostConstruct
	public void createPartControl(Composite parent, MApplication app, IProjectsManager projectsManager) { 
		app.getContext().set(IVariablesView.class, this);
		this.projectsManager = projectsManager;
		toolkit = new FormToolkit(parent.getDisplay());

		detailsForm = toolkit.createForm(parent);
		detailsForm.setText("Variables View");
		detailsForm.getBody().setLayout(new FillLayout());
		toolkit.decorateFormHeading(detailsForm);
		
		jobScrolledComposite = new ScrolledComposite( detailsForm.getBody(),  SWT.H_SCROLL | SWT.V_SCROLL);
		jobScrolledComposite.setExpandVertical(true);
		jobScrolledComposite.setExpandHorizontal(true);
		jobScrolledComposite.setAlwaysShowScrollBars(true);

		mainBody =  toolkit.createComposite(jobScrolledComposite);
		
		
		mainBody.setLayout(new GridLayout(1, false));
		
		

		// Create filter section
		Section filterSection = toolkit.createSection(mainBody, Section.TITLE_BAR);
		filterSection.setText("Filter Variables");
		filterSection.setLayoutData(new GridData(SWT.FILL, SWT.TOP, true, false));

		Composite filterComposite = toolkit.createComposite(filterSection);
		filterComposite.setLayout(new GridLayout(4, false));

		Label searchLabel = toolkit.createLabel(filterComposite, "");
		GridData labelData = new GridData(SWT.FILL, SWT.CENTER, false, false);
		labelData.widthHint = 150;
		searchLabel.setLayoutData(labelData);
		Text searchText = toolkit.createText(filterComposite, "", SWT.BORDER);
		searchText.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false, 2, 1));

		Button searchButton = toolkit.createButton(filterComposite, "Search", SWT.PUSH);
		searchButton.setImage(SharedImages.SEARCH);

		filterSection.setClient(filterComposite);

		// Create variables section
		Section variablesSection = toolkit.createSection(mainBody, Section.TITLE_BAR);
		variablesSection.setText("Variables");
		variablesSection.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));

		variablesComposite = toolkit.createComposite(variablesSection);
		variablesComposite.setLayout(new GridLayout(1, false));
		renderVariablesSection();
		variablesSection.setClient(variablesComposite);
		
		jobScrolledComposite.setMinSize(mainBody.computeSize(SWT.DEFAULT, SWT.DEFAULT));

	    jobScrolledComposite.setContent(mainBody);
	    jobScrolledComposite.setExpandVertical(true);
	    jobScrolledComposite.setExpandHorizontal(true);
	    jobScrolledComposite.setAlwaysShowScrollBars(true);
	    
//	 // add resize handler to the parent
//	    parent.addControlListener(new ControlListener() {
//			
//			@Override
//			public void controlResized(ControlEvent e) {
//				// TODO Auto-generated method stub
//				if (mainBody != null && !mainBody.isDisposed()) {
//					jobScrolledComposite.setMinSize(mainBody.computeSize(SWT.DEFAULT, SWT.DEFAULT));
//				}
//				
//
//			}
//			
//			@Override
//			public void controlMoved(ControlEvent e) {
//				// TODO Auto-generated method stub
//				
//			}
//		});
	}


}
