package com.biotechvana.netools.internal;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.MultiStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.e4.core.di.annotations.Optional;
import org.eclipse.e4.ui.di.UISynchronize;
import org.eclipse.e4.ui.model.application.MApplication;
import org.eclipse.e4.ui.progress.IProgressService;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.IMessageProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Button;
import org.eclipse.ui.forms.events.HyperlinkAdapter;
import org.eclipse.ui.forms.widgets.Form;
import org.eclipse.ui.forms.widgets.FormText;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.ScrolledForm;
import org.eclipse.ui.forms.widgets.Section;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.biotechvana.commons.SharedImages;
import com.biotechvana.netools.projects.Dataset;
import com.biotechvana.netools.projects.IProjectsManager;
import com.biotechvana.netools.projects.NetworkBuild;
import com.biotechvana.netools.projects.Project;
import com.biotechvana.netools.projects.Variable;
import com.biotechvana.netools.projects.jobs.NetworkBuildJob;
import com.biotechvana.netools.IVariablesView;
import com.biotechvana.netools.projects.BNLearnNetworkBuild;
import com.biotechvana.workflow.IValidationResult;
import com.biotechvana.workflow.ValidationResult;
import com.biotechvana.workflow.WorkflowValidationSet;
import com.biotechvana.workflow.views.WorkflowJobBasicComposite;

import jakarta.annotation.PostConstruct;
import jakarta.inject.Inject;
import jakarta.inject.Named;

import org.eclipse.jface.fieldassist.ControlDecoration;
import org.eclipse.jface.fieldassist.FieldDecorationRegistry;

public class NetworkBuilderView implements IVariablesView {
	public static final Logger logger = LoggerFactory.getLogger(NetworkBuilderView.class);
	FormToolkit toolkit;
	Form detailsForm;
	
	
	Dataset selected_dataset;
	Project activeProject;
	IProjectsManager projectsManager;

	@Inject
	IProgressService progressService;


	@Inject
	UISynchronize sync;


	Composite mainComposite;
	Composite mainCompositeBody;
	TableViewer tableViewer;

	Group variableEditGroup;
	Variable currentSelectedVariable;

	private Combo methodCombo;
	private FormText methodDescription;

	private BNLearnNetworkBuild bnLearnNetworkBuild;

	private boolean isFormValid = true;
	private boolean isNetworkNameValid = true;
	private boolean isNBootstrapValid = true;
	private boolean isScoreThresholdValid = true;
	private boolean isMaxParentsValid = true;
	private ScrolledComposite jobScrolledComposite;
	private Composite mainBody;

	@Inject
	@Optional 
	public void setActiveProject(@Named(IProjectsManager.ACTIVE_PROJECT) Project project) {
		activeProject = project;
		selected_dataset = null;
		sync.asyncExec(
				() -> updateUI()
				);
	}

	@Inject
	@Optional
	public void setSelectedDataset(@Named(IProjectsManager.SELECTED_DATASET) Dataset dataset) {
		selected_dataset = dataset;

	}


	/**
	 * Update GUI in must be called in the display thread
	 */
	private void updateUI() {
		if (detailsForm == null || detailsForm.isDisposed()) {
			return;
		}
		logger.debug("Updating UI for the View");



	}

	private void renderMainSection(String selectedMethod) {
		if (selectedMethod == null) {
			return;
		}
		if (mainCompositeBody != null && !mainCompositeBody.isDisposed()) {
			mainCompositeBody.dispose();
		}


		mainCompositeBody = toolkit.createComposite(mainComposite);
		mainCompositeBody.setLayout(new GridLayout(2, true));
		mainCompositeBody.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));

		if (NetworkBuild.NETWORK_METHOD_BNLEAN.equals(selectedMethod)) {
			bnLearnNetworkBuild = new BNLearnNetworkBuild();

			GridData gd = new GridData(SWT.FILL, SWT.FILL, false, false);
			gd.widthHint = 200;
			gd.horizontalIndent = 10;

			// Create the widgets
			toolkit.createLabel(mainCompositeBody, "Network Name:");
			Text networkNameText = toolkit.createText(mainCompositeBody, "", SWT.BORDER);
			networkNameText.setLayoutData(gd);
			ControlDecoration networkNameDecoration = new ControlDecoration(networkNameText, SWT.TOP | SWT.LEFT);
			networkNameDecoration.setDescriptionText("Network name cannot be empty and must contain only letters or underscores.");
			networkNameDecoration.setImage(FieldDecorationRegistry.getDefault().getFieldDecoration(FieldDecorationRegistry.DEC_ERROR).getImage());
			networkNameDecoration.hide();

			networkNameText.addModifyListener(e -> {
				String networkName = networkNameText.getText();
				if (networkName.isEmpty() || !networkName.matches("[a-zA-Z_]+")) {
					networkNameDecoration.show();
					isNetworkNameValid = false;
				} else {
					networkNameDecoration.hide();
					isNetworkNameValid = true;
				}
				bnLearnNetworkBuild.setNetworkName(networkName);
				validateForm();
			});

			toolkit.createLabel(mainCompositeBody, "Score:");
			Combo scoreCombo = new Combo(mainCompositeBody, SWT.DROP_DOWN | SWT.READ_ONLY);
			scoreCombo.setLayoutData(gd);
			scoreCombo.setItems(new String[] { "AIC", "BIC", "Loglik" });
			scoreCombo.select(1); // Default to BIC
			scoreCombo.addSelectionListener(new SelectionAdapter() {
				@Override
				public void widgetSelected(SelectionEvent e) {
					bnLearnNetworkBuild.setScore(com.biotechvana.netools.projects.NetworkBuild.NetworkScore.valueOf(scoreCombo.getText()));
				}
			});

			toolkit.createLabel(mainCompositeBody, "Score Threshold:");
			Text scoreThresholdText = toolkit.createText(mainCompositeBody, String.valueOf(bnLearnNetworkBuild.getBootstrapScoreThreshold()), SWT.BORDER);
			scoreThresholdText.setLayoutData(gd);
			ControlDecoration scoreThresholdDecoration = new ControlDecoration(scoreThresholdText, SWT.TOP | SWT.LEFT);
			scoreThresholdDecoration.setDescriptionText("Please enter a valid double value.");
			scoreThresholdDecoration.setImage(FieldDecorationRegistry.getDefault().getFieldDecoration(FieldDecorationRegistry.DEC_ERROR).getImage());
			scoreThresholdDecoration.hide();

			scoreThresholdText.addModifyListener(e -> {
				try {
					double scoreThreshold = Double.parseDouble(scoreThresholdText.getText());
					bnLearnNetworkBuild.setScoreThreshold(scoreThreshold);
					scoreThresholdDecoration.hide();
					isScoreThresholdValid = true;
				} catch (NumberFormatException ex) {
					scoreThresholdDecoration.show();
					isScoreThresholdValid = false;
				}
				validateForm();
			});

			toolkit.createLabel(mainCompositeBody, "Max Parents:");
			Text maxParentsText = toolkit.createText(mainCompositeBody, String.valueOf(bnLearnNetworkBuild.getMaxParents()), SWT.BORDER);
			maxParentsText.setLayoutData(gd);
			ControlDecoration maxParentsDecoration = new ControlDecoration(maxParentsText, SWT.TOP | SWT.LEFT);
			maxParentsDecoration.setDescriptionText("Please enter a valid integer.");
			maxParentsDecoration.setImage(FieldDecorationRegistry.getDefault().getFieldDecoration(FieldDecorationRegistry.DEC_ERROR).getImage());
			maxParentsDecoration.hide();

			maxParentsText.addModifyListener(e -> {
				try {
					int maxParents = Integer.parseInt(maxParentsText.getText());
					bnLearnNetworkBuild.setMaxParents(maxParents);
					maxParentsDecoration.hide();
					isMaxParentsValid = true;
				} catch (NumberFormatException ex) {
					maxParentsDecoration.show();
					isMaxParentsValid = false;
				}
				validateForm();
			});

			toolkit.createLabel(mainCompositeBody, "Bootstrap:");
			Button bootstrapButton = toolkit.createButton(mainCompositeBody, "", SWT.CHECK);
			bootstrapButton.setLayoutData(gd);
			bootstrapButton.setSelection(bnLearnNetworkBuild.isBootstrap());


			toolkit.createLabel(mainCompositeBody, "Number of Bootstraps:");
			Text nBootstrapText = toolkit.createText(mainCompositeBody, String.valueOf(bnLearnNetworkBuild.getNBootstrap()), SWT.BORDER);
			nBootstrapText.setLayoutData(gd);
			ControlDecoration nBootstrapDecoration = new ControlDecoration(nBootstrapText, SWT.TOP | SWT.LEFT);
			nBootstrapDecoration.setDescriptionText("Please enter a valid number.");
			nBootstrapDecoration.setImage(FieldDecorationRegistry.getDefault().getFieldDecoration(FieldDecorationRegistry.DEC_ERROR).getImage());
			nBootstrapDecoration.hide();

			nBootstrapText.addModifyListener(e -> {
				try {
					int nBootstrap = Integer.parseInt(nBootstrapText.getText());
					bnLearnNetworkBuild.setNBootstrap(nBootstrap);
					nBootstrapDecoration.hide();
					isNBootstrapValid = true;
				} catch (NumberFormatException ex) {
					nBootstrapDecoration.show();
					isNBootstrapValid = false;
				}
				validateForm();
			});


			bootstrapButton.addSelectionListener(new SelectionAdapter() {
				@Override
				public void widgetSelected(SelectionEvent e) {
					boolean isSelected = bootstrapButton.getSelection();
					bnLearnNetworkBuild.setBootstrap(isSelected);
					nBootstrapText.setEnabled(isSelected);
					if (!isSelected) {
						nBootstrapText.setText("");
						isNBootstrapValid = true;
					} else {
						nBootstrapText.setText(String.valueOf(bnLearnNetworkBuild.getNBootstrap()));
					}
					validateForm();
				}
			});

			nBootstrapText.setEnabled(bnLearnNetworkBuild.isBootstrap());
		}

		mainCompositeBody.layout();
	}

	private void validateForm() {
		isFormValid = isNetworkNameValid && isNBootstrapValid && isScoreThresholdValid && isMaxParentsValid;
	}

	@PostConstruct
	public void createPartControl(Composite parent, MApplication app, IProjectsManager projectsManager) { 
		app.getContext().set(IVariablesView.class, this);
		this.projectsManager = projectsManager;
		toolkit = new FormToolkit(parent.getDisplay());

		detailsForm = toolkit.createForm(parent);
		detailsForm.setText("Structure Learning");
		detailsForm.getBody().setLayout(new FillLayout());
		toolkit.decorateFormHeading(detailsForm);
		detailsForm.addMessageHyperlinkListener(new HyperlinkAdapter());

		detailsForm.getToolBarManager().add(new Action("Run") {
			{
				this.setImageDescriptor(SharedImages.RUNNING_DESC);
			}
			@Override
			public void run() {
				logger.debug("Running");
				runNetworkBuild();
			}

		});
		detailsForm.getToolBarManager().add(new Action("Help") {
			{
				this.setImageDescriptor(SharedImages.RUNNING_DESC);
				this.setImageDescriptor(SharedImages.HELP_DESC);
				this.setToolTipText("Toggle Help/Manual"); // Properties
			}
			@Override
			public void run() {
				logger.debug("Help");
			}
		});
		detailsForm.getToolBarManager().update(true);

		
		jobScrolledComposite = new ScrolledComposite( detailsForm.getBody(),  SWT.H_SCROLL | SWT.V_SCROLL);
//		jobScrolledComposite.setExpandVertical(true);
//		jobScrolledComposite.setExpandHorizontal(true);
//		jobScrolledComposite.setAlwaysShowScrollBars(true);
		mainBody =  toolkit.createComposite(jobScrolledComposite);
	
		mainBody.setLayout(new GridLayout(1, false));
		
		
		// Create filter section
		Section descSection = toolkit.createSection(mainBody, Section.TITLE_BAR);
		descSection.setText("Desc Section");
		descSection.setLayoutData(new GridData(SWT.FILL, SWT.TOP, true, false));

		Composite descComposite = toolkit.createComposite(descSection);
		descComposite.setLayout(new GridLayout(3, false));

		createDescSection(descComposite);

		descSection.setClient(descComposite);

		// Create variables section
		Section mainSection = toolkit.createSection(mainBody, Section.TITLE_BAR);
		mainSection.setText("Build Network");
		mainSection.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));

		mainComposite = toolkit.createComposite(mainSection);
		mainComposite.setLayout(new GridLayout(1, false));
		renderMainSection(NetworkBuild.NETWORK_METHOD_BNLEAN);
		mainSection.setClient(mainComposite);
		
		
		jobScrolledComposite.setMinSize(mainBody.computeSize(SWT.DEFAULT, SWT.DEFAULT));

	    jobScrolledComposite.setContent(mainBody);
	    jobScrolledComposite.setExpandVertical(true);
	    jobScrolledComposite.setExpandHorizontal(true);
	    //jobScrolledComposite.setAlwaysShowScrollBars(true);
	}

	private void createDescSection(Composite descComposite) {
		toolkit.createLabel(descComposite, "Select Method:");
		methodCombo = new Combo(descComposite, SWT.DROP_DOWN | SWT.READ_ONLY);
		methodCombo.setItems(new String[] { NetworkBuild.NETWORK_METHOD_BNLEAN });
		methodCombo.select(0);
		methodCombo.addSelectionListener(new SelectionAdapter()  {
			@Override
			public void widgetSelected(SelectionEvent e) {
				renderMainSection(methodCombo.getText());
				updateMethodDescription();
			}
		});

		methodDescription = toolkit.createFormText(descComposite, false);
		methodDescription.setLayoutData(new GridData(SWT.FILL, SWT.TOP, true, false, 3, 1));
		updateMethodDescription();
	}

	private void updateMethodDescription() {
		String selectedMethod = methodCombo.getText();
		if (NetworkBuild.NETWORK_METHOD_BNLEAN.equals(selectedMethod)) {
			methodDescription.setText("BNLearn - pipeline: Description about BNLearn method.", false, false);
		}
	}


	private void runNetworkBuild() {
		// TODO Auto-generated method stub
		// simulate a run with validation
		
		



		IValidationResult  validation =  getValidation();

		if(validation.isOK() && isFormValid) {
			detailsForm.setMessage("OK", IMessageProvider.INFORMATION);
			detailsForm.setImage( SharedImages.ACCEPT);

			
			
			
			NetworkBuildJob job = new NetworkBuildJob(this.bnLearnNetworkBuild);
			job.setUser(true);
			job.schedule();


		} else {
			detailsForm.setMessage("Not OK", IMessageProvider.ERROR);
			// detailsForm.setImage(SharedImages.CANCEL);
			ErrorDialog.openError(this.detailsForm.getShell(), "Error",
					"Can not run current Task.",
					createMultiStatus(validation));
		}

	}

	IValidationResult getValidation () {
		if(true)
			return ValidationResult.Valid_RESULT;
		WorkflowValidationSet vSet = new WorkflowValidationSet(IValidationResult.ERROR,-1,"Invalid or missing configartions parameters");

		// add some sample errors
		vSet.add(new ValidationResult(IValidationResult.ERROR, -1,"Error 1"));
		vSet.add(new ValidationResult(IValidationResult.ERROR, -1,"Error 2"));
		vSet.add(new ValidationResult(IValidationResult.WARNING, -1,"Error 3"));



		return vSet;
	}

	private IStatus createMultiStatus(IValidationResult res) {

		
		
		
		if(res.isMultiValidation()) {
			//			Throwable ex = res.getException();
			//			if(ex == null) {
			//				ex = new Exception("Invalid Parameters");
			//			}
			MultiStatus ms = new MultiStatus(this.getClass(),  res.getCode(),res.getMessage(),res.getException());
			for(IValidationResult vs : res.getChildren()) {
				ms.add(new Status(vs.getSeverity(), this.getClass(),vs.getCode(),vs.getMessage(),vs.getException() ));
			}
			if(res.getException() == null) {

			}
			return ms;
		}
		else {
			return new Status(res.getSeverity(), this.getClass(),res.getCode(),res.getMessage(),res.getException() );
		}

	}

}
