package com.biotechvana.netools.internal;

import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.e4.core.di.annotations.Optional;
import org.eclipse.e4.core.di.extensions.EventTopic;
import org.eclipse.e4.ui.di.UIEventTopic;
import org.eclipse.e4.ui.di.UISynchronize;
import org.eclipse.e4.ui.model.application.MApplication;
import org.eclipse.e4.ui.progress.IProgressService;
import org.eclipse.swt.widgets.Composite;

import com.biotechvana.commons.SharedImages;
import com.biotechvana.netools.projects.DTFile;
import com.biotechvana.netools.projects.Dataset;
import com.biotechvana.netools.projects.IDataTable;
import com.biotechvana.netools.projects.IProjectsManager;
import com.biotechvana.netools.projects.Project;
import com.biotechvana.netools.projects.ProjectEvents;
import com.biotechvana.netools.projects.jobs.TestJob;
import com.biotechvana.netools.ui.IDatasetViewer;

import jakarta.annotation.PostConstruct;
import jakarta.inject.Inject;
import jakarta.inject.Named;

import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.IBaseLabelProvider;
import org.eclipse.jface.viewers.ILabelProviderListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.TreeViewerColumn;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.BusyIndicator;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeColumn;
import org.eclipse.ui.forms.widgets.Form;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.ScrolledForm;
import org.eclipse.ui.forms.widgets.Section;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.ITreeViewerListener;
import org.eclipse.jface.viewers.TreeExpansionEvent;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.swt.events.MenuDetectListener;
import org.eclipse.swt.events.MenuDetectEvent;
import org.eclipse.swt.widgets.Menu;

/**
 * View to display datasets in a project
 */
public class DatasetsView implements IDatasetViewer {

	private static final Logger logger = LoggerFactory.getLogger(DatasetsView.class);
	Project activeProject;
	private TreeViewer treeViewer;

	@Inject
	IProgressService progressService;


	@Inject
	UISynchronize sync;


	Form form;
	MApplication app;

	@PostConstruct
	public void createPartControl(Composite parent,MApplication app) {
		this.app = app;
		app.getContext().set(IDatasetViewer.class, this);
		FormToolkit toolkit = new FormToolkit(parent.getDisplay());
		form = toolkit.createForm(parent);
		//form.getVerticalBar().setEnabled(true); // Enable vertical scroll
		//form.getHorizontalBar().setEnabled(false); // Disable horizontal scroll
		form.setText("Project - {}");
		form.getBody().setLayout(new FillLayout());

		// Creating the Screen
		Section section = toolkit.createSection(form.getBody(), Section.DESCRIPTION
				| Section.TITLE_BAR   );
		section.setText("Project Datasets"); //$NON-NLS-1$
		section.setDescription("Add, remove and manage datasets"); //$NON-NLS-1$"
		section.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));


		createTreeViewer(section);
		section.setClient(treeViewer.getTree());



	}

	private void createTreeViewer(Composite parent) {
		treeViewer = new TreeViewer(parent, SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL | SWT.FULL_SELECTION | SWT.BORDER);
		createColumns(parent, treeViewer);
		treeViewer.setContentProvider(new ITreeContentProvider() {
			@Override
			public Object[] getElements(Object inputElement) {
				return ((List<?>) inputElement).toArray();
			}

			@Override
			public Object[] getChildren(Object parentElement) {
				if (parentElement instanceof Dataset) {
					return ((Dataset) parentElement).getDTFiles().toArray();
				}
				return new Object[0];
			}

			@Override
			public Object getParent(Object element) {
				if (element instanceof DTFile) {
					return ((DTFile) element).getDataset();
				}
				return null;
			}

			@Override
			public boolean hasChildren(Object element) {
				return element instanceof Dataset && !((Dataset) element).getDTFiles().isEmpty();
			}
		});



		final Tree tree = treeViewer.getTree();
		tree.setHeaderVisible(true);
		tree.setLinesVisible(true);

		// Add selection changed listener
		treeViewer.addSelectionChangedListener(new ISelectionChangedListener() {
			@Override
			public void selectionChanged(SelectionChangedEvent event) {
				logger.debug("Selection changed: {}", event.getSelection().toString());
				setSelection(event.getSelection());
			}
		});

		// Add tree expansion listener
		treeViewer.addTreeListener(new ITreeViewerListener() {
			@Override
			public void treeCollapsed(TreeExpansionEvent event) {
				logger.debug("Tree collapsed: {}", event.getElement().toString());
			}

			@Override
			public void treeExpanded(TreeExpansionEvent event) {
				logger.debug("Tree expanded: {}", event.getElement().toString());
			}
		});

		// Add context menu
		MenuManager menuManager = new MenuManager();
		Menu menu = menuManager.createContextMenu(tree);
		tree.setMenu(menu);

		tree.addMenuDetectListener(new MenuDetectListener() {
			@Override
			public void menuDetected(MenuDetectEvent e) {
				contextMenuHandler(menuManager);
			}
		});

		// Add double-click listener
		treeViewer.addDoubleClickListener(event -> {
			IStructuredSelection selection = treeViewer.getStructuredSelection();
			Object selectedElement = selection.getFirstElement();
			if (selectedElement instanceof Dataset) {
				logger.debug("Double-clicked on Dataset: {}", ((Dataset) selectedElement).getDatasetName());
				// Add logic to handle double-click on Dataset
			} else if (selectedElement instanceof DTFile) {
				logger.debug("Double-clicked on DTFile: {}", ((DTFile) selectedElement).getFileName());
				// Add logic to handle double-click on DTFile
			}
		});

		if (activeProject != null) {
			updateUI();
		}
	}

	protected void setSelection(ISelection selection) {
		if (selection instanceof IStructuredSelection) {
			Object selectedElement = ((IStructuredSelection) selection).getFirstElement();
			Dataset selectedDataset = null;
			if (selectedElement instanceof DTFile) {
				selectedDataset = ((DTFile) selectedElement).getDataset();
			}
			if (selectedElement instanceof Dataset) {
				selectedDataset = (Dataset) selectedElement;

				// Add logic to handle the selected dataset
			}
			if (selectedDataset != null) {
				logger.debug("Selected dataset: {}", selectedDataset.getDatasetName());

			}
			// activeProject.getProjectManager().setSelectedDataset(selectedDataset);
			app.getContext().set(IProjectsManager.SELECTED_DATASET, selectedDataset);
			// refresh layout
			form.layout();
		}
	}

	protected void contextMenuHandler(MenuManager menuManager) {
		menuManager.removeAll();
		Object selectedElement = ((IStructuredSelection) treeViewer.getSelection()).getFirstElement();
		// menu Item
		// add dataset menuitem
		Action addDatasetAction = new Action("Add Dataset") {
			@Override
			public void run() {
				// Add logic to add dataset
			}
		};
		addDatasetAction.setImageDescriptor(SharedImages.ADD_ITEM_DESC);
		addDatasetAction.setToolTipText("Add a new dataset to the project");


		addDatasetAction.setEnabled(true);

		menuManager.add(addDatasetAction);

		// delete menu item
		Action deleteAction = new Action("Delete") {
			@Override
			public void run() {
				// Add logic to add dataset
			}
		};
		deleteAction.setImageDescriptor(SharedImages.REMOVE_ITEM_DESC);
		deleteAction.setToolTipText("Delete selected item");
		deleteAction.setEnabled(true);
		menuManager.add(deleteAction);


		// dynamic menu items TODO :: 
		//		if (selectedElement instanceof Dataset) {
		//			menuManager.add(new Action("Edit Dataset") {
		//				@Override
		//				public void run() {
		//					// Add logic to edit dataset
		//					TestJob job = new TestJob("Edit Dataset");
		//					job.setUser(true);
		//					job.schedule();
		//				}
		//			});
		//			menuManager.add(new Action("Delete Dataset") {
		//				@Override
		//				public void run() {
		//					// Add logic to delete dataset
		//				}
		//			});
		//		} else if (selectedElement instanceof DTFile) {
		//			menuManager.add(new Action("Open File") {
		//				@Override
		//				public void run() {
		//					// Add logic to open file
		//				}
		//			});
		//			menuManager.add(new Action("Delete File") {
		//				@Override
		//				public void run() {
		//					// Add logic to delete file
		//				}
		//			});
		//		}

	}

	private void createColumns(final Composite parent, final TreeViewer viewer) {
		String[] titles = { "Name", "Type/Description" , "Status",  "Samples"  , "Variables"};
		int[] bounds = { 200, 200 , 50	,100,100 };

		// Name column 
		TreeViewerColumn col = createTreeViewerColumn(titles[0], bounds[0], 0);
		col.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(Object element) {
				if (element instanceof Dataset) {
					return ((Dataset) element).getDatasetName();
				} else if (element instanceof DTFile) {
					return ((DTFile) element).getFileName();
				}
				return super.getText(element);
			}

			@Override

			public Image getImage(Object element) {
				if (element instanceof Dataset) {
					return SharedImages.PROJECT;
				} else if (element instanceof DTFile) {
					return SharedImages.TABLE;
				}
				return null;
			}

		});

		// Type/Description column
		col = createTreeViewerColumn(titles[1], bounds[1], 1);
		col.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(Object element) {
				if (element instanceof Dataset) {
					return ((Dataset) element).getDatasetType().toString();
				} else if (element instanceof DTFile) {
					return ((DTFile) element).getDescription();
				}
				return super.getText(element);
			}
		});

		// Status column
		col = createTreeViewerColumn(titles[2], bounds[2], 2);
		col.getColumn().setAlignment(SWT.CENTER);
		col.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(Object element) {
				return "";
			}
			@Override
			public Image getImage(Object element) {

				//				if (element instanceof IDataTable) {
				//					return SharedImages.ACCEPT;
				//				} else
				if (element instanceof IDataTable) {
					return ViewUtils.getExecStatusImage(((IDataTable) element).getExecStatus());
//					if (((IDataTable) element).getExecStatus() == null)
//						return SharedImages.CANCEL;
//					switch (((IDataTable) element).getExecStatus()) {
//					case Failed:
//						return SharedImages.ERROR;
//					case Running:
//						return SharedImages.PROCESS_INFO;
//					case Finished:
//						return SharedImages.ACCEPT;
//					case Warning :
//						return SharedImages.FILE_WARNING;
//					case Unknown:
//						return SharedImages.CANCEL;
//
//					default:
//						return SharedImages.PROCESS_INFO;
//					}

				}
				return null;
			}
		});
		// Samples column
		col = createTreeViewerColumn(titles[3], bounds[3], 3);
		col.getColumn().setAlignment(SWT.CENTER);
		col.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(Object element) {
				if (element instanceof IDataTable) {
					return String.valueOf(((IDataTable) element).getNumberOfSamples());
				}
				return "-";
			}

		});

		//	Variables column
		col = createTreeViewerColumn(titles[4], bounds[4], 4);
		col.getColumn().setAlignment(SWT.CENTER);
		col.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(Object element) {
				if (element instanceof IDataTable) {
					return String.valueOf(((IDataTable) element).getNumberOfVariables());
				}
				return "-";
			}
		});

	}

	private TreeViewerColumn createTreeViewerColumn(String title, int bound, final int colNumber) {
		final TreeViewerColumn viewerColumn = new TreeViewerColumn(treeViewer, SWT.NONE);
		final TreeColumn column = viewerColumn.getColumn();
		column.setText(title);
		column.setWidth(bound);
		column.setResizable(true);
		column.setMoveable(true);
		return viewerColumn;
	}

	@Inject
	void setActiveProject(@Optional @Named(IProjectsManager.ACTIVE_PROJECT) Project activeProject) {
		this.activeProject = activeProject;
		updateUI();
	}

	private void updateUI() {




		sync.asyncExec(() -> {
			//			BusyIndicator.showWhile(treeViewer.getTree().getDisplay(), () -> {
			//	            // Long-running content loading logic here
			//	            try {
			//	                Thread.sleep(3000); // Simulate a delay for loading content
			//	            } catch (InterruptedException e) {
			//	                e.printStackTrace();
			//	            }
			//	            // Populate the view
			//	        });




			if (treeViewer == null || treeViewer.getControl().isDisposed())
				return;

			if (activeProject != null) {
				try {
					//progressService.busyCursorWhile(
					progressService.run(true, false, 	monitor -> {
						// if loading is slow, show a progress monitor to support a lazy loading
						monitor.beginTask("Loading Datasets...", IProgressMonitor.UNKNOWN);
						activeProject.getDatasets();
						monitor.done();
					});
				} catch (Exception e) {
					e.printStackTrace();
				}
				form.setText("Project - " + activeProject.getProjectName());
				treeViewer.setInput(activeProject.getDatasets());
			} else {
				form.setText("No active project");
				treeViewer.setInput(null);
			}
		});
	}




	@Inject
	@Optional
	void subscribeDatasetAdded(@EventTopic(ProjectEvents.DATASET_ADDED) Dataset dataset) {
		logger.debug("DatasetView :  Dataset added: {}", dataset.getDatasetName());
		updateUI();
		//		if (treeViewer != null && !treeViewer.getControl().isDisposed()) {
		//			sync.asyncExec(() -> {
		//				treeViewer.update(dataset, null);
		//			});
		//		}
	}
	@Inject
	@Optional
	void subscribeDTFileAdded(@EventTopic(ProjectEvents.DTFILE_ADDED) DTFile file) {
		logger.debug("DatasetView :  DTFile added: {}", file.getFileName());
		updateUI();
	}

	@Inject
	@Optional
	void subscribeItemUpdated(@EventTopic(ProjectEvents.DTFILE_UPDATED) DTFile file) {
		logger.debug("DatasetView :  DTFile updated: {}", file.getFileName());
		if (treeViewer != null && !treeViewer.getControl().isDisposed()) {
			sync.asyncExec(() -> {
				treeViewer.update(file, null);
			});
		}
	}
	@Inject
	@Optional
	void subscribeItemUpdated(@EventTopic(ProjectEvents.DATASET_UPDATED) Dataset dataset) {
		logger.debug("DatasetView :  Dataset updated: {}", dataset.getDatasetName());
		if (treeViewer != null && !treeViewer.getControl().isDisposed()) {
			sync.asyncExec(() -> {
				treeViewer.update(dataset, null);
			});
		}
	}
	
	@Inject
	@Optional
	void subscribeItemDeleted(@UIEventTopic(ProjectEvents.DATASET_REMOVED) Dataset dataset) {
		logger.debug("DatasetView :  Dataset removed: {}", dataset.getDatasetName());
		if (treeViewer != null && !treeViewer.getControl().isDisposed()) {
			treeViewer.setInput(activeProject.getDatasets());
		}
	}
	

	@Override
	public void refreshDatasetView() {

		logger.debug("DatasetView :  Refreshing view");
		updateUI();

	}
}
