package com.biotechvana.javabiotoolkit.io;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;

import com.biotechvana.javabiotoolkit.BioSequence;
import com.biotechvana.javabiotoolkit.DNASequence;
import com.biotechvana.javabiotoolkit.ProteinSequence;
import com.biotechvana.javabiotoolkit.RNASequence;
import com.biotechvana.javabiotoolkit.exceptions.FastaReaderNotParsedException;
import com.biotechvana.javabiotoolkit.exceptions.FastaWriterMultipleSequenceException;
import com.biotechvana.javabiotoolkit.exceptions.InvalidSequenceCharacterException;
import com.biotechvana.javabiotoolkit.exceptions.SequenceTooLongException;
import com.biotechvana.javabiotoolkit.text.LineSeparatorFormat;

/**
 * store records and open and close reader writer as needed
 * @author ahmed
 *
 */
public class FASTAFileHandler {
	public enum FastaRecordType {
		DNA,
		RNA,
		PROTEIN,

		/**
		 * is set when reading file with unknown type
		 */
		Mixed,
		Unknown
	}


	FastaRecordType recordsType = null;


	/**
	 * File associated to each instance of FastaReader. Contains one or more FASTA sequences. Nucleotide and amino 
	 * acid sequences are both allowed in the same file.
	 */
	private File filePath;
	private FastaRecordCollection fastaRecords;
	Charset fileCharset;
	LineSeparatorFormat fileLineSeparatorFormat; 
	boolean ignoreBlankLines;

	// Use FASTAReader const for now
	private int sortMethod = FASTAReader.SORT_NO;


	private FASTAFileHandler(File filePath , Charset fileCharset, LineSeparatorFormat fileLineSeparatorFormat, boolean ignoreBlankLines) {
		this.fileCharset = fileCharset;
		this.fileLineSeparatorFormat = fileLineSeparatorFormat;
		this.ignoreBlankLines = ignoreBlankLines;
		this.fastaRecords = new FastaRecordCollection();
		this.filePath = filePath;
	}


	public File getFilePath() {
		return this.filePath;
	}


	public Charset getFileCharset() {
		return this.fileCharset;
	}


	public LineSeparatorFormat getFileLineSeparatorFormat(){
		return this.fileLineSeparatorFormat;
	}


	public boolean getIgnoreBlankLines() {
		return this.ignoreBlankLines;
	}

	public void setSortMethod(int sortMethod) {
		this.sortMethod = sortMethod;
	}


	public FastaRecordCollection getFastaRecords(){
		return this.fastaRecords;
	}


	public FastaRecordType getRecordType () {
		return recordsType;
	}

	//^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^

	static public FASTAFileHandler createFastaFileHandler(File filePath, Charset fileCharset, LineSeparatorFormat fileLineSeparatorFormat, boolean ignoreBlankLines , IProgressMonitor progressMonitor) 
			throws FileNotFoundException, OperationCanceledException, IOException {
		FASTAFileHandler fileHandler = new FASTAFileHandler(filePath,fileCharset,fileLineSeparatorFormat,ignoreBlankLines);
		FASTAReader2 reader = new FASTAReader2(fileHandler);
		reader.parse(progressMonitor);
		if(reader.isDnaFile()) {
			fileHandler.recordsType = FastaRecordType.DNA;
		} else if(reader.isRnaFile()) {
			fileHandler.recordsType = FastaRecordType.RNA;
		} else if(reader.isProteinFile()) {
			fileHandler.recordsType = FastaRecordType.PROTEIN;
		}

		return fileHandler;
	}

	static public FASTAFileHandler createFastaFileHandler(File filePath, Charset fileCharset, LineSeparatorFormat fileLineSeparatorFormat, boolean ignoreBlankLines ) 
			throws FileNotFoundException, OperationCanceledException, IOException {
		return createFastaFileHandler(filePath,fileCharset, fileLineSeparatorFormat, ignoreBlankLines , null);

	}

	static public FASTAFileHandler createFastaFileHandler(File filePath , Charset fileCharset) 
			throws FileNotFoundException, OperationCanceledException, IOException {

		return createFastaFileHandler(filePath,fileCharset, LineSeparatorFormat.SYSTEM_DEFAULT, true , null);
	}

	static public FASTAFileHandler createFastaFileHandler(File filePath , Charset fileCharset , IProgressMonitor progressMonitor) 
			throws FileNotFoundException, OperationCanceledException, IOException {
		return createFastaFileHandler(filePath,fileCharset, LineSeparatorFormat.SYSTEM_DEFAULT, true , progressMonitor);
	}

	static public FASTAFileHandler createFastaFileHandler(File filePath) 
			throws FileNotFoundException, OperationCanceledException, IOException {
		return createFastaFileHandler(filePath,Charset.defaultCharset(), LineSeparatorFormat.SYSTEM_DEFAULT, true , null);
	}



	// ##################################
	/**
	 * Returns all the sequence blocks starting and ending byte offsets within the file associated to this 
	 * <code>FastaReader</code>.
	 *
	 * @return	starting and ending offset of all FASTA records stored in this file. The returned <code>List</code> 
	 * 			has an even number of elements where each even index is the starting byte (inclusive) and each odd 
	 * 			index is the ending byte (exclusive). Therefore, the byte offsets of the <code>i</code>th record are 
	 * 			in the <code>List</code>&rsquo;s <code>i/2</code>th element.
	 * 
	 * @since	1.0rc3
	 */
	private List<Long> getSequenceByteRanges()
	{
		List<Long> sequenceRanges =
				new ArrayList<Long>(fastaRecords.size() * 2);

		for (int i = 0 ; i < fastaRecords.size() ; i++)
		{
			sequenceRanges.add(fastaRecords.get(i).sequenceByteOffset);
			sequenceRanges.add(fastaRecords.get(i).sequenceByteOffset + fastaRecords.get(i).sequenceBytes);
		}
		//Collections.sort(sequenceRanges);
		return sequenceRanges;
	}





	/**
	 * Returns the total sum of bytes which store sequence blocks in the file of this <code>FastaReader</code>. 
	 *  
	 * @return	Number of bytes between the comment sections of this <code>FastaReader</code>&rsquo;s file. Possibly 
	 * <code>0</code>.
	 * 
	 * @since	1.0rc3
	 */
	long getTotalSequenceBytes()
	{
		List<Long> byteRanges = getSequenceByteRanges();
		return getTotalSequenceBytes(byteRanges);
	}
	long getTotalSequenceBytes(List<Long> byteRanges)  {
		long sequenceByteCount = 0;
		for (int i = 0 ; i < byteRanges.size() - 1 ; i = i + 2)
		{
			sequenceByteCount += byteRanges.get(i + 1) - byteRanges.get(i);
		}
		return sequenceByteCount;
	}

	/**
	 * Returns a series of starting and ending sequence byte offsets that total  a specified number of bytes, 
	 * beginning at a byte offset. 
	 *
	 * @param byteOffsetStart	byte offset to start reading. If this byte is part of a comment, it will be adjusted 
	 * 							to the next nearest sequence block starting byte.
	 * @param nBytes	number of bytes to read.
	 *
	 * @return	Starting and ending offset of all FASTA records stored in this file. The returned <code>List</code> 
	 * 			has an even number of elements where each even index is the starting byte (inclusive) and each odd 
	 * 			index is the ending byte (exclusive). Therefore, the byte offsets of the <code>i</code>th record are 
	 * 			in the <code>List</code>'s <code>i/2</code>th element.
	 * 
	 * @throws	IllegalArgumentException	if <code>byteOffsetStart</code> is negative or larger than the file size; 
	 * 										if <code>nBytes</code> is larger than the total number of sequence block 
	 * 										bytes.
	 *  
	 * @since	1.0rc3
	 */
	List<Long> getSequenceByteRanges(long byteOffsetStart, int nBytes) throws IllegalArgumentException
	{	
		List<Long> toReadByteRanges = new ArrayList<Long>();

		if (fastaRecords.size() == 0)
		{
			toReadByteRanges.add(0L);
			toReadByteRanges.add(0L);
			return toReadByteRanges;
		}

		if (byteOffsetStart < 0 || byteOffsetStart > filePath.length() - 1)
		{
			throw new IllegalArgumentException(byteOffsetStart + ": the " +
					"starting offset must be a value between 0 and (file size - 1)");
		}
		if (nBytes > getTotalSequenceBytes())
		{
			throw new IllegalArgumentException(nBytes + ": the number of " +
					"bytes to read is larger than the file size");
		}

		// Get the ranges that store the sequence blocks and...
		List<Long> sequenceByteRanges = getSequenceByteRanges();

		// ... check there are actual bytes that can be read...
		boolean somethingToRead = false;
		for (int i = 0 ; i < sequenceByteRanges.size() - 1; i = i + 2)
		{
			if (sequenceByteRanges.get(i + 1) - sequenceByteRanges.get(i) > 0)
			{
				somethingToRead = true;
				break;
			}
		}
		if (!somethingToRead)
		{
			return toReadByteRanges;
		}

		// ... find the offset position in the list
		int byteOffsetStartPosition = Collections.binarySearch(sequenceByteRanges, byteOffsetStart);

		if (byteOffsetStartPosition < 0)
		{	// If it's not there (very likely), where is it?
			byteOffsetStartPosition = -(byteOffsetStartPosition + 1);

			if (byteOffsetStartPosition % 2 != 0)
			{   // If it's in a sequence block, adjust position to start
				byteOffsetStartPosition--;
			}
			else
			{	// If it's a comment block, set offset to the start offset
				byteOffsetStart = sequenceByteRanges.get(byteOffsetStartPosition);
			}
		}
		else
		{	// Adjust to a starting offset if it's a comment block
			if (byteOffsetStartPosition % 2 != 0)
			{
				byteOffsetStartPosition++;
				byteOffsetStart = sequenceByteRanges.get(byteOffsetStartPosition);
			}
		}
		// byteOffsetStart is now in the middle or start of a sequence block

		// Read nBytes starting at the given offset, only sequence block ranges
		long currentBlockStartOffset = byteOffsetStart;
		long currentBlockEndOffset = sequenceByteRanges.get(byteOffsetStartPosition + 1);
		while (nBytes > 0)
		{	// While there are bytes to be read: add the current start position
			toReadByteRanges.add(currentBlockStartOffset);
			// If nBytes is greater than the end, add the end...
			if (currentBlockStartOffset + nBytes >= currentBlockEndOffset)
			{
				toReadByteRanges.add(currentBlockEndOffset);
				// ... decrement the amount of bytes read...
				nBytes -= currentBlockEndOffset - currentBlockStartOffset;
			}
			else
			{
				toReadByteRanges.add(currentBlockStartOffset + nBytes);
				break;
			}

			// ... if there's at least another sequence left ...
			if (byteOffsetStartPosition + 2 < sequenceByteRanges.size())
			{
				// skip (the comment) to the next sequence range
				byteOffsetStartPosition += 2;
				currentBlockStartOffset = sequenceByteRanges.get(byteOffsetStartPosition);
				currentBlockEndOffset = sequenceByteRanges.get(byteOffsetStartPosition + 1);
			}
		}
		return toReadByteRanges;
	}


	/**
	 * Returns the highest byte offset at which reading a determined amount of sequence bytes in this file will not 
	 * fail due to reaching EOF. 
	 * 
	 * @param	byteCount	number of bytes to read.
	 * 
	 * @return	byte offset at which the call
	 * 			<br /><code>readByteRanges(..., sequenceByteRanges(..., offset))</code><br />
	 * 			will not fail because of reaching EOF.
	 * 
	 * @throws	IllegalArgumentException	if <code>byteCount</code> is negative or greater than the value returned 
	 * 			by <code>totalSequenceBytes()</code>.
	 * 
	 * @since	1.0rc3
	 */
	long countSequenceBytesBackwards(long byteCount) throws IllegalArgumentException
	{
		if (byteCount < 0 || byteCount > getTotalSequenceBytes())
		{
			throw new IllegalArgumentException(
					byteCount + ": the amount of bytes to read must be between 0 and " + getTotalSequenceBytes());
		}

		List<Long> byteRanges = getSequenceByteRanges();
		int i = byteRanges.size() - 1;

		if (byteCount == 0)
		{
			return filePath.length() - 1;
		}

		while (byteCount > 0 && i >= 0)
		{
			byteCount -= byteRanges.get(i) - byteRanges.get(i - 1);
			i = i - 2;
		}
		return byteRanges.get(i + 1) - byteCount - 1;
	}

	// ################


	int position;

	/**
	 * Returns the index of the <code>FastaRecord</code> in this file which will be returned by a call to 
	 * <code><netxSequence()</code>.
	 * 
	 * @return	A value between 0 and <code>getFastaRecords().size()</code>
	 * 
	 * @since	1.0rc2
	 */
	public int getPosition()
	{
		return position;
	}

	/**
	 * Set the reading index to the specified value.
	 * 
	 * @param	position	New position of the reading index.
	 * 
	 * @throws	IllegalArgumentException	if <code>position</code> is negative or greater than the number of records 
	 * 			in the file associated to this <code>FastaReader</code>.
	 * @throws FastaReaderNotParsedException 
	 * 
	 * @since	1.0rc2	
	 */
	public void setPosition(int position) throws IllegalArgumentException
	{
		if (position < 0 || position > fastaRecords.size())
		{
			throw new IllegalArgumentException(position + ": position must " + 
					"be a value between 0 and the number of FASTA records");
		}
		this.position = position;
	}

	// ^^^^^^^^^^^^^^^^^^^^^^^^


	// ################################

	public DNASequence getDnaSequence(int index) throws FileNotFoundException, IOException, InvalidSequenceCharacterException, SequenceTooLongException
	{
		if (index < 0 || index > fastaRecords.size() - 1)
		{
			throw new IndexOutOfBoundsException(
					index + ": index must be a positive value smaller than fastaRecords.size()");
		}
		return fastaRecords.get(index).retrieveDnaSequence(true);
	}

	public RNASequence getRnaSequence(int index)
			throws FileNotFoundException, IOException, InvalidSequenceCharacterException, SequenceTooLongException
	{
		if (index < 0 || index > fastaRecords.size() - 1)
		{
			throw new IndexOutOfBoundsException(
					index + ": index must be a positive value smaller than fastaRecords.size()");
		}

		return fastaRecords.get(index).retrieveRnaSequence(true);
	}

	public ProteinSequence getProteinSequence(int index)
			throws FileNotFoundException, IOException, InvalidSequenceCharacterException
	{

		if (index < 0 || index > fastaRecords.size() - 1)
		{
			throw new IndexOutOfBoundsException(
					index + ": index must be a positive value smaller than fastaRecords.size()");
		}

		return fastaRecords.get(index).retrieveProteinSequence(true);
	}


	/**
	 * Returns the next DNA sequence or <code>null</code> if there are no more 
	 * sequences of this type between the current position and the end.
	 * 
	 * @return
	 * @throws SequenceTooLongException 
	 * @throws InvalidSequenceCharacterException 
	 * @throws FileNotFoundException 
	 * 
	 * @throws IOException
	 * @throws FastaReaderNotParsedException 
	 * @throws RecordTooLongException 
	 * 
	 * @since	1.0rc2
	 */
	public DNASequence nextDnaSequence(boolean removeGaps)
			throws FileNotFoundException, IOException, InvalidSequenceCharacterException, SequenceTooLongException
	{

		while (position < fastaRecords.size())
		{
			position++;
			return fastaRecords.get(position - 1).retrieveDnaSequence(true);
		}
		return null;
	}


	/**
	 * Returns the next RNA sequence or <code>null</code> if there are no more 
	 * sequences of this type between the current position and the end.
	 * 
	 * @return
	 * 
	 * @throws IOException
	 * @throws FastaReaderNotParsedException 
	 * @throws RecordTooLongException 
	 * 
	 * @since	1.0rc2
	 */
	public RNASequence nextRnaSequence(boolean removeGaps)
			throws FileNotFoundException, IOException, InvalidSequenceCharacterException, SequenceTooLongException
	{
		while (position < fastaRecords.size())
		{
			position++;
			return fastaRecords.get(position - 1).retrieveRnaSequence(true);
		}
		return null;
	}

	/**
	 * Returns the next protein sequence or <code>null</code> if there are no 
	 * more sequences of this type between the current position and the end.
	 * 
	 * @return
	 * 
	 * @throws IOException
	 * @throws FastaReaderNotParsedException 
	 * @throws RecordTooLongException 
	 * 
	 * @since	1.0rc2
	 */
	public ProteinSequence nextProteinSequence()
			throws FileNotFoundException, IOException, InvalidSequenceCharacterException, SequenceTooLongException
	{

		while (position < fastaRecords.size())
		{
			position++;
			return fastaRecords.get(position - 1).retrieveProteinSequence(true);
		}
		return null;
	}


	// ^^^^^^^^^^^^^^^^^^^^^^^^^^

	public BioSequence getSequence(int index) throws FileNotFoundException, IOException, InvalidSequenceCharacterException, SequenceTooLongException
	{


		if (index < 0 || index > fastaRecords.size() - 1)
		{
			throw new IndexOutOfBoundsException(
					index + ": index must be a positive value smaller than fastaRecords.size()");
		}

		BioSequence bs = fastaRecords.get(index).retrieveBioSequence();
		return bs;
	}


//	public void write(List<BioSequence> seqsToSave  , IProgressMonitor progressMonitor) 
//			throws IOException, IllegalArgumentException, SequenceTooLongException, InvalidSequenceCharacterException {
//		write(seqsToSave,null,progressMonitor);
//	}

	/**
	 * write selected sequence to the ref file, and update record upon finishing
	 * @param seqsDescLine
	 * @param seqsToSave
	 * @param records
	 * @param progressMonitor
	 * @throws IOException 
	 * TODO :: remove record from here if possible !!
	 */
	public void write(List<String> seqsDescLine, List<BioSequence> seqsToSave, List<FASTAFileRecord> records,
			IProgressMonitor progressMonitor) throws  FileNotFoundException, IOException, SequenceTooLongException, InvalidSequenceCharacterException  {
		if(records == null)
		{
			records = new ArrayList<FASTAFileRecord>();
			for(BioSequence seq : seqsToSave) {
				records.add(null);
			}
		}
		FastaRecordCollection newFastaRecords = new FastaRecordCollection();

		progressMonitor.beginTask("Saving file ... ", seqsToSave.size());
		// do not create an new one
		File tmpFile = File.createTempFile("tmp.", "", filePath.getParentFile());
		FASTAWriter2 faw = new FASTAWriter2(tmpFile,this);

		try {
			for(int i = 0 ; i< seqsToSave.size();i++) {
				BioSequence seq = seqsToSave.get(i);
				FASTAFileRecord newRec = null;
				// if seq is null this mean no change on this seq avoid reading and coping back if not needed
				if(seq != null)
					newRec = faw.append(records.get(i),seq, progressMonitor);
				else
					newRec = faw.copy(seqsDescLine.get(i),records.get(i), progressMonitor);
				newFastaRecords.add(newRec);
				//seq.updateRecord(far);
			}
		}
		catch (Exception e) {
			tmpFile.delete();
			throw e;
		}
		tmpFile.renameTo(filePath);	
		this.fastaRecords  = newFastaRecords;	
		progressMonitor.done();

	}

	public void write(List<BioSequence> seqsToSave, List<FASTAFileRecord> records, IProgressMonitor progressMonitor) 
			throws IOException, IllegalArgumentException, SequenceTooLongException, InvalidSequenceCharacterException {
		if(records == null)
		{
			records = new ArrayList<FASTAFileRecord>();
			for(BioSequence seq : seqsToSave) {
				records.add(null);
			}
		}

		// do not create an new one
		File tmpFile = File.createTempFile("tmp.", "", filePath.getParentFile());


		FastaRecordCollection newFastaRecords = new FastaRecordCollection();
		//		final FASTAReader far =   new FASTAReader(tmpFile);

		progressMonitor.beginTask("Saving file ... ", seqsToSave.size());
		//		far.parse(false, progressMonitor);
		FASTAWriter2 faw = new FASTAWriter2(tmpFile,this);

		try {
			for(int i = 0 ; i< seqsToSave.size();i++) {
				BioSequence seq = seqsToSave.get(i);
				FASTAFileRecord rec = records.get(i);
				if(seq != null)
					faw.append(rec,seq, progressMonitor);
//				else
//					faw.copy(rec, progressMonitor);
				newFastaRecords.add(rec);
				//seq.updateRecord(far);
			}
		}
		catch (Exception e) {
			tmpFile.delete();
			throw e;
		}
		tmpFile.renameTo(filePath);	
		this.fastaRecords  = newFastaRecords;		
	}


	public void replace(BioSequence bioSequence, FASTAFileRecord fastaRecord, IProgressMonitor progressMonitor) throws IOException, IllegalArgumentException, FastaWriterMultipleSequenceException, SequenceTooLongException, InvalidSequenceCharacterException {


		//		File tmpFile = File.createTempFile("tmp.", "", filePath.getParentFile());
		FASTAWriter2 faw = new FASTAWriter2(this);

		faw.replace(fastaRecord,bioSequence, progressMonitor);

	}





}
