/*
 * @author		Alfonso Muñoz-Pomer Fuentes, 
 * 				<a href="mailto:alfonso.munozpomer@biotechvana.com">
 * 				alfonso.munozpomer@biotechvana.com</a>,  
 * 				<a href="http://www.biotechvana.com">Biotechvana</a>
 *
 * @date		2010-11-01
 * 
 * @copyright	Copyright Biotech Vana, S.L. 2006-2010
 */

package com.biotechvana.javabiotoolkit;

import java.io.Serializable;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Set;

import org.eclipse.swt.graphics.Point;

/**
 * A sequence annotation is a region in a <code>BioSequence</code> that highlights a specific feature.
 * 
 * @version	1.5, 2011-09-22
 * 
 * @author	<a href="mailto:alfonso.munozpomer@biotechvana.com">Alfonso Muñoz-Pomer Fuentes</a>,
 * 			<a href="http://www.biotechvana.com">Biotechvana</a>.
 * 
 * @see		DNASequence
 * @see		RNASequence
 * @see		ProteinSequence
 *
 * <style type="text/css">
 * 		table.t0 {
 * 			border:0px solid black;
 * 			border-collapse: collapse;
 * 		}
 * 		table.t0 td {
 * 			text-align: center;
 * 			padding: 4px;
 * 		}
 * 		tr.d0 td {
 * 			background-color: #FFFFFF; color: black;
 * 		}
 * 		tr.d1 td {
 * 			background-color: #DDDDDD; color: black;
 * 		}
 * </style>
 */
public class Annotation implements Comparable<Annotation>, Cloneable, Serializable
{
	
	
	
	private static final long serialVersionUID = 2799834224725445827L;	// Autogenerated by Eclipse
	
	protected StringBuilder descriptionSB;
	// An inclusive-exclusive range.
	private Point range;
	private boolean complement;
	private boolean proteinProduct;
	private boolean checked = true;

	
	protected Annotation()
	{
		
	}
	
	/**
	 * Create a new annotation. <code>Annotation</code>s can be left-to-right and right-to-left, (i.e. {2, 5} is the 
	 * same range as {4, 1} but direction differs).
	 *
	 * @param	description	description of the feature highlighted by this annotaion.
	 * @param	fromIndex	index of the first position of the region to annotate (inclusive).
	 * @param	toIndex		index of the last position of the region to annotate (exclusive).
	 * 
	 * @since	1.5
	 */
	public Annotation(String description, int fromIndex, int toIndex)
	{
		descriptionSB = new StringBuilder(description);
		range = new Point(fromIndex, toIndex);
		complement = false;
		proteinProduct = false;
	}
	
	public Annotation(String description, int fromIndex, int toIndex, boolean complement, boolean proteinProduct)
	{
		descriptionSB = new StringBuilder(description);
		range = new Point(fromIndex, toIndex);
		this.complement = complement;
		this.proteinProduct = proteinProduct;
	}

	/**
	 * Returns the description of this annotation.
	 * 
	 * @return	the description of this annotation, usually a function or feature in the annotation&rsquo;s region.
	 * 
	 * @since	1.1
	 */
	public String getDescription()
	{
		return descriptionSB.toString();
	}
	
	/**
	 * 
	 * @return
	 */
	public StringBuilder getDescriptionSB()
	{
		return descriptionSB;
	}

	/**
	 * Returns the range of this annotation.
	 * 
	 * @return	the start index (inclusive) and end index (exclusive) of the positions delimited by this annotation.
	 * 
	 * @since	1.1
	 */
	public Point getRange()
	{
		return range;
	}
	
	/**
	 * 
	 * @param range
	 */
	public void setRange(Point range)
	{
		this.range = range;
	}
	
	public void setRange(int x, int y)
	{
		range.x = x;
		range.y = y;
	}
	
	/**
	 * 
	 * @return
	 * 
	 * @since	1.5
	 */
	public boolean getComplement()
	{
		return complement;
	}
	
	/**
	 * 
	 * @return
	 * 
	 * @since	1.5
	 */
	public boolean getProteinProduct()
	{
		return proteinProduct;
	}
	
	/**
	 * Sets a new description for this annotation.
	 * 
	 * @param	description	new text to describe this annotation.
	 * 
	 * @since	1.1
	 */
	public void setDescription(String description)
	{
		descriptionSB.delete(0, descriptionSB.length());
		descriptionSB.append(description);
	}

	public void setComplement(boolean complement)
	{
		this.complement = complement;
	}
	
	public void setProteinProduct(boolean proteinProduct)
	{
		this.proteinProduct = proteinProduct;
	}
	
	/**
	 * 
	 * @return
	 *
	 * @since	1.3
	 */
	/*
	public String location()
	{
		if (range.x < range.y)
		{
			switch (frameReference)
			{
				case 0: return "Forward";
				case 1: return "Frame +1";
				case 2: return "Frame +2";
				case 3: return "Frame +3";
				default: return "";
			}
		}
		else if (range.y < range.x)
		{
			switch (frameReference)
			{
				case 0: return "Reverse";
				case 1: return "Frame -1";
				case 2: return "Frame -2";
				case 3: return "Frame -3";
				default: return "";
			}
		}
		else
		{
			return "";
		}
	}
	*/
	
	/**
	 * Ranges are changed to start on 1 and end at sequenceLengt + 1
	 * 
	 * @return
	 *
	 * @since	1.4***
	 */
	public int humanReadableStart()
	{
		return range.x + 1;
	}
	
	/**
	 * 
	 * @return
	 *
	 * @since	1.4***
	 */
	public int humanReadableEnd()
	{
		if (range.x < range.y)
		{
			return range.y;
		}
		else // if (range.x > range.y)
		{
			return range.y + 2;
		}
	}
	
	
	
	
	
	
	/**
	 * FIXME :: please revise if this is a 0-based or 1 based
	 * @return
	 */
	public int getStart()
	{
		return range.x;
	}
	
	/**
	 * 
	 * @return
	 */
	public int getEnd()
	{
		return range.y;
	}
	
	
	public String getLocation() {
		
		if(complement)
			return "-";
		return "+";
	}
	
	
	/**
	 * 
	 * @return
	 *
	 * @since	x.y.z***
	 */
	public int normalizedStart()
	{
		if (range.x < range.y)
		{
			return range.x;
		}
		else // if (range.x > range.y)
		{
			return range.y + 1;
		}
	}
	
	/**
	 * 
	 * @return
	 *
	 * @since	x.y.z***
	 */
	public int normalizedEnd()
	{
		if (range.x < range.y)
		{
			return range.y;
		}
		else // if (range.x > range.y)
		{
			return range.x + 1;
		}
	}
	
	/**
	 * 
	 * @return
	 *
	 * @since	1.4
	 */
	public int getLength()
	{
		return Math.abs(getRange().y - getRange().x);
	}
	

	
	/*
	 * (non-Javadoc) @see java.lang.Object#clone()
	 */
	public Annotation clone()
	{
		return new Annotation(descriptionSB.toString(), range.x, range.y);
	}

	/*
	 * (non-Javadoc) @see java.lang.Comparable#compareTo(java.lang.Object)
	 */
	public int compareTo(Annotation a)
	{
		// This annotation starts before
		if (this.normalizedStart() < a.normalizedStart())
		{
			return -1;
		}
		// This annotation starts in the same position...
		else if (this.normalizedStart() == a.normalizedStart())
		{
			if (this.normalizedEnd() < a.normalizedEnd())
			{
				return -1;
			}
			else if (this.normalizedEnd() == a.normalizedEnd())
			{
				return 0;
			}
			else
			{
				return 1;
			}
		}
		// This annotation starts later
		else
		{
			return 1;
		}
	}
	
	/**
	 * 
	 * @return
	 */
	public boolean isChecked()
	{
		return checked;
	}
	/**
	 * 
	 * @param checked
	 */
	public void setChecked(boolean checked)
	{
		this.checked = checked;
	}
	
	// Features GTF extend

	/**
	 * Extensions to annotation add more attributes
	 */
	protected HashMap<String, String> attributes = new HashMap<String, String>();
	
	/**
	 * Extensions to annotation add more attributes
	 * @return value of the attribute or empty str in it has not this attribute
	 */
	public String getAttribute(String key)
	{
		if(attributes.containsKey(key))
			return attributes.get(key);
		return "";
	}

	/**
	 * add attributes with key and value
	 * @param key
	 * @param value
	 */
	public void addAttribute(String key, String value)
	{
		attributes.put(key, value);
	}
	
	public void addAttribute(String key, double value) {
		this.addAttribute(key,Double .toString(value ));
	}
	
	public Set<String> getAttributesKeys()
	{
		return  new HashSet<String>( attributes.keySet());
	}

	

	
	
}